/*
 * This program is released under the Common Public License V1.0
 *
 * You should have received a copy of Common Public License V1.0 along with
 * with this program.
 *
 * Copyright IBM Corp. 2015
 */

#include <errno.h>

#include "ica_api.h"
#include "test_vec.h"

#ifdef ICA_FIPS
const struct rsa_tv RSA_TV[] = {
{
.n = (unsigned char []){
0xba, 0xd4, 0x7a, 0x84, 0xc1, 0x78, 0x2e, 0x4d, 0xbd, 0xd9, 0x13, 0xf2, 0xa2,
0x61, 0xfc, 0x8b, 0x65, 0x83, 0x84, 0x12, 0xc6, 0xe4, 0x5a, 0x20, 0x68, 0xed,
0x6d, 0x7f, 0x16, 0xe9, 0xcd, 0xf4, 0x46, 0x2b, 0x39, 0x11, 0x95, 0x63, 0xca,
0xfb, 0x74, 0xb9, 0xcb, 0xf2, 0x5c, 0xfd, 0x54, 0x4b, 0xda, 0xe2, 0x3b, 0xff,
0x0e, 0xbe, 0x7f, 0x64, 0x41, 0x04, 0x2b, 0x7e, 0x10, 0x9b, 0x9a, 0x8a, 0xfa,
0xa0, 0x56, 0x82, 0x1e, 0xf8, 0xef, 0xaa, 0xb2, 0x19, 0xd2, 0x1d, 0x67, 0x63,
0x48, 0x47, 0x85, 0x62, 0x2d, 0x91, 0x8d, 0x39, 0x5a, 0x2a, 0x31, 0xf2, 0xec,
0xe8, 0x38, 0x5a, 0x81, 0x31, 0xe5, 0xff, 0x14, 0x33, 0x14, 0xa8, 0x2e, 0x21,
0xaf, 0xd7, 0x13, 0xba, 0xe8, 0x17, 0xcc, 0x0e, 0xe3, 0x51, 0x4d, 0x48, 0x39,
0x00, 0x7c, 0xcb, 0x55, 0xd6, 0x84, 0x09, 0xc9, 0x7a, 0x18, 0xab, 0x62, 0xfa,
0x6f, 0x9f, 0x89, 0xb3, 0xf9, 0x4a, 0x27, 0x77, 0xc4, 0x7d, 0x61, 0x36, 0x77,
0x5a, 0x56, 0xa9, 0xa0, 0x12, 0x7f, 0x68, 0x24, 0x70, 0xbe, 0xf8, 0x31, 0xfb,
0xec, 0x4b, 0xcd, 0x7b, 0x50, 0x95, 0xa7, 0x82, 0x3f, 0xd7, 0x07, 0x45, 0xd3,
0x7d, 0x1b, 0xf7, 0x2b, 0x63, 0xc4, 0xb1, 0xb4, 0xa3, 0xd0, 0x58, 0x1e, 0x74,
0xbf, 0x9a, 0xde, 0x93, 0xcc, 0x46, 0x14, 0x86, 0x17, 0x55, 0x39, 0x31, 0xa7,
0x9d, 0x92, 0xe9, 0xe4, 0x88, 0xef, 0x47, 0x22, 0x3e, 0xe6, 0xf6, 0xc0, 0x61,
0x88, 0x4b, 0x13, 0xc9, 0x06, 0x5b, 0x59, 0x11, 0x39, 0xde, 0x13, 0xc1, 0xea,
0x29, 0x27, 0x49, 0x1e, 0xd0, 0x0f, 0xb7, 0x93, 0xcd, 0x68, 0xf4, 0x63, 0xf5,
0xf6, 0x4b, 0xaa, 0x53, 0x91, 0x6b, 0x46, 0xc8, 0x18, 0xab, 0x99, 0x70, 0x65,
0x57, 0xa1, 0xc2, 0xd5, 0x0d, 0x23, 0x25, 0x77, 0xd1
},
.p = (unsigned char []){
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,	/* 8 bytes zero padding */
0xe7, 0xc9, 0xe4, 0xb3, 0xe5, 0xd7, 0xac, 0x9e, 0x83, 0xbe, 0x08, 0x32, 0x81,
0x05, 0x35, 0x6d, 0xfe, 0xef, 0xe2, 0x22, 0xf2, 0x6c, 0x95, 0x37, 0x8e, 0xff,
0xd2, 0x15, 0x0f, 0xad, 0xf7, 0xba, 0x23, 0xf5, 0xb4, 0x70, 0x5d, 0x82, 0xe4,
0xf1, 0xbc, 0x45, 0x05, 0x70, 0x67, 0xc7, 0xde, 0xf7, 0x3e, 0x21, 0x00, 0xf7,
0x56, 0xee, 0x6d, 0x54, 0x79, 0x65, 0xfa, 0x4f, 0x24, 0xb8, 0x5d, 0x68, 0x86,
0x7f, 0x03, 0xd7, 0xc8, 0x86, 0xd1, 0xdb, 0xcc, 0xa4, 0xc5, 0x89, 0x74, 0x57,
0x01, 0xb3, 0x62, 0xa1, 0xf1, 0x41, 0x7f, 0x47, 0x1d, 0x84, 0x75, 0xb6, 0xb7,
0xa1, 0x6a, 0x4c, 0x48, 0xef, 0x1f, 0x55, 0x6e, 0xdc, 0x3f, 0x0f, 0xf6, 0xba,
0x13, 0xd3, 0x65, 0xd6, 0xe8, 0x27, 0x51, 0xf2, 0x07, 0xd9, 0x11, 0x01, 0xc8,
0xee, 0xa1, 0x01, 0x3c, 0xcd, 0xd9, 0xe1, 0xde, 0x4c, 0x38, 0x7f
},
.q = (unsigned char []){
0xce, 0x58, 0x60, 0x2e, 0x05, 0x1f, 0x0f, 0x46, 0x47, 0xc4, 0xec, 0x57, 0xf6,
0x82, 0xe5, 0x73, 0x7f, 0xc4, 0x82, 0xa8, 0xa1, 0xff, 0xac, 0x90, 0x43, 0xbb,
0xa4, 0xfb, 0xa3, 0x38, 0x7d, 0x7d, 0xd2, 0x15, 0x45, 0x07, 0xaf, 0x1e, 0x28,
0xbd, 0x81, 0xb6, 0x1f, 0xcd, 0xfe, 0x35, 0xf9, 0x73, 0x4e, 0x0d, 0x9b, 0x53,
0x68, 0x2e, 0xc7, 0x85, 0xf1, 0xf6, 0xe6, 0x22, 0x4f, 0x63, 0xd1, 0x0b, 0xf7,
0x84, 0x84, 0xb8, 0x3a, 0x42, 0x54, 0xf3, 0x33, 0xd0, 0xfb, 0x3f, 0x3e, 0x9e,
0x18, 0x34, 0xbe, 0xde, 0x52, 0xe3, 0x07, 0x8a, 0xc2, 0x79, 0xa8, 0x62, 0xfb,
0x90, 0xaf, 0x26, 0x6d, 0x75, 0x91, 0xc8, 0x1f, 0x20, 0xb7, 0x18, 0xd0, 0x7d,
0x51, 0xbf, 0xc2, 0x21, 0xb6, 0x6a, 0x25, 0x40, 0x3b, 0x4a, 0xc1, 0xa6, 0x8d,
0x67, 0x3f, 0xdd, 0x95, 0x9b, 0x01, 0xec, 0xf3, 0xd0, 0xa7, 0xaf
},
.e = (unsigned char []){
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x01
},
.d = (unsigned char []){
0x40, 0xd6, 0x0f, 0x24, 0xb6, 0x1d, 0x76, 0x78, 0x3d, 0x3b, 0xb1, 0xdc, 0x00,
0xb5, 0x5f, 0x96, 0xa2, 0xa6, 0x86, 0xf5, 0x9b, 0x37, 0x50, 0xfd, 0xb1, 0x5c,
0x40, 0x25, 0x1c, 0x37, 0x0c, 0x65, 0xca, 0xda, 0x22, 0x26, 0x73, 0x81, 0x1b,
0xc6, 0xb3, 0x05, 0xed, 0x7c, 0x90, 0xff, 0xcb, 0x3a, 0xbd, 0xdd, 0xc8, 0x33,
0x66, 0x12, 0xff, 0x13, 0xb4, 0x2a, 0x75, 0xcb, 0x7c, 0x88, 0xfb, 0x93, 0x62,
0x91, 0xb5, 0x23, 0xd8, 0x0a, 0xcc, 0xe5, 0xa0, 0x84, 0x2c, 0x72, 0x4e, 0xd8,
0x5a, 0x13, 0x93, 0xfa, 0xf3, 0xd4, 0x70, 0xbd, 0xa8, 0x08, 0x3f, 0xa8, 0x4d,
0xc5, 0xf3, 0x14, 0x99, 0x84, 0x4f, 0x0c, 0x7c, 0x1e, 0x93, 0xfb, 0x1f, 0x73,
0x4a, 0x5a, 0x29, 0xfb, 0x31, 0xa3, 0x5c, 0x8a, 0x08, 0x22, 0x45, 0x5f, 0x1c,
0x85, 0x0a, 0x49, 0xe8, 0x62, 0x97, 0x14, 0xec, 0x6a, 0x26, 0x57, 0xef, 0xe7,
0x5e, 0xc1, 0xca, 0x6e, 0x62, 0xf9, 0xa3, 0x75, 0x6c, 0x9b, 0x20, 0xb4, 0x85,
0x5b, 0xdc, 0x9a, 0x3a, 0xb5, 0x8c, 0x43, 0xd8, 0xaf, 0x85, 0xb8, 0x37, 0xa7,
0xfd, 0x15, 0xaa, 0x11, 0x49, 0xc1, 0x19, 0xcf, 0xe9, 0x60, 0xc0, 0x5a, 0x9d,
0x4c, 0xea, 0x69, 0xc9, 0xfb, 0x6a, 0x89, 0x71, 0x45, 0x67, 0x48, 0x82, 0xbf,
0x57, 0x24, 0x1d, 0x77, 0xc0, 0x54, 0xdc, 0x4c, 0x94, 0xe8, 0x34, 0x9d, 0x37,
0x62, 0x96, 0x13, 0x7e, 0xb4, 0x21, 0x68, 0x61, 0x59, 0xcb, 0x87, 0x8d, 0x15,
0xd1, 0x71, 0xed, 0xa8, 0x69, 0x28, 0x34, 0xaf, 0xc8, 0x71, 0x98, 0x8f, 0x20,
0x3f, 0xc8, 0x22, 0xc5, 0xdc, 0xee, 0x7f, 0x6c, 0x48, 0xdf, 0x66, 0x3e, 0xa3,
0xdc, 0x75, 0x5e, 0x7d, 0xc0, 0x6a, 0xeb, 0xd4, 0x1d, 0x05, 0xf1, 0xca, 0x28,
0x91, 0xe2, 0x67, 0x97, 0x83, 0x24, 0x4d, 0x06, 0x8f
},
.ciphertext = (unsigned char []){
0x70, 0x99, 0x2c, 0x9d, 0x95, 0xa4, 0x90, 0x8d, 0x2a, 0x94, 0xb3, 0xab, 0x9f,
0xa1, 0xcd, 0x64, 0x3f, 0x12, 0x0e, 0x32, 0x6f, 0x9d, 0x78, 0x08, 0xaf, 0x50,
0xca, 0xc4, 0x2c, 0x4b, 0x0b, 0x4e, 0xeb, 0x7f, 0x0d, 0x4d, 0xf3, 0x03, 0xa5,
0x68, 0xfb, 0xfb, 0x82, 0xb0, 0xf5, 0x83, 0x00, 0xd2, 0x53, 0x57, 0x64, 0x57,
0x21, 0xbb, 0x71, 0x86, 0x1c, 0xaf, 0x81, 0xb2, 0x7a, 0x56, 0x08, 0x2c, 0x80,
0xa1, 0x46, 0x49, 0x9f, 0xb4, 0xea, 0xb5, 0xbd, 0xe4, 0x49, 0x3f, 0x5d, 0x00,
0xf1, 0xa4, 0x37, 0xbb, 0xc3, 0x60, 0xdf, 0xcd, 0x80, 0x56, 0xfe, 0x6b, 0xe1,
0x0e, 0x60, 0x8a, 0xdb, 0x30, 0xb6, 0xc2, 0xf7, 0x65, 0x24, 0x28, 0xb8, 0xd3,
0x2d, 0x36, 0x29, 0x45, 0x98, 0x2a, 0x46, 0x58, 0x5d, 0x21, 0x02, 0xef, 0x79,
0x95, 0xa8, 0xba, 0x6e, 0x8a, 0xd8, 0xfd, 0x16, 0xbd, 0x7a, 0xe8, 0xf5, 0x3c,
0x3d, 0x7f, 0xcf, 0xba, 0x29, 0x0b, 0x57, 0xce, 0x7f, 0x8f, 0x09, 0xc8, 0x28,
0xd6, 0xf2, 0xd3, 0xce, 0x56, 0xf1, 0x31, 0xbd, 0x94, 0x61, 0xe5, 0x66, 0x7e,
0x5b, 0x73, 0xed, 0xac, 0x77, 0xf5, 0x04, 0xda, 0xc4, 0xf2, 0x02, 0xa9, 0x57,
0x0e, 0xb4, 0x51, 0x5b, 0x2b, 0xf5, 0x16, 0x40, 0x7d, 0xb8, 0x31, 0x51, 0x8d,
0xb8, 0xa2, 0x08, 0x3e, 0xc7, 0x01, 0xe8, 0xfd, 0x38, 0x7c, 0x43, 0x0b, 0xb1,
0xa7, 0x2d, 0xec, 0xa5, 0xb4, 0x9d, 0x42, 0x9c, 0xf9, 0xde, 0xb0, 0x9c, 0xc4,
0x51, 0x8d, 0xc5, 0xf5, 0x7c, 0x08, 0x9a, 0xa2, 0xd3, 0x42, 0x0e, 0x56, 0x7e,
0x73, 0x21, 0x02, 0xc2, 0xc9, 0x2b, 0x88, 0xa0, 0x7c, 0x69, 0xd7, 0x09, 0x17,
0x14, 0x0a, 0xb3, 0x82, 0x3c, 0x63, 0xf3, 0x12, 0xd3, 0xf1, 0x1f, 0xa8, 0x7b,
0xa2, 0x9d, 0xa3, 0xc7, 0x22, 0x4b, 0x4f, 0xb4, 0xbc
},
.plaintext = (unsigned char []){
0x7e, 0x65, 0xb9, 0x98, 0xa0, 0x5f, 0x62, 0x6b, 0x02, 0x8c, 0x75, 0xdc, 0x3f,
0xbf, 0x98, 0x96, 0x3d, 0xce, 0x66, 0xd0, 0xf4, 0xc3, 0xae, 0x42, 0x37, 0xcf,
0xf3, 0x04, 0xd8, 0x4d, 0x88, 0x36, 0xcb, 0x6b, 0xad, 0x9a, 0xc8, 0x6f, 0x9d,
0x1b, 0x8a, 0x28, 0xdd, 0x70, 0x40, 0x47, 0x88, 0xb8, 0x69, 0xd2, 0x42, 0x9f,
0x1e, 0xc0, 0x66, 0x3e, 0x51, 0xb7, 0x53, 0xf7, 0x45, 0x1c, 0x6b, 0x46, 0x45,
0xd9, 0x91, 0x26, 0xe4, 0x57, 0xc1, 0xda, 0xc4, 0x95, 0x51, 0xd8, 0x6a, 0x8a,
0x97, 0x4a, 0x31, 0x31, 0xe9, 0xb3, 0x71, 0xd5, 0xc2, 0x14, 0xcc, 0x9f, 0xf2,
0x40, 0xc2, 0x99, 0xbd, 0x0e, 0x62, 0xdb, 0xc7, 0xa9, 0xa2, 0xda, 0xd9, 0xfa,
0x54, 0x04, 0xad, 0xb0, 0x06, 0x32, 0xd3, 0x63, 0x32, 0xd5, 0xbe, 0x61, 0x06,
0xe9, 0xe6, 0xec, 0x81, 0xca, 0xc4, 0x5c, 0xd3, 0x39, 0xcc, 0x87, 0xab, 0xbe,
0x7f, 0x89, 0x43, 0x08, 0x00, 0xe1, 0x6e, 0x03, 0x2a, 0x66, 0x21, 0x0b, 0x25,
0xe9, 0x26, 0xed, 0xa2, 0x43, 0xd9, 0xf0, 0x99, 0x55, 0x49, 0x6d, 0xdb, 0xc7,
0x7e, 0xf7, 0x4f, 0x17, 0xfe, 0xe4, 0x1c, 0x44, 0x35, 0xe7, 0x8b, 0x46, 0x96,
0x5b, 0x71, 0x3d, 0x72, 0xce, 0x8a, 0x31, 0xaf, 0x64, 0x15, 0x38, 0xad, 0xd3,
0x87, 0xfe, 0xdf, 0xd8, 0x8b, 0xb2, 0x2a, 0x42, 0xeb, 0x3b, 0xda, 0x40, 0xf7,
0x2e, 0xca, 0xd9, 0x41, 0xdb, 0xff, 0xdd, 0x47, 0xb3, 0xe7, 0x77, 0x37, 0xda,
0x74, 0x15, 0x53, 0xa4, 0x5b, 0x63, 0x0d, 0x07, 0x0b, 0xcc, 0x52, 0x05, 0x80,
0x4b, 0xf8, 0x0e, 0xe2, 0xd5, 0x16, 0x12, 0x87, 0x5d, 0xbc, 0x47, 0x96, 0x96,
0x00, 0x52, 0xf1, 0x68, 0x7e, 0x00, 0x74, 0x00, 0x7e, 0x6a, 0x33, 0xab, 0x8b,
0x20, 0x85, 0xc0, 0x33, 0xf9, 0x89, 0x2b, 0x6f, 0x74
},
.mod = 2048,
.dq = (unsigned char []){
0xCB, 0x5B, 0x75, 0x8E, 0x65, 0x25, 0xB3, 0x1C, 0x04, 0x67, 0x22, 0xB5, 0x9E,
0x10, 0x6A, 0xA9, 0x65, 0x65, 0xF9, 0x8E, 0xCF, 0xF8, 0xF5, 0xCC, 0x47, 0xB4,
0x8A, 0x68, 0x24, 0xF7, 0xEB, 0x55, 0x30, 0xD2, 0x1D, 0x71, 0xF1, 0x37, 0xBB,
0xD6, 0xE8, 0x1A, 0xE1, 0x45, 0x56, 0x1F, 0x43, 0x74, 0x4B, 0x9C, 0x45, 0x53,
0xF1, 0xFD, 0x08, 0x4E, 0x0D, 0xA4, 0xD2, 0x31, 0x46, 0x24, 0x8D, 0x45, 0x49,
0xEE, 0xA4, 0x0E, 0x1B, 0xFC, 0x7B, 0x54, 0x23, 0xE0, 0xF5, 0x7B, 0xE2, 0xEE,
0x53, 0xD4, 0xC7, 0xB6, 0xBB, 0xE1, 0x5A, 0xF1, 0x79, 0xB0, 0x04, 0x2F, 0x68,
0xD5, 0xB2, 0x88, 0x20, 0xA0, 0x19, 0xC8, 0x2C, 0x45, 0x45, 0xAD, 0x82, 0xEB,
0xBE, 0xC1, 0x49, 0x45, 0x88, 0x4D, 0x1B, 0xE1, 0x2D, 0x8E, 0x03, 0x43, 0x6D,
0x60, 0x30, 0x24, 0xE4, 0xE8, 0x1B, 0x01, 0xB7, 0x58, 0x0F, 0xA9
},
.dp = (unsigned char []){
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,	/* 8 bytes zero padding */
0x7A, 0x54, 0x57, 0x5B, 0x8A, 0x86, 0x55, 0x18, 0x23, 0xB7, 0x7B, 0x64, 0x0C,
0x2B, 0xCF, 0x93, 0x09, 0xE5, 0x58, 0x97, 0xD1, 0x11, 0xF5, 0x7E, 0xA8, 0xFD,
0xE2, 0xC6, 0x32, 0xCC, 0x41, 0xDD, 0xFF, 0x74, 0x81, 0x92, 0x7C, 0x07, 0xC5,
0x04, 0xBF, 0xB4, 0x3D, 0xB6, 0xA1, 0x4F, 0x1F, 0x21, 0xA5, 0xAC, 0x8D, 0x71,
0xFC, 0x73, 0x41, 0x48, 0x3F, 0xF9, 0xCE, 0x2D, 0x4E, 0xD7, 0xEE, 0x94, 0x67,
0xC7, 0x49, 0x23, 0xED, 0x35, 0x1A, 0x34, 0xC2, 0xC0, 0x83, 0x70, 0x93, 0xF2,
0x07, 0x77, 0xC1, 0x2A, 0x6E, 0xC5, 0x7E, 0x82, 0xED, 0xD5, 0x26, 0xBB, 0x14,
0x17, 0x20, 0x9E, 0xBA, 0x2F, 0x59, 0x7E, 0x21, 0x50, 0x24, 0x08, 0x2C, 0xD2,
0xEF, 0xFB, 0xB5, 0xD7, 0xC2, 0x16, 0x5E, 0x6D, 0x3A, 0x2F, 0x19, 0x86, 0xB3,
0xA1, 0x1D, 0x60, 0x25, 0xD3, 0x0E, 0x96, 0xE2, 0x57, 0x9F, 0xC9
},
.qinv = (unsigned char []){
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,	/* 8 bytes zero padding */
0x17, 0x2B, 0xDA, 0x79, 0xDE, 0x9F, 0xEE, 0x37, 0x61, 0xA3, 0x49, 0xC9, 0x4D,
0x28, 0xA0, 0x24, 0x03, 0xB6, 0x0A, 0xDE, 0x88, 0xC9, 0x26, 0xEB, 0xC6, 0xAF,
0x46, 0x49, 0xBE, 0x73, 0x33, 0x30, 0x72, 0xAB, 0xFE, 0x7B, 0x3E, 0xE4, 0x4C,
0xAF, 0xA9, 0x0C, 0xD0, 0xF7, 0x61, 0xFC, 0xCE, 0x20, 0x35, 0xE5, 0x2E, 0x53,
0x9B, 0x05, 0x29, 0x3C, 0xAC, 0x05, 0x63, 0xED, 0x32, 0x93, 0xB3, 0x9D, 0xB8,
0x9D, 0x53, 0x5E, 0xB8, 0xC5, 0xDE, 0xC9, 0xEF, 0x6B, 0x2F, 0x1D, 0x94, 0x1B,
0x2F, 0xE9, 0xB4, 0x8A, 0xE6, 0xCA, 0x15, 0xCA, 0xC7, 0x08, 0xDA, 0x35, 0xD2,
0xBA, 0x6F, 0x33, 0x31, 0x18, 0x1C, 0xE8, 0x46, 0xED, 0x6D, 0xFA, 0xEF, 0xE5,
0x57, 0x6A, 0x65, 0x01, 0x9C, 0xD7, 0xF7, 0x04, 0xB3, 0xFF, 0xA5, 0x94, 0xCF,
0x41, 0x8B, 0x14, 0xCA, 0x62, 0x02, 0xD0, 0x9E, 0x22, 0x60, 0xE4
},
},
};

const struct des3_ctr_tv DES3_CTR_TV[] = {
{
.len = 64,
.key = {
0x8e, 0x73, 0xb0, 0xf7, 0xda, 0x0e, 0x64, 0x52, 0xc8, 0x10, 0xf3, 0x2b, 0x80,
0x90, 0x79, 0xe5, 0x62, 0xf8, 0xea, 0xd2, 0x52, 0x2c, 0x6b, 0x7b
},
.ctr = {
0xf8, 0xf9, 0xfa, 0xfb, 0xfc, 0xfd, 0xfe, 0xff
},
.plaintext = (unsigned char []){
0x6b, 0xc1, 0xbe, 0xe2, 0x2e, 0x40, 0x9f, 0x96, 0xe9, 0x3d, 0x7e, 0x11, 0x73,
0x93, 0x17, 0x2a, 0xae, 0x2d, 0x8a, 0x57, 0x1e, 0x03, 0xac, 0x9c, 0x9e, 0xb7,
0x6f, 0xac, 0x45, 0xaf, 0x8e, 0x51, 0x30, 0xc8, 0x1c, 0x46, 0xa3, 0x5c, 0xe4,
0x11, 0xe5, 0xfb, 0xc1, 0x19, 0x1a, 0x0a, 0x52, 0xef, 0xf6, 0x9f, 0x24, 0x45,
0xdf, 0x4f, 0x9b, 0x17, 0xad, 0x2b, 0x41, 0x7b, 0xe6, 0x6c, 0x37, 0x10
},
.ciphertext = (unsigned char []){
0x17, 0x4d, 0xdf, 0xde, 0x7b, 0xe0, 0x2f, 0xb7, 0x58, 0x49, 0x76, 0xe5, 0x80,
0xbd, 0x49, 0x45, 0x64, 0x3a, 0xe4, 0x42, 0xfe, 0x4c, 0x25, 0xd4, 0x79, 0x74,
0xf0, 0xe6, 0x0b, 0x3d, 0x20, 0xac, 0x04, 0x42, 0xe1, 0x00, 0xe8, 0xd0, 0x35,
0x18, 0xad, 0xf6, 0x44, 0x23, 0x4b, 0xb7, 0x05, 0xf7, 0xf8, 0x3a, 0xf7, 0x9a,
0x8b, 0x62, 0x46, 0x3c, 0x2c, 0xaa, 0x45, 0x77, 0xbe, 0xab, 0x94, 0x80
},
},
};

const struct aes_ctr_tv AES_CTR_TV[] = {
{
.keylen = AES128_KEYLEN,
.len = 64,
.key = (unsigned char []){
0x2b, 0x7e, 0x15, 0x16, 0x28, 0xae, 0xd2, 0xa6, 0xab, 0xf7, 0x15, 0x88, 0x09,
0xcf, 0x4f, 0x3c
},
.ctr = {
0xf0, 0xf1, 0xf2, 0xf3, 0xf4, 0xf5, 0xf6, 0xf7, 0xf8, 0xf9, 0xfa, 0xfb, 0xfc,
0xfd, 0xfe, 0xff
},
.plaintext = (unsigned char []){
0x6b, 0xc1, 0xbe, 0xe2, 0x2e, 0x40, 0x9f, 0x96, 0xe9, 0x3d, 0x7e, 0x11, 0x73,
0x93, 0x17, 0x2a, 0xae, 0x2d, 0x8a, 0x57, 0x1e, 0x03, 0xac, 0x9c, 0x9e, 0xb7,
0x6f, 0xac, 0x45, 0xaf, 0x8e, 0x51, 0x30, 0xc8, 0x1c, 0x46, 0xa3, 0x5c, 0xe4,
0x11, 0xe5, 0xfb, 0xc1, 0x19, 0x1a, 0x0a, 0x52, 0xef, 0xf6, 0x9f, 0x24, 0x45,
0xdf, 0x4f, 0x9b, 0x17, 0xad, 0x2b, 0x41, 0x7b, 0xe6, 0x6c, 0x37, 0x10
},
.ciphertext = (unsigned char []){
0x87, 0x4d, 0x61, 0x91, 0xb6, 0x20, 0xe3, 0x26, 0x1b, 0xef, 0x68, 0x64, 0x99,
0x0d, 0xb6, 0xce, 0x98, 0x06, 0xf6, 0x6b, 0x79, 0x70, 0xfd, 0xff, 0x86, 0x17,
0x18, 0x7b, 0xb9, 0xff, 0xfd, 0xff, 0x5a, 0xe4, 0xdf, 0x3e, 0xdb, 0xd5, 0xd3,
0x5e, 0x5b, 0x4f, 0x09, 0x02, 0x0d, 0xb0, 0x3e, 0xab, 0x1e, 0x03, 0x1d, 0xda,
0x2f, 0xbe, 0x03, 0xd1, 0x79, 0x21, 0x70, 0xa0, 0xf3, 0x00, 0x9c, 0xee
},
},
{
.keylen = AES192_KEYLEN,
.len = 64,
.key = (unsigned char []){
0x8e, 0x73, 0xb0, 0xf7, 0xda, 0x0e, 0x64, 0x52, 0xc8, 0x10, 0xf3, 0x2b, 0x80,
0x90, 0x79, 0xe5, 0x62, 0xf8, 0xea, 0xd2, 0x52, 0x2c, 0x6b, 0x7b
},
.ctr = {
0xf0, 0xf1, 0xf2, 0xf3, 0xf4, 0xf5, 0xf6, 0xf7, 0xf8, 0xf9, 0xfa, 0xfb, 0xfc,
0xfd, 0xfe, 0xff
},
.plaintext = (unsigned char []){
0x6b, 0xc1, 0xbe, 0xe2, 0x2e, 0x40, 0x9f, 0x96, 0xe9, 0x3d, 0x7e, 0x11, 0x73,
0x93, 0x17, 0x2a, 0xae, 0x2d, 0x8a, 0x57, 0x1e, 0x03, 0xac, 0x9c, 0x9e, 0xb7,
0x6f, 0xac, 0x45, 0xaf, 0x8e, 0x51, 0x30, 0xc8, 0x1c, 0x46, 0xa3, 0x5c, 0xe4,
0x11, 0xe5, 0xfb, 0xc1, 0x19, 0x1a, 0x0a, 0x52, 0xef, 0xf6, 0x9f, 0x24, 0x45,
0xdf, 0x4f, 0x9b, 0x17, 0xad, 0x2b, 0x41, 0x7b, 0xe6, 0x6c, 0x37, 0x10
},
.ciphertext = (unsigned char []){
0x1a, 0xbc, 0x93, 0x24, 0x17, 0x52, 0x1c, 0xa2, 0x4f, 0x2b, 0x04, 0x59, 0xfe,
0x7e, 0x6e, 0x0b, 0x09, 0x03, 0x39, 0xec, 0x0a, 0xa6, 0xfa, 0xef, 0xd5, 0xcc,
0xc2, 0xc6, 0xf4, 0xce, 0x8e, 0x94, 0x1e, 0x36, 0xb2, 0x6b, 0xd1, 0xeb, 0xc6,
0x70, 0xd1, 0xbd, 0x1d, 0x66, 0x56, 0x20, 0xab, 0xf7, 0x4f, 0x78, 0xa7, 0xf6,
0xd2, 0x98, 0x09, 0x58, 0x5a, 0x97, 0xda, 0xec, 0x58, 0xc6, 0xb0, 0x50
},
},
{
.keylen = AES256_KEYLEN,
.len = 64,
.key = (unsigned char []){
0x60, 0x3d, 0xeb, 0x10, 0x15, 0xca, 0x71, 0xbe, 0x2b, 0x73, 0xae, 0xf0, 0x85,
0x7d, 0x77, 0x81, 0x1f, 0x35, 0x2c, 0x07, 0x3b, 0x61, 0x08, 0xd7, 0x2d, 0x98,
0x10, 0xa3, 0x09, 0x14, 0xdf, 0xf4
},
.ctr = {
0xf0, 0xf1, 0xf2, 0xf3, 0xf4, 0xf5, 0xf6, 0xf7, 0xf8, 0xf9, 0xfa, 0xfb, 0xfc,
0xfd, 0xfe, 0xff
},
.plaintext = (unsigned char []){
0x6b, 0xc1, 0xbe, 0xe2, 0x2e, 0x40, 0x9f, 0x96, 0xe9, 0x3d, 0x7e, 0x11, 0x73,
0x93, 0x17, 0x2a, 0xae, 0x2d, 0x8a, 0x57, 0x1e, 0x03, 0xac, 0x9c, 0x9e, 0xb7,
0x6f, 0xac, 0x45, 0xaf, 0x8e, 0x51, 0x30, 0xc8, 0x1c, 0x46, 0xa3, 0x5c, 0xe4,
0x11, 0xe5, 0xfb, 0xc1, 0x19, 0x1a, 0x0a, 0x52, 0xef, 0xf6, 0x9f, 0x24, 0x45,
0xdf, 0x4f, 0x9b, 0x17, 0xad, 0x2b, 0x41, 0x7b, 0xe6, 0x6c, 0x37, 0x10
},
.ciphertext = (unsigned char []){
0x60, 0x1e, 0xc3, 0x13, 0x77, 0x57, 0x89, 0xa5, 0xb7, 0xa7, 0xf5, 0x04, 0xbb,
0xf3, 0xd2, 0x28, 0xf4, 0x43, 0xe3, 0xca, 0x4d, 0x62, 0xb5, 0x9a, 0xca, 0x84,
0xe9, 0x90, 0xca, 0xca, 0xf5, 0xc5, 0x2b, 0x09, 0x30, 0xda, 0xa2, 0x3d, 0xe9,
0x4c, 0xe8, 0x70, 0x17, 0xba, 0x2d, 0x84, 0x98, 0x8d, 0xdf, 0xc9, 0xc5, 0x8d,
0xb6, 0x7a, 0xad, 0xa6, 0x13, 0xc2, 0xdd, 0x08, 0x45, 0x79, 0x41, 0xa6
},
},
};

const struct des3_cmac_tv DES3_CMAC_TV[] = {
{
.msglen = 32,
.maclen = 8,
.key = {
0x37, 0xea, 0xe9, 0x8f, 0xf4, 0x2a, 0xfb, 0x25, 0xf2, 0x23, 0x1c, 0x02, 0x8c,
0x29, 0xda, 0x9d, 0xef, 0x3d, 0xa8, 0xd0, 0xc7, 0x7f, 0xbf, 0x45
},
.msg = (unsigned char []){
0xf2, 0x66, 0xce, 0xc0, 0x1c, 0x5f, 0xc0, 0x8c, 0x0b, 0xda, 0xbc, 0x95, 0x37,
0xbd, 0x1a, 0xa2, 0xdf, 0x9f, 0x2b, 0x8f, 0xfb, 0xe5, 0xcc, 0x94, 0x72, 0x2a,
0x3b, 0xca, 0x8d, 0xe5, 0x29, 0xea
},
.mac = (unsigned char []){
0xb3, 0x80, 0x9c, 0x8b, 0x0e, 0xb9, 0xdd, 0x8d
},
.rv = 0,
},
{
.msglen = 32,
.maclen = 8,
.key = {
0x13, 0x58, 0xfb, 0x67, 0x15, 0x5e, 0x01, 0x45, 0xd0, 0x2c, 0x54, 0xa1, 0x20,
0x6b, 0x5d, 0x7f, 0x1c, 0x04, 0xba, 0x46, 0xc7, 0x4a, 0x5d, 0x49
},
.msg = (unsigned char []){
0xe3, 0x8b, 0x4c, 0x3e, 0x7a, 0x82, 0x64, 0x3b, 0xeb, 0x31, 0x92, 0x42, 0x65,
0x55, 0xad, 0x9c, 0x9b, 0x26, 0x20, 0xd6, 0x77, 0x37, 0x3f, 0xc4, 0x0c, 0x9d,
0xdb, 0xc4, 0xcd, 0x53, 0x13, 0x47
},
.mac = (unsigned char []){
0xb0, 0x00, 0xe2, 0xea, 0x1e, 0xf4, 0x8a, 0x8f
},
.rv = EFAULT,
},
};

const struct aes_cmac_tv AES_CMAC_TV[] = {
{
.keylen = AES128_KEYLEN,
.msglen = 65,
.maclen = 16,
.key = (unsigned char []){
0xd1, 0x49, 0x44, 0x1e, 0x66, 0x7b, 0x24, 0x5d, 0x46, 0x40, 0xe0, 0x4c, 0x53,
0xca, 0x6f, 0x51
},
.msg = (unsigned char []){
0xcb, 0xb3, 0x47, 0x94, 0xbc, 0x8b, 0xfd, 0xf9, 0x3d, 0x3c, 0x8d, 0x9f, 0x87,
0xec, 0x14, 0x82, 0xb5, 0x16, 0xb4, 0x8b, 0x1e, 0x8a, 0x89, 0xb5, 0xe3, 0xb5,
0xdf, 0x70, 0xc4, 0x23, 0xa2, 0x43, 0x38, 0x42, 0x15, 0xb4, 0xbc, 0x69, 0xc7,
0x6c, 0x6b, 0x18, 0xc4, 0x97, 0xcf, 0x82, 0x08, 0x8a, 0xf7, 0x48, 0x39, 0xa8,
0xc9, 0x88, 0x95, 0x86, 0x9a, 0x16, 0x29, 0x4d, 0xfc, 0x09, 0x43, 0x60, 0xd7
},
.mac = (unsigned char []){
0x64, 0xf5, 0xe8, 0xdc, 0xe5, 0xc3, 0xe0, 0xf9, 0xcc, 0x22, 0x4e, 0x30, 0x6d,
0xe7, 0x0b, 0x87
},
.rv = 0,
},
{
.keylen = AES128_KEYLEN,
.msglen = 65,
.maclen = 16,
.key = (unsigned char []){
0xc2, 0xf5, 0xd4, 0x83, 0x7f, 0x9f, 0x75, 0xfb, 0x44, 0x0c, 0x3b, 0xcc, 0xad,
0x7c, 0x2e, 0x69
},
.msg = (unsigned char []){
0x6a, 0x84, 0x59, 0x4c, 0x4b, 0x38, 0x65, 0xf0, 0x47, 0xc9, 0x60, 0x38, 0x06,
0x0b, 0x5b, 0x41, 0x3d, 0xb0, 0xd4, 0xe0, 0x81, 0xc6, 0x2e, 0x40, 0x5b, 0x81,
0x5e, 0xcd, 0x9e, 0x3b, 0xe6, 0x51, 0xf8, 0xb9, 0x07, 0x5d, 0xc8, 0xb0, 0x32,
0xeb, 0x2f, 0x87, 0xc1, 0x41, 0x6a, 0x5f, 0xe4, 0x19, 0x5f, 0x51, 0xde, 0xfe,
0x75, 0xf6, 0x71, 0xf9, 0xa9, 0x2d, 0x96, 0x6d, 0xdf, 0x18, 0x72, 0x40, 0x75
},
.mac = (unsigned char []){
0xdf, 0x8c, 0x8c, 0x61, 0xe8, 0xd6, 0x04, 0xe2, 0x4c, 0x7e, 0x3d, 0x01, 0x15,
0xdb, 0xe8, 0x98
},
.rv = EFAULT,
},
{
.keylen = AES192_KEYLEN,
.msglen = 64,
.maclen = 8,
.key = (unsigned char []){
0x20, 0xd2, 0x2c, 0x3b, 0x6a, 0xb3, 0x8c, 0x59, 0x95, 0xe2, 0x2b, 0x34, 0x1f,
0x35, 0x9b, 0xe2, 0x56, 0x16, 0xb2, 0xb8, 0xc7, 0x26, 0x95, 0x10
},
.msg = (unsigned char []){
0xa1, 0xc0, 0x41, 0xd1, 0xd4, 0xe7, 0xcd, 0x6a, 0x95, 0x3f, 0x2e, 0x48, 0x37,
0xe3, 0xe6, 0x76, 0xed, 0x48, 0x63, 0x3a, 0x2f, 0x15, 0x82, 0x8f, 0x5f, 0x35,
0x51, 0xd5, 0xad, 0x2a, 0x19, 0xc8, 0x38, 0xa4, 0x9c, 0xaf, 0x75, 0x52, 0x9b,
0xd5, 0xd5, 0xf8, 0x9b, 0x3d, 0xa2, 0xc2, 0xe9, 0x92, 0x2a, 0xd8, 0xdc, 0x5d,
0x20, 0x32, 0x5a, 0x7b, 0x3f, 0xae, 0x9d, 0xcd, 0x30, 0x5f, 0x37, 0x31
},
.mac = (unsigned char []){
0xaa, 0xc4, 0xd4, 0xf4, 0x17, 0x2e, 0x1f, 0x05
},
.rv = 0,
},
{
.keylen = AES192_KEYLEN,
.msglen = 64,
.maclen = 8,
.key = (unsigned char []){
0x22, 0xe2, 0x9a, 0xa7, 0x54, 0x7e, 0x5e, 0xd3, 0xa6, 0x46, 0x11, 0xe0, 0x4f,
0x1d, 0x55, 0xf7, 0xa3, 0x97, 0xc1, 0x61, 0x96, 0x69, 0x87, 0x9c
},
.msg = (unsigned char []){
0x77, 0x3b, 0x57, 0x7b, 0x95, 0xe2, 0x9d, 0x36, 0xfb, 0x30, 0x77, 0x9d, 0x2e,
0xa2, 0x3e, 0x2f, 0xfe, 0xd9, 0xe1, 0xb4, 0x6a, 0xed, 0xe4, 0x2b, 0xbe, 0x03,
0xa9, 0x04, 0xfe, 0x22, 0xef, 0x8f, 0x87, 0x42, 0x98, 0xb5, 0xf4, 0xa6, 0xaf,
0xe6, 0x3f, 0x6c, 0xa9, 0x52, 0x28, 0x63, 0xeb, 0x5c, 0xdb, 0x1c, 0x8d, 0x4b,
0xcd, 0x44, 0x5e, 0x43, 0xe7, 0x30, 0x28, 0x75, 0xe6, 0xba, 0x35, 0x92
},
.mac = (unsigned char []){
0x16, 0xbf, 0x98, 0xc7, 0xa5, 0xde, 0xff, 0x18
},
.rv = EFAULT,
},
{
.keylen = AES256_KEYLEN,
.msglen = 64,
.maclen = 8,
.key = (unsigned char []){
0xf7, 0x0b, 0x8a, 0x4e, 0xee, 0x35, 0x18, 0xbb, 0xa0, 0x71, 0xaf, 0x55, 0xf2,
0x5f, 0x7b, 0x69, 0x8a, 0x5b, 0x7d, 0xc8, 0x86, 0x5c, 0xda, 0xca, 0x6d, 0x1c,
0x79, 0x93, 0x65, 0x7a, 0xcc, 0x95
},
.msg = (unsigned char []){
0x79, 0x5e, 0xe1, 0xaf, 0x75, 0x04, 0x62, 0x1a, 0xac, 0x32, 0x9f, 0x50, 0x81,
0x91, 0x2d, 0xe5, 0x45, 0xfa, 0x11, 0x17, 0x4f, 0x39, 0x79, 0xb1, 0x4f, 0x11,
0xaa, 0x30, 0xdf, 0x81, 0x3a, 0x23, 0x5b, 0x46, 0x7f, 0xd8, 0xf3, 0xa1, 0x47,
0x34, 0xfe, 0x5a, 0xc9, 0xe3, 0x91, 0x05, 0xdc, 0xb2, 0x51, 0x84, 0x67, 0x38,
0x85, 0xcd, 0x19, 0xbc, 0x70, 0xee, 0x5a, 0x53, 0xdd, 0x4e, 0x81, 0x49
},
.mac = (unsigned char []){
0x93, 0x54, 0x27, 0x34, 0xd6, 0xcd, 0x43, 0xde
},
.rv = 0,
},
{
.keylen = AES256_KEYLEN,
.msglen = 64,
.maclen = 8,
.key = (unsigned char []){
0x0c, 0x45, 0x6d, 0x19, 0x9a, 0xba, 0xe4, 0x75, 0x87, 0x34, 0xf5, 0x06, 0xc4,
0xe9, 0xcc, 0xdb, 0x76, 0x7e, 0x4f, 0xd1, 0x56, 0xd5, 0xa4, 0x08, 0x57, 0x26,
0xf3, 0x93, 0x8a, 0x51, 0x6d, 0x74
},
.msg = (unsigned char []){
0x78, 0xf3, 0xbf, 0x56, 0x8f, 0x1c, 0x3f, 0x28, 0x66, 0xef, 0xf8, 0xa2, 0x46,
0xa7, 0x0c, 0xf0, 0xfa, 0xee, 0x4c, 0x30, 0x78, 0xf3, 0xfb, 0x27, 0xc4, 0xbd,
0xd5, 0x33, 0x12, 0xbf, 0x50, 0x81, 0x2b, 0xac, 0x22, 0x80, 0x11, 0x8c, 0x03,
0x96, 0xe6, 0x10, 0xb4, 0x11, 0x0a, 0x22, 0x40, 0x60, 0x84, 0xc1, 0x82, 0x83,
0xa3, 0x0c, 0xe7, 0xc0, 0xe4, 0x9c, 0x76, 0x98, 0x17, 0x17, 0x0d, 0xf9
},
.mac = (unsigned char []){
0xc4, 0xc5, 0xbe, 0x3c, 0x94, 0xfb, 0x7b, 0x9c
},
.rv = EFAULT,
},
};

const struct aes_xts_tv AES_XTS_TV[] = {
{
.len = 256 / 8,
.keylen = AES128_KEYLEN,
.key1 = (unsigned char []){
0x78, 0x3a, 0x83, 0xec, 0x52, 0xa2, 0x74, 0x05, 0xdf, 0xf9, 0xde, 0x4c, 0x57,
0xf9, 0xc9, 0x79
},
.key2 = (unsigned char []){
0xb3, 0x60, 0xb6, 0xa5, 0xdf, 0x88, 0xd6, 0x7e, 0xc1, 0xa0, 0x52, 0xe6, 0xf5,
0x82, 0xa7, 0x17
},
.tweak = {
0x88, 0x6e, 0x97, 0x5b, 0x29, 0xbd, 0xf6, 0xf0, 0xc0, 0x1b, 0xb4, 0x7f, 0x61,
0xf6, 0xf0, 0xf5
},
.plaintext = (unsigned char []){
0xb0, 0x4d, 0x84, 0xda, 0x85, 0x6b, 0x9a, 0x59, 0xce, 0x2d, 0x62, 0x67, 0x46,
0xf6, 0x89, 0xa8, 0x05, 0x1d, 0xac, 0xd6, 0xbc, 0xe3, 0xb9, 0x90, 0xaa, 0x90,
0x1e, 0x40, 0x30, 0x64, 0x88, 0x79
},
.ciphertext = (unsigned char []){
0xf9, 0x41, 0x03, 0x9e, 0xba, 0xb8, 0xca, 0xc3, 0x9d, 0x59, 0x24, 0x7c, 0xbb,
0xcb, 0x4d, 0x81, 0x6c, 0x72, 0x6d, 0xae, 0xd1, 0x15, 0x77, 0x69, 0x2c, 0x55,
0xe4, 0xac, 0x6d, 0x3e, 0x68, 0x20
},
},
{
.len = 384 / 8,
.keylen = AES256_KEYLEN,
.key1 = (unsigned char []){
0x7f, 0xb0, 0x92, 0x2f, 0xce, 0x09, 0xed, 0xdd, 0x36, 0x65, 0xa1, 0x1f, 0x52,
0x35, 0xd5, 0x1e, 0xf7, 0x72, 0x06, 0xa7, 0xde, 0x45, 0x47, 0x75, 0xb6, 0x9d,
0xcc, 0x54, 0x59, 0xac, 0xdb, 0x24
},
.key2 = (unsigned char []){
0xcc, 0xf0, 0x5c, 0x41, 0x5a, 0xf5, 0xab, 0x8a, 0x06, 0x62, 0x3d, 0x19, 0x8d,
0x5b, 0x91, 0x85, 0x95, 0xa9, 0xdc, 0xaa, 0xf5, 0x39, 0x2e, 0xe7, 0x17, 0xc1,
0x04, 0x7f, 0x2f, 0x08, 0xf6, 0x2b
},
.tweak = {
0x8e, 0xa3, 0x63, 0x8b, 0x9d, 0x40, 0x62, 0xf1, 0x69, 0x19, 0x6f, 0xf4, 0x55,
0x5a, 0xd0, 0xaf
},
.plaintext = (unsigned char []){
0xbb, 0xb2, 0x34, 0xdb, 0x01, 0xbe, 0x79, 0xd3, 0xdc, 0x7d, 0xcf, 0x49, 0xbc,
0x53, 0xb8, 0xef, 0xc6, 0x2f, 0xe7, 0x17, 0x94, 0x39, 0x06, 0x1d, 0x73, 0xa6,
0xb2, 0x0e, 0xdb, 0x9b, 0x7d, 0xa0, 0x45, 0x0b, 0x19, 0xf0, 0x20, 0x82, 0x02,
0x09, 0xaf, 0xe0, 0x81, 0x12, 0xae, 0x4a, 0xfd, 0xd3
},
.ciphertext = (unsigned char []){
0xcb, 0xf6, 0x88, 0x48, 0xc4, 0x20, 0x49, 0xef, 0xb1, 0x46, 0xe2, 0xd1, 0xe4,
0x11, 0x5f, 0x62, 0xe4, 0xfa, 0xa4, 0xef, 0xff, 0x19, 0x8f, 0x1a, 0x0a, 0xa0,
0xc9, 0x8b, 0xa0, 0x2c, 0xc4, 0x4d, 0xa3, 0x76, 0xd7, 0x62, 0xd6, 0x15, 0x1f,
0x23, 0x45, 0x87, 0xbf, 0x12, 0x8b, 0x6a, 0x7f, 0xfa
},
},
};
const struct aes_gcm_tv AES_GCM_TV[] = {
{
.keylen = AES128_KEYLEN,
.ivlen = 1024 / 8,
.len = 408 / 8,
.aadlen = 720 / 8,
.taglen = 32 / 8,
.key = (unsigned char []){
0x73, 0xa0, 0xf9, 0xc5, 0xd2, 0x70, 0xf6, 0xcf, 0x41, 0xae, 0x23, 0x1c, 0x4f,
0x5a, 0x8c, 0x66
},
.iv = (unsigned char []){
0xf7, 0x32, 0xde, 0x5e, 0xcb, 0x76, 0x90, 0xd7, 0x25, 0xad, 0x2a, 0xdd, 0x4c,
0x6d, 0x22, 0xc6, 0x79, 0x74, 0xfd, 0x3b, 0x73, 0x5a, 0x58, 0xf8, 0x5c, 0x2a,
0x4e, 0x03, 0x10, 0x6d, 0xf2, 0x4e, 0x80, 0xaa, 0x85, 0x3f, 0x7c, 0x12, 0xbd,
0xaa, 0xe2, 0x73, 0x80, 0xb3, 0x44, 0x55, 0xb0, 0xfd, 0x14, 0x98, 0x63, 0xb6,
0xb0, 0x62, 0xcc, 0xe5, 0x7b, 0x9b, 0xaf, 0x74, 0x95, 0x1a, 0x10, 0xa3, 0x5a,
0xec, 0x7e, 0x00, 0xf5, 0xc3, 0x81, 0xfe, 0x18, 0x73, 0xdf, 0x48, 0xe6, 0x72,
0x6e, 0x11, 0x85, 0xe3, 0xc9, 0x06, 0xb5, 0x42, 0xf1, 0x3d, 0x60, 0x9d, 0xab,
0x9c, 0xc4, 0xfc, 0xf9, 0xfc, 0xfb, 0xb0, 0x73, 0x11, 0x3d, 0xcd, 0xa0, 0x37,
0x5d, 0xc5, 0x9b, 0x2c, 0x92, 0x88, 0x8c, 0x80, 0x1a, 0x34, 0x8a, 0x79, 0x72,
0x87, 0xca, 0x13, 0xca, 0x73, 0xfc, 0x88, 0x27, 0xda, 0x0e, 0x39
},
.plaintext = NULL,
.aad = (unsigned char []){
0x28, 0x7a, 0x84, 0xad, 0xae, 0x06, 0x17, 0xde, 0x9d, 0xf7, 0x55, 0x2f, 0xd1,
0x18, 0xde, 0x05, 0x6b, 0x17, 0xf5, 0xe5, 0x75, 0x20, 0x39, 0x48, 0xf8, 0xae,
0x11, 0x0e, 0xce, 0x97, 0xb2, 0x1b, 0x09, 0xfd, 0xf0, 0x82, 0xba, 0x30, 0x23,
0x63, 0x7e, 0x29, 0x97, 0x30, 0x29, 0x08, 0xd8, 0x83, 0x0a, 0xf5, 0x8c, 0x4d,
0xd0, 0x9d, 0xc0, 0x76, 0x78, 0xc1, 0xc0, 0xde, 0x31, 0x16, 0x73, 0x8c, 0x20,
0x86, 0x4f, 0xb2, 0x0c, 0xb3, 0xae, 0xe1, 0xfc, 0x65, 0x7d, 0x94, 0xfc, 0xda,
0x10, 0xa2, 0x47, 0xe8, 0x9f, 0x98, 0x99, 0x26, 0x57, 0xb6, 0xee, 0xdf
},
.tag = (unsigned char []){
0x73, 0x9f, 0xf5, 0x69
},
.ciphertext = (unsigned char []){
0xdb, 0x7c, 0x4e, 0x60, 0x6d, 0xb6, 0xe3, 0xbc, 0x14, 0x7a, 0x9e, 0xec, 0xfc,
0x76, 0x18, 0x2e, 0x66, 0xdd, 0xaa, 0x8e, 0xdb, 0x21, 0x5a, 0x97, 0x9d, 0x58,
0x2c, 0xfc, 0xc8, 0xb0, 0x28, 0x6b, 0x73, 0xdb, 0x6c, 0xbb, 0x18, 0x28, 0x1c,
0xb4, 0xa7, 0x62, 0xd0, 0x32, 0xad, 0x1c, 0x80, 0x16, 0xce, 0xa9, 0x2a
},
.rv = EFAULT,
},
{
.keylen = AES128_KEYLEN,
.ivlen = 1024 / 8,
.len = 408 / 8,
.aadlen = 720 / 8,
.taglen = 32 / 8,
.key = (unsigned char []){
0x39, 0xe0, 0xf6, 0xf4, 0x21, 0x8c, 0x54, 0x81, 0x71, 0xfa, 0xb2, 0x72, 0xdf,
0xe0, 0x65, 0x03
},
.iv = (unsigned char []){
0xe3, 0x90, 0x6e, 0x24, 0x83, 0x91, 0x1f, 0xc3, 0x4d, 0xf8, 0x59, 0xed, 0x45,
0x70, 0xe6, 0x7c, 0x2b, 0xf6, 0x15, 0x83, 0xa9, 0xd3, 0xbe, 0xb7, 0x57, 0x1d,
0xaa, 0x78, 0xd2, 0x5f, 0x26, 0xd1, 0xda, 0xe0, 0xad, 0xfe, 0x47, 0xb4, 0xad,
0x15, 0x6b, 0x73, 0x29, 0x2c, 0xe9, 0x04, 0xc5, 0xab, 0xa4, 0x22, 0x31, 0xec,
0x6b, 0xad, 0x86, 0x4a, 0x51, 0x32, 0x86, 0xac, 0xcd, 0xac, 0x7a, 0xaa, 0xea,
0x2c, 0xc5, 0x6b, 0x8f, 0x35, 0xd7, 0x91, 0x0c, 0x5c, 0xa8, 0x0e, 0x0d, 0xb9,
0xd5, 0xca, 0x57, 0x88, 0x78, 0x20, 0x88, 0x65, 0x28, 0x7f, 0x7b, 0x60, 0x61,
0xc1, 0x1d, 0x14, 0x6d, 0x95, 0x23, 0x28, 0x04, 0xca, 0x35, 0xd5, 0x6e, 0xad,
0x4d, 0xa5, 0xdd, 0x95, 0x0a, 0x16, 0xda, 0xf4, 0xcc, 0x22, 0x13, 0xe2, 0x5f,
0xe0, 0x14, 0x4a, 0x2a, 0xa9, 0x35, 0x73, 0xce, 0x22, 0xe1, 0x6b
},
.plaintext = (unsigned char []) {
0x4d, 0x81, 0x76, 0x44, 0x79, 0x42, 0x4b, 0xb7, 0xff, 0x0b, 0x36, 0x1f, 0x2f,
0xef, 0x8d, 0x0e, 0xcc, 0x47, 0xe7, 0xed, 0xdc, 0x4b, 0x98, 0xe7, 0xb3, 0xb6,
0x58, 0x19, 0x26, 0x93, 0x9c, 0x5c, 0xea, 0x8f, 0x58, 0xa6, 0x17, 0x57, 0x74,
0xd4, 0x7e, 0xdf, 0xe4, 0xb8, 0xf3, 0xa6, 0xc0, 0xb9, 0x3b, 0x56, 0x53
},
.aad = (unsigned char []){
0xbf, 0xb9, 0x37, 0x07, 0x39, 0xdc, 0x3d, 0x0c, 0x5c, 0xc1, 0x68, 0x25, 0xa8,
0xd9, 0x9d, 0x03, 0x69, 0xc3, 0x4c, 0x4b, 0xf0, 0x46, 0x18, 0x33, 0xe3, 0x4d,
0xf8, 0x1a, 0xb3, 0xff, 0xa0, 0xbf, 0x3f, 0x8f, 0xf8, 0xb5, 0xbc, 0xd3, 0x4c,
0x3a, 0xc8, 0xb6, 0xf1, 0xff, 0x35, 0x44, 0x28, 0x24, 0x1a, 0xd7, 0x03, 0x25,
0x34, 0xa3, 0x76, 0x1e, 0x05, 0xae, 0xb6, 0x8c, 0x2b, 0xa0, 0x0f, 0xf6, 0x3f,
0x27, 0xe6, 0x40, 0xf0, 0x01, 0xb6, 0xe7, 0x4e, 0xc9, 0x46, 0xc5, 0x60, 0xf2,
0xcb, 0x17, 0x14, 0xbb, 0x3a, 0xc9, 0x8b, 0xff, 0xc9, 0x92, 0xd6, 0xa2
},
.tag = (unsigned char []){
0x36, 0x2e, 0x23, 0x7d
},
.ciphertext = (unsigned char []){
0x3c, 0xaf, 0x65, 0xe4, 0xb9, 0x77, 0x3c, 0xee, 0x60, 0xbb, 0x85, 0xf6, 0xc5,
0xa4, 0x4b, 0xeb, 0x7d, 0xd3, 0xdc, 0xb3, 0x35, 0x12, 0xd5, 0x52, 0x48, 0x18,
0x4d, 0x6b, 0x72, 0xc9, 0x80, 0x38, 0xb3, 0xbc, 0xae, 0x17, 0xbe, 0x58, 0xd1,
0x49, 0x28, 0xe6, 0xbb, 0xa9, 0xbf, 0x92, 0x9d, 0x34, 0xf1, 0x0f, 0xfe
},
.rv = 0,
},
{
.keylen = AES192_KEYLEN,
.ivlen = 1024 / 8,
.len = 408 / 8,
.aadlen = 720 / 8,
.taglen = 32 / 8,
.key = (unsigned char []){
0xd0, 0x98, 0xb5, 0x67, 0xfa, 0x48, 0x22, 0x5c, 0xbe, 0x27, 0xdb, 0x9f, 0x89,
0xb4, 0xc0, 0x66, 0x57, 0xad, 0xf9, 0x77, 0x3a, 0x3e, 0xa9, 0x60
},
.iv = (unsigned char []){
0xaf, 0xd3, 0xda, 0x93, 0x10, 0x20, 0x56, 0xe7, 0x6b, 0x72, 0x82, 0x68, 0x51,
0x95, 0x9c, 0x06, 0xb5, 0x8a, 0x74, 0x33, 0x5f, 0x0a, 0x26, 0x4a, 0x7b, 0x24,
0xc7, 0x3f, 0x25, 0x5d, 0xb8, 0x73, 0x8b, 0xc6, 0xbd, 0x00, 0x42, 0xaf, 0x16,
0xe6, 0x1e, 0xbb, 0x64, 0x18, 0x6a, 0x88, 0xf5, 0xdc, 0xb7, 0x40, 0xfb, 0xe5,
0xeb, 0x4c, 0xb9, 0x13, 0xc4, 0xf2, 0xb9, 0x8b, 0x47, 0x24, 0x0c, 0x04, 0xde,
0x7c, 0x8c, 0x3c, 0x41, 0xca, 0x67, 0x93, 0x06, 0xb6, 0xa8, 0x3f, 0x6f, 0xad,
0x7e, 0x33, 0xa6, 0x2f, 0x30, 0x81, 0x6b, 0x6f, 0x1b, 0xde, 0x10, 0x6b, 0x1a,
0x4c, 0xa5, 0x81, 0xef, 0x36, 0x49, 0x04, 0x32, 0xfd, 0x4c, 0x83, 0x61, 0x33,
0x12, 0xdb, 0xd5, 0xd4, 0xc0, 0x3e, 0xf8, 0x83, 0x47, 0x6d, 0x97, 0x65, 0xae,
0xd3, 0xd4, 0x2a, 0xa7, 0x71, 0x67, 0xb7, 0xae, 0xeb, 0xcf, 0x88
},
.plaintext = NULL,
.aad = (unsigned char []){
0x8f, 0xcf, 0xfa, 0x39, 0x5f, 0xc0, 0x11, 0x81, 0xc0, 0xe0, 0xc0, 0x96, 0xa9,
0x8d, 0x40, 0x53, 0x57, 0x08, 0xa3, 0xf7, 0x1c, 0xe4, 0x4d, 0x8b, 0xba, 0xcd,
0xf7, 0xa7, 0xdd, 0x11, 0x9a, 0x50, 0x3a, 0xe5, 0x4b, 0x91, 0xbe, 0x91, 0x7c,
0xd0, 0x31, 0xb3, 0x53, 0x64, 0x31, 0x49, 0x4e, 0xc2, 0x5a, 0x6e, 0x1c, 0x26,
0x5c, 0x79, 0xa2, 0xe8, 0x47, 0x98, 0xe0, 0xca, 0xf1, 0x64, 0xd3, 0x16, 0x33,
0x2c, 0x9a, 0xf2, 0x67, 0x29, 0x3e, 0x1c, 0x06, 0x06, 0xe0, 0xa6, 0x46, 0x1a,
0x59, 0xb4, 0x81, 0xe0, 0x6a, 0x69, 0x79, 0x43, 0x11, 0x76, 0xfe, 0x74
},
.tag = (unsigned char []){
0x49, 0x5c, 0x55, 0x5d
},
.ciphertext = (unsigned char []){
0x2c, 0x81, 0x49, 0x87, 0xdc, 0x8f, 0xe0, 0x45, 0x6f, 0x17, 0xb3, 0x29, 0x21,
0x1c, 0x29, 0xc2, 0xe5, 0xa3, 0xb5, 0x60, 0x02, 0x80, 0xd4, 0x13, 0x6f, 0x09,
0xa2, 0x88, 0x2a, 0xa1, 0x06, 0x74, 0x2c, 0x55, 0x99, 0xb2, 0xe4, 0xd9, 0x3d,
0xf8, 0x87, 0x71, 0x73, 0x1f, 0x30, 0xfd, 0xbe, 0xf0, 0x7b, 0x46, 0x48
},
.rv = EFAULT,
},
{
.keylen = AES192_KEYLEN,
.ivlen = 1024 / 8,
.len = 408 / 8,
.aadlen = 720 / 8,
.taglen = 32 / 8,
.key = (unsigned char []){
0x89, 0xd9, 0x4a, 0x67, 0x35, 0x35, 0x90, 0x0d, 0x5e, 0xbc, 0x1f, 0x60, 0x1c,
0x03, 0xff, 0xfd, 0xaf, 0xb6, 0x7e, 0xfe, 0xb7, 0xda, 0x1c, 0x0b
},
.iv = (unsigned char []){
0x71, 0x5e, 0xb8, 0x27, 0x88, 0xa7, 0x76, 0x57, 0x52, 0x30, 0x34, 0x6c, 0x7d,
0x7c, 0xbc, 0x20, 0x1f, 0xd7, 0x42, 0xd2, 0x78, 0x16, 0x6b, 0x2d, 0x9f, 0x77,
0xb4, 0x78, 0x64, 0xf6, 0xb9, 0x8e, 0x1e, 0xd1, 0x90, 0x48, 0x0b, 0x4e, 0x4a,
0x79, 0xdc, 0xa0, 0xd9, 0xef, 0xd9, 0xd2, 0x6b, 0x90, 0x10, 0x54, 0x9d, 0x7c,
0x3d, 0x8c, 0x2e, 0xbd, 0x01, 0xf5, 0x59, 0xef, 0xd6, 0xf5, 0x36, 0x1a, 0xb2,
0xfd, 0xab, 0x1f, 0x0c, 0xc7, 0xd0, 0x36, 0x52, 0x8c, 0xcc, 0x80, 0x7e, 0xb4,
0x02, 0xd2, 0x1b, 0x66, 0xa5, 0x2a, 0x8f, 0xf4, 0xaa, 0xb0, 0xd3, 0x37, 0x7a,
0x6b, 0xca, 0x45, 0xcc, 0x9a, 0x58, 0x47, 0x9a, 0x93, 0x98, 0x16, 0xb5, 0x1c,
0x86, 0xc4, 0x45, 0xdc, 0x51, 0xeb, 0xf3, 0x6d, 0x6d, 0x72, 0xdb, 0x7f, 0xe5,
0xf7, 0xf9, 0x0c, 0x13, 0x14, 0x06, 0x4b, 0xc2, 0x29, 0xb6, 0xca
},
.plaintext = (unsigned char []) {
0x8f, 0x8b, 0x42, 0xb5, 0x2c, 0xc9, 0xa0, 0x80, 0xe8, 0x4b, 0x48, 0x0e, 0xca,
0x78, 0xd1, 0xb6, 0x74, 0x5c, 0x63, 0x69, 0x4a, 0x70, 0xa3, 0xb4, 0xc1, 0xd4,
0x57, 0xfe, 0xa9, 0x82, 0x1b, 0x63, 0x34, 0x38, 0xa5, 0x11, 0x19, 0xd4, 0x2b,
0xc7, 0x01, 0xa0, 0x5a, 0xb7, 0x55, 0x3f, 0xd4, 0xe6, 0x17, 0x13, 0x64
},
.aad = (unsigned char []){
0x40, 0x66, 0x52, 0xd7, 0x9e, 0x0f, 0xef, 0x78, 0x1e, 0x5d, 0x74, 0xe9, 0x01,
0x73, 0x60, 0x42, 0x3f, 0x31, 0x8c, 0x95, 0x41, 0x8b, 0xaa, 0x49, 0x8e, 0xe0,
0xc5, 0x0d, 0xad, 0x71, 0xc4, 0xed, 0xaa, 0xe9, 0xbd, 0x6f, 0x57, 0x57, 0x32,
0xb7, 0x36, 0xf2, 0x8d, 0xf1, 0x5e, 0x92, 0x1f, 0x15, 0x0c, 0xab, 0x05, 0x87,
0x25, 0xab, 0xda, 0xa4, 0x23, 0x12, 0x7e, 0x47, 0x12, 0x26, 0xff, 0x0c, 0x53,
0x8c, 0xcd, 0x8e, 0x36, 0x3b, 0x21, 0x58, 0xc0, 0x70, 0x55, 0xc2, 0xaa, 0x97,
0x57, 0x7d, 0xd1, 0xec, 0x19, 0x58, 0x43, 0x49, 0x48, 0xb0, 0x95, 0x0e
},
.tag = (unsigned char []){
0x64, 0x35, 0xff, 0xde
},
.ciphertext = (unsigned char []){
0x12, 0x92, 0xe7, 0xac, 0xb4, 0x8a, 0xb1, 0xda, 0xd5, 0x0f, 0x28, 0x1c, 0x6e,
0xc3, 0x5c, 0x98, 0x00, 0x1d, 0x3d, 0xbf, 0x98, 0x34, 0x28, 0x23, 0x88, 0xaa,
0x43, 0xcd, 0x17, 0x51, 0x1b, 0x65, 0x6e, 0xa7, 0x95, 0x5b, 0x57, 0x61, 0xe4,
0x2f, 0x7a, 0xa6, 0xcb, 0x58, 0xc9, 0xb0, 0x0f, 0xc2, 0x7b, 0x22, 0x4d
},
.rv = 0,
},
{
.keylen = AES256_KEYLEN,
.ivlen = 1024 / 8,
.len = 408 / 8,
.aadlen = 720 / 8,
.taglen = 32 / 8,
.key = (unsigned char []){
0xc8, 0x18, 0xc1, 0xa7, 0x53, 0xe7, 0x84, 0x1f, 0x58, 0x43, 0x87, 0xeb, 0x7a,
0x38, 0x1f, 0x99, 0xaf, 0xee, 0x72, 0xf4, 0x63, 0x4b, 0x6d, 0x46, 0xe4, 0xce,
0xad, 0x3d, 0xf2, 0x0b, 0x3b, 0x5a
},
.iv = (unsigned char []){
0xc3, 0x76, 0x0e, 0x41, 0x3d, 0x6c, 0x98, 0x62, 0x81, 0xc2, 0x09, 0xb4, 0xaf,
0xdc, 0x67, 0x4a, 0xea, 0x43, 0xc7, 0xfe, 0xe6, 0x86, 0x3f, 0x80, 0xfe, 0x15,
0x16, 0xcd, 0xb2, 0x1e, 0xe5, 0x6d, 0xb7, 0x73, 0x5b, 0x98, 0xba, 0xe8, 0x14,
0xb6, 0xe1, 0x49, 0x4b, 0x57, 0xda, 0xdd, 0xb4, 0x92, 0xef, 0x08, 0xcf, 0xff,
0x73, 0xcf, 0x58, 0x9f, 0x1d, 0xa9, 0x44, 0xa4, 0x21, 0xe2, 0x49, 0xdd, 0x02,
0xec, 0xc5, 0x91, 0x1a, 0x9a, 0xed, 0x11, 0xd7, 0x87, 0x17, 0xb4, 0xe6, 0xcb,
0x56, 0xbf, 0x77, 0x7f, 0x94, 0x3f, 0xd0, 0xc4, 0x8f, 0x4a, 0xa8, 0xff, 0xdf,
0x5a, 0x12, 0x9e, 0x71, 0x6d, 0xbe, 0xa7, 0x14, 0x39, 0xd0, 0x77, 0xfb, 0x69,
0x3d, 0x0e, 0x29, 0x3a, 0xb2, 0xe1, 0xd5, 0xb9, 0x26, 0xa0, 0x1c, 0xef, 0xe3,
0xe6, 0x88, 0x12, 0xc2, 0x80, 0xd5, 0x65, 0xa5, 0xc5, 0x71, 0x02
},
.plaintext = NULL,
.aad = (unsigned char []){
0x16, 0xbc, 0xa6, 0xec, 0xc8, 0x7f, 0x32, 0xb7, 0x78, 0xed, 0xc9, 0x06, 0xab,
0x86, 0x23, 0xef, 0x37, 0xee, 0x16, 0x70, 0x67, 0x26, 0xa9, 0x8a, 0x23, 0x5a,
0xe5, 0x89, 0xd5, 0x1a, 0x1f, 0x56, 0x1d, 0xc6, 0xfd, 0x30, 0x24, 0x48, 0xf7,
0x08, 0x25, 0x6f, 0xcc, 0x62, 0xec, 0xd4, 0x1b, 0xaa, 0x9e, 0xe3, 0x23, 0x56,
0x42, 0xb0, 0xff, 0x12, 0x47, 0x03, 0xd1, 0x8f, 0xb7, 0x48, 0x63, 0xe7, 0x5a,
0x80, 0x55, 0xc2, 0xc9, 0x7b, 0x49, 0xc6, 0xdc, 0xb0, 0xaa, 0xeb, 0xb6, 0x24,
0x00, 0x32, 0x84, 0x2f, 0xf5, 0xd1, 0xb4, 0x29, 0xa0, 0xa5, 0x1d, 0xd7
},
.tag = (unsigned char []){
0xfb, 0xaf, 0xce, 0xd4
},
.ciphertext = (unsigned char []){
0xaa, 0x58, 0x42, 0x8e, 0x79, 0xa7, 0x4a, 0x14, 0x35, 0x9b, 0x89, 0xea, 0x78,
0x62, 0xfe, 0xb1, 0xb7, 0xb7, 0x65, 0x72, 0xe9, 0x57, 0x35, 0x9c, 0x9a, 0x62,
0x85, 0xe2, 0x9a, 0x0f, 0xb4, 0x7a, 0x90, 0xca, 0x7e, 0xe7, 0x1d, 0x41, 0x36,
0x50, 0x23, 0x21, 0xb3, 0x78, 0xe5, 0xa0, 0x4b, 0xb0, 0xc9, 0xeb, 0x5c
},
.rv = EFAULT,
},
{
.keylen = AES256_KEYLEN,
.ivlen = 1024 / 8,
.len = 408 / 8,
.aadlen = 720 / 8,
.taglen = 32 / 8,
.key = (unsigned char []){
0xe1, 0x7c, 0x5c, 0xa6, 0x66, 0x29, 0xba, 0xd6, 0x86, 0xd9, 0x83, 0x51, 0x03,
0xf8, 0xcb, 0x28, 0x43, 0xc0, 0x98, 0x0b, 0xc6, 0xef, 0x29, 0x47, 0xea, 0x5c,
0xce, 0xc3, 0x84, 0xa2, 0x55, 0x99
},
.iv = (unsigned char []){
0x4b, 0x3e, 0xe3, 0x05, 0x78, 0xad, 0x93, 0x88, 0x40, 0x44, 0xdc, 0xff, 0xdf,
0x28, 0x6a, 0x1a, 0xf8, 0xa9, 0xae, 0x27, 0x39, 0xac, 0x13, 0x89, 0x18, 0x89,
0xb0, 0xf9, 0x8a, 0x60, 0xde, 0x71, 0x02, 0xf5, 0x8a, 0xaf, 0xf0, 0xc4, 0x3d,
0x85, 0x0f, 0xf9, 0x79, 0x86, 0x5d, 0xb6, 0xea, 0xb2, 0xd1, 0x7d, 0xb4, 0x1a,
0x12, 0x0e, 0xe5, 0x09, 0xf9, 0xbf, 0x44, 0x66, 0x10, 0xa9, 0x86, 0x35, 0x81,
0xd6, 0xa5, 0x65, 0x83, 0x52, 0x42, 0xf1, 0x11, 0xbe, 0x68, 0x60, 0x33, 0xa7,
0xbb, 0xec, 0x53, 0x28, 0xa8, 0xe3, 0x6b, 0x02, 0xa9, 0x5e, 0xef, 0xf6, 0x18,
0xdc, 0xb2, 0x03, 0x31, 0x9b, 0xc4, 0x61, 0x47, 0xf2, 0x87, 0x03, 0x8b, 0xa3,
0x37, 0x3c, 0xfb, 0xbe, 0xcd, 0x20, 0xbb, 0xaa, 0xc2, 0xdb, 0xf0, 0x0c, 0x1b,
0x26, 0xb6, 0x44, 0x9f, 0xa1, 0xd0, 0x22, 0x1d, 0x5b, 0x35, 0xe6
},
.plaintext = (unsigned char []) {
0x50, 0x46, 0xc7, 0xfe, 0x0d, 0x32, 0x90, 0x52, 0x5b, 0x78, 0x92, 0xbf, 0x19,
0xd6, 0x4c, 0x5d, 0x99, 0x88, 0x1a, 0x3b, 0xfb, 0xb2, 0x1f, 0x00, 0x07, 0xe6,
0x0c, 0xea, 0x99, 0x2d, 0x22, 0x54, 0x9e, 0xce, 0x21, 0x91, 0x21, 0x6b, 0x6c,
0x68, 0x20, 0x28, 0x64, 0x49, 0x2f, 0x1f, 0xd1, 0x5b, 0x89, 0x16, 0xd3
},
.aad = (unsigned char []){
0x51, 0xc8, 0xaa, 0x89, 0x03, 0x3a, 0x74, 0x8f, 0x0a, 0x63, 0xec, 0x08, 0x95,
0xd5, 0xbf, 0x9d, 0x5f, 0xe2, 0x76, 0x69, 0x9f, 0x2a, 0x04, 0xfc, 0xf5, 0x84,
0xbf, 0x1d, 0xfe, 0x0a, 0x26, 0x7b, 0x81, 0x28, 0xff, 0x38, 0x87, 0x3e, 0x45,
0xf4, 0xf8, 0x53, 0x84, 0x7e, 0x45, 0xa9, 0x95, 0x60, 0x18, 0x6e, 0x08, 0x81,
0xf8, 0xe9, 0xd3, 0xce, 0xcb, 0x93, 0xdb, 0x4a, 0x8c, 0xc8, 0x6c, 0x05, 0xff,
0x44, 0x92, 0x41, 0xf1, 0xe5, 0x65, 0xbe, 0x9a, 0x68, 0xb6, 0x18, 0xa7, 0x01,
0x61, 0x46, 0xdb, 0x62, 0x57, 0x7d, 0xe7, 0xd0, 0x5c, 0x29, 0x5b, 0x7b
},
.tag = (unsigned char []){
0xf9, 0x05, 0xb7, 0xf7
},
.ciphertext = (unsigned char []){
0xc4, 0xbc, 0x2e, 0x5d, 0xc4, 0xec, 0x97, 0x66, 0x5d, 0x0c, 0xc5, 0x59, 0x7e,
0x4b, 0x18, 0x8e, 0xff, 0x9d, 0x54, 0x79, 0x87, 0x24, 0xfe, 0xf2, 0x8e, 0x93,
0x8b, 0x61, 0x1c, 0x3e, 0x15, 0xfb, 0x32, 0x23, 0xdd, 0x87, 0x91, 0x34, 0x62,
0x4f, 0x2d, 0x07, 0x1e, 0x31, 0xa8, 0xbd, 0x79, 0x8e, 0x94, 0x9d, 0xb5
},
.rv = 0,
},
{
.keylen = AES256_KEYLEN,
.ivlen = 96 / 8,
.len = 128 / 8,
.aadlen = 128 / 8,
.taglen = 104 / 8,
.key = (unsigned char []){
0x57, 0xc2, 0xf2, 0x48, 0x59, 0x89, 0x2c, 0xfd, 0x8c, 0x80, 0xa9, 0xb9, 0xdb,
0xab, 0x4f, 0x78, 0xb2, 0x2b, 0xdd, 0xed, 0xec, 0x57, 0xe0, 0xbd, 0x30, 0xb8,
0xd9, 0x3d, 0xf4, 0xeb, 0x16, 0x74
},
.iv = (unsigned char []){
0x85, 0x5b, 0x19, 0x8c, 0xc1, 0xd8, 0x57, 0x77, 0x0d, 0xc9, 0xee, 0x93
},
.plaintext = (unsigned char []) {
0xf9, 0x7f, 0x43, 0x2d, 0xb7, 0x61, 0xec, 0xe8, 0xdc, 0x1e, 0x7b, 0xdd, 0x47,
0x6d, 0xc2, 0xef
},
.aad = (unsigned char []){
0xcd, 0x32, 0xfd, 0xa7, 0x6b, 0x12, 0x2c, 0x60, 0x9d, 0x0a, 0xf6, 0x0f, 0x2c,
0x39, 0x2b, 0x8b
},
.tag = (unsigned char []){
0x2b, 0xd9, 0x69, 0xae, 0x43, 0xa2, 0xdb, 0x5b, 0xaf, 0xd7, 0x9e, 0x17, 0x51
},
.ciphertext = (unsigned char []){
0xe0, 0x7b, 0xa0, 0x42, 0x16, 0xd5, 0x00, 0x99, 0x9a, 0x25, 0xad, 0xd7, 0x10,
0x2c, 0x70, 0x36
},
.rv = 0,
},
{
.keylen = AES256_KEYLEN,
.ivlen = 96 / 8,
.len = 128 / 8,
.aadlen = 128 / 8,
.taglen = 104 / 8,
.key = (unsigned char []){
0xe3, 0xc0, 0x0c, 0xe8, 0x1f, 0xfb, 0xca, 0xcc, 0x56, 0x7a, 0x06, 0x34, 0x28,
0x49, 0x53, 0x86, 0xad, 0x51, 0xae, 0x83, 0x13, 0x22, 0x3a, 0xba, 0xbb, 0x2c,
0x59, 0xa2, 0x2d, 0x35, 0x34, 0x28
},
.iv = (unsigned char []){
0x91, 0x96, 0xfe, 0x6f, 0x8d, 0x34, 0x1f, 0x69, 0xf8, 0x86, 0x32, 0xff
},
.plaintext = (unsigned char []) {
0x54, 0x0c, 0x5d, 0x33, 0xd9, 0x43, 0x96, 0xea, 0xd0, 0x9b, 0x84, 0x56, 0xff,
0x12, 0x73, 0x50
},
.aad = (unsigned char []){
0x88, 0x0f, 0x98, 0xc2, 0xa0, 0xfb, 0xed, 0x6b, 0xf3, 0xbf, 0xf7, 0x1d, 0xea,
0x42, 0x7f, 0x5f
},
.tag = (unsigned char []){
0x08, 0xfe, 0xad, 0xdb, 0x09, 0x05, 0x9a, 0xce, 0xa8, 0x67, 0xa8, 0xcd, 0xe2
},
.ciphertext = (unsigned char []){
0xc2, 0x26, 0x0f, 0xc8, 0x37, 0x9d, 0x0f, 0x11, 0xbd, 0x06, 0xe5, 0xd7, 0x44,
0x77, 0xd3, 0xe7
},
.rv = 0,
},
{
.keylen = AES192_KEYLEN,
.ivlen = 1024 / 8,
.len = 408 / 8,
.aadlen = 128 / 8,
.taglen = 96 / 8,
.key = (unsigned char []){
0x37, 0xb3, 0x56, 0x2a, 0x59, 0x12, 0xcc, 0xf6, 0xcb, 0xcc, 0xdd, 0x3f, 0xcf,
0x9d, 0x76, 0x95, 0xfb, 0x8a, 0xf1, 0x08, 0x52, 0xb2, 0x1f, 0x1d
},
.iv = (unsigned char []){
0x2f, 0x59, 0x77, 0xf7, 0x42, 0x65, 0xa0, 0x7e, 0xd8, 0x47, 0xad, 0x27, 0xb5,
0xcb, 0x85, 0xba, 0x25, 0x66, 0xe4, 0x63, 0xb9, 0xc6, 0x15, 0xf8, 0xd5, 0xb1,
0x83, 0xd7, 0xb5, 0x39, 0x35, 0xe2, 0xb5, 0x7f, 0xc7, 0x5e, 0x36, 0x88, 0x10,
0x2d, 0xa0, 0x46, 0x44, 0x29, 0x3c, 0x6c, 0x86, 0xb6, 0xc2, 0x42, 0xc6, 0xcd,
0xaf, 0xb2, 0x91, 0x99, 0xb8, 0x0c, 0x4f, 0x46, 0x53, 0xa1, 0x1b, 0xbb, 0x0a,
0xc3, 0x1e, 0xc1, 0x63, 0x0e, 0x03, 0x2d, 0x39, 0xec, 0x02, 0x83, 0xce, 0x6e,
0xc0, 0x99, 0xb3, 0x32, 0xae, 0x4e, 0x0a, 0xc1, 0x97, 0x67, 0xa1, 0x05, 0x0c,
0x48, 0xef, 0xab, 0xa1, 0xc1, 0xb2, 0xaa, 0xcb, 0xa9, 0x57, 0x0e, 0xcf, 0xbb,
0xed, 0xe4, 0xd2, 0x32, 0x73, 0xd3, 0xd5, 0x18, 0xef, 0x68, 0x59, 0x26, 0xc1,
0x70, 0xe1, 0xed, 0xf0, 0x46, 0x74, 0xc6, 0x22, 0xfe, 0x98, 0x4d
},
.plaintext = (unsigned char []) {
0xd0, 0xd8, 0x95, 0xb6, 0xb8, 0x0f, 0x97, 0xe4, 0xa6, 0x09, 0x55, 0x60, 0x94,
0x2d, 0xdb, 0xa4, 0xd7, 0x3c, 0xda, 0xf0, 0x9f, 0xf6, 0xa4, 0x38, 0x1b, 0x5b,
0x03, 0xf4, 0xbd, 0x75, 0x16, 0x49, 0x42, 0x68, 0x66, 0x89, 0xf3, 0xac, 0xc6,
0x2b, 0x02, 0x43, 0x1a, 0x9a, 0xd4, 0x03, 0x5e, 0x3f, 0x8e, 0x20, 0xc6
},
.aad = (unsigned char []){
0xd0, 0xed, 0xbf, 0x82, 0x83, 0x2c, 0x6f, 0x35, 0x8d, 0xd5, 0x00, 0xb9, 0x79,
0xc1, 0x5f, 0x47
},
.tag = (unsigned char []){
0xa0, 0xea, 0x4b, 0xbd, 0xc4, 0x9a, 0x10, 0x29, 0x8c, 0x9a, 0x97, 0x3d
},
.ciphertext = (unsigned char []){
0xd3, 0x29, 0xc6, 0x6a, 0x4f, 0x46, 0x0a, 0xe6, 0xf0, 0x5b, 0x0d, 0xfd, 0x2e,
0xb4, 0x47, 0x6c, 0x0f, 0x29, 0x3c, 0x08, 0xce, 0xc5, 0x7e, 0x6d, 0x6a, 0xda,
0xe9, 0x61, 0x14, 0x45, 0xb8, 0x9f, 0x32, 0xb2, 0x3f, 0xdc, 0x5c, 0x1d, 0xd3,
0x3f, 0xa6, 0x2d, 0x19, 0x0f, 0x4b, 0x5b, 0xd8, 0x1f, 0x0e, 0x97, 0xa8
},
.rv = 0,
},
{
.keylen = AES128_KEYLEN,
.ivlen = 96 / 8,
.len = 408 / 8,
.aadlen = 720 / 8,
.taglen = 120 / 8,
.key = (unsigned char []){
0x32, 0x66, 0x43, 0x0c, 0x1c, 0xad, 0x19, 0x32, 0x10, 0x13, 0xce, 0xec, 0x77,
0xdc, 0x54, 0x09
},
.iv = (unsigned char []){
0x44, 0xe3, 0xc6, 0xf6, 0x19, 0xa3, 0x73, 0xc7, 0x97, 0xda, 0x3e, 0x62
},
.plaintext = (unsigned char []) {
0xe7, 0x3c, 0x85, 0x98, 0x26, 0x8b, 0x79, 0xf4, 0x0a, 0x34, 0x14, 0x75, 0xf0,
0x3f, 0xe8, 0x08, 0x56, 0xa6, 0xd9, 0x36, 0xf7, 0xa0, 0x57, 0xa9, 0xec, 0x5b,
0xeb, 0x41, 0x85, 0x34, 0x01, 0x35, 0x83, 0xf3, 0xd9, 0x08, 0x23, 0x59, 0xec,
0xb4, 0x0e, 0x95, 0x76, 0xec, 0x58, 0xcb, 0x5b, 0xfe, 0x28, 0x63, 0xb5
},
.aad = (unsigned char []){
0x45, 0xee, 0x7d, 0x52, 0xf7, 0xd8, 0x0a, 0x6c, 0x6b, 0x5f, 0xb0, 0xf2, 0x67,
0xd3, 0x06, 0x79, 0xc6, 0xa4, 0xbe, 0x41, 0xe4, 0xc7, 0xd2, 0xcd, 0x43, 0x1c,
0x8d, 0xe7, 0x57, 0x70, 0x10, 0x91, 0xa3, 0x77, 0x6e, 0x40, 0xd5, 0xc9, 0xec,
0xd7, 0xc5, 0x0c, 0xc8, 0x50, 0xef, 0x3a, 0x3f, 0xeb, 0xe4, 0x30, 0xaf, 0x0c,
0xb2, 0xdd, 0xc5, 0x03, 0x3f, 0x5d, 0x68, 0x72, 0x8f, 0xf4, 0xe3, 0x61, 0x37,
0x4c, 0x88, 0xdc, 0x7f, 0x33, 0x38, 0x60, 0x4a, 0xc3, 0x62, 0x8a, 0xe1, 0x2a,
0x13, 0x2b, 0x5f, 0x42, 0x13, 0x13, 0x5c, 0x81, 0x14, 0x0e, 0xc5, 0xbe
},
.tag = (unsigned char []){
0x85, 0x43, 0xd4, 0xe7, 0x20, 0x50, 0xdc, 0x37, 0xc5, 0x85, 0xef, 0xbd, 0xa5,
0xdf, 0x95
},
.ciphertext = (unsigned char []){
0x6d, 0x1f, 0x6b, 0xc3, 0xd7, 0xba, 0x16, 0xef, 0x28, 0x98, 0x91, 0xee, 0x89,
0x12, 0xae, 0x54, 0xdb, 0xf2, 0xd2, 0xcd, 0x84, 0xff, 0xb8, 0x1b, 0xe5, 0xb9,
0x9c, 0x5a, 0x06, 0x9b, 0xfa, 0xf4, 0xea, 0x50, 0xb3, 0x54, 0xdf, 0x24, 0x35,
0x65, 0x46, 0x2a, 0xba, 0x25, 0xe5, 0x20, 0xa4, 0xdc, 0x71, 0x1b, 0xdb
},
.rv = 0,
},
{
.keylen = AES192_KEYLEN,
.ivlen = 8 / 8,
.len = 104 / 8,
.aadlen = 160 / 8,
.taglen = 64 / 8,
.key = (unsigned char []){
0x27, 0xd2, 0x0c, 0x93, 0xe1, 0x2c, 0xa2, 0x1a, 0x59, 0x09, 0x54, 0x25, 0xfc,
0xe8, 0x46, 0xad, 0x45, 0x71, 0x62, 0x8e, 0xce, 0x9f, 0x6b, 0x91
},
.iv = (unsigned char []){
0x43
},
.plaintext = (unsigned char []) {
0xbb, 0x0f, 0x56, 0x3e, 0x30, 0x33, 0x7f, 0x0c, 0x38, 0x47, 0xdc, 0xe3, 0xd2
},
.aad = (unsigned char []){
0xcc, 0xca, 0xe3, 0xab, 0x11, 0x6b, 0x5a, 0x3d, 0xfd, 0x8c, 0x8b, 0xf7, 0xd5,
0xe7, 0xb6, 0xa0, 0xa4, 0xcd, 0xb4, 0x0a
},
.tag = (unsigned char []){
0x1a, 0xa6, 0xb6, 0x6a, 0x18, 0xd5, 0x98, 0xe6
},
.ciphertext = (unsigned char []){
0x2f, 0x83, 0xe6, 0xaa, 0xf6, 0xfb, 0xcd, 0x07, 0x28, 0xcb, 0x99, 0x2a, 0x0d
},
.rv = 0,
},
{
.keylen = AES128_KEYLEN,
.ivlen = 1024 / 8,
.len =  408 / 8,
.aadlen = 128 / 8,
.taglen = 112 / 8,
.key = (unsigned char []){
0x49, 0xa4, 0xf6, 0x0c, 0xb3, 0xf4, 0x3f, 0xf3, 0x77, 0xc3, 0x43, 0x64, 0x24,
0x79, 0xd6, 0x4b
},
.iv = (unsigned char []){
0xcb, 0x38, 0x64, 0xdf, 0x26, 0xb1, 0x66, 0xda, 0x4d, 0x1a, 0x32, 0x5d, 0x91,
0xb1, 0xc8, 0x34, 0xf7, 0xe7, 0x8f, 0xf5, 0x44, 0x85, 0x68, 0xdf, 0xdc, 0x96,
0x6d, 0xab, 0x49, 0xe9, 0xf7, 0x73, 0x97, 0xbf, 0xb4, 0xc8, 0x3c, 0xef, 0xaa,
0x9c, 0xaa, 0x64, 0xbe, 0x4d, 0xce, 0x61, 0x0b, 0x26, 0xca, 0xe6, 0xe0, 0x3d,
0x6c, 0xdc, 0xe0, 0x34, 0x5b, 0x6b, 0xd4, 0xcc, 0xa2, 0x27, 0x93, 0xb4, 0x89,
0x4b, 0xe9, 0x50, 0x02, 0x4a, 0xf4, 0x22, 0xa4, 0x85, 0xdb, 0xb0, 0x99, 0xe5,
0xbb, 0x3f, 0x79, 0x2d, 0x8a, 0xed, 0x9c, 0x87, 0xba, 0x9f, 0x0e, 0xa3, 0x75,
0x00, 0x62, 0xcf, 0x9a, 0x8d, 0x7b, 0x88, 0x4e, 0x1f, 0x8e, 0x88, 0x09, 0x74,
0x34, 0x3d, 0x7f, 0x7d, 0x03, 0x26, 0x74, 0x84, 0x86, 0x15, 0x72, 0x59, 0x0b,
0x2f, 0x96, 0x90, 0x6f, 0x0e, 0x76, 0x34, 0xa6, 0x47, 0xcb, 0x79
},
.plaintext = (unsigned char []) {
0xee, 0xda, 0x6c, 0x7c, 0x50, 0x05, 0x61, 0xe6, 0xad, 0xcf, 0x94, 0xb8, 0x45,
0x51, 0xd9, 0xbb, 0xca, 0x36, 0xd7, 0x0a, 0x54, 0x94, 0x9f, 0x3f, 0x96, 0xc9,
0xb3, 0x93, 0x1d, 0xe6, 0x04, 0x07, 0x13, 0x0d, 0xfa, 0xb1, 0x02, 0xcc, 0x26,
0x11, 0x7e, 0x9e, 0xad, 0x80, 0x0c, 0x55, 0x09, 0x18, 0x4f, 0xc9, 0xcf
},
.aad = (unsigned char []){
0xb7, 0x47, 0x07, 0x07, 0xbd, 0x7a, 0xe9, 0xdc, 0x18, 0x1b, 0xb9, 0xd8, 0xaf,
0xb9, 0x59, 0x60
},
.tag = (unsigned char []){
0x4d, 0xbe, 0x3c, 0xb5, 0x45, 0x58, 0x31, 0x45, 0xe0, 0x89, 0x66, 0xe9, 0x10,
0x90
},
.ciphertext = (unsigned char []){
0x08, 0xc1, 0xe9, 0x19, 0x6a, 0x22, 0xed, 0x47, 0x90, 0xa6, 0x59, 0x5c, 0x3f,
0xeb, 0x95, 0xe7, 0xd4, 0x8c, 0xb3, 0x5c, 0x0c, 0x1e, 0x57, 0x95, 0x14, 0x83,
0x7b, 0xec, 0xca, 0xaf, 0xf5, 0x52, 0x9d, 0x9f, 0x80, 0x5f, 0xe5, 0x72, 0x25,
0xf9, 0x81, 0x88, 0x90, 0xd9, 0xfb, 0xc9, 0x82, 0xf9, 0x4d, 0x26, 0x48
},
.rv = 0,
},
{
.keylen = AES256_KEYLEN,
.ivlen = 1024 / 8,
.len = 408 / 8,
.aadlen = 720 / 8,
.taglen = 128 / 8,
.key = (unsigned char []){
0x87, 0x1a, 0xd3, 0x7e, 0xeb, 0xc6, 0x84, 0x79, 0x44, 0xf7, 0xe4, 0x2f, 0xff,
0x5a, 0x65, 0xae, 0x62, 0xc5, 0x08, 0x54, 0x48, 0x6f, 0x63, 0xc0, 0x8f, 0x60,
0x80, 0xf8, 0x3c, 0x08, 0xd6, 0x6a
},
.iv = (unsigned char []){
0xf9, 0x1b, 0x2d, 0x31, 0x61, 0x7e, 0x9a, 0x52, 0x04, 0x20, 0x44, 0xf5, 0x70,
0x37, 0xb0, 0x96, 0x47, 0x7d, 0x90, 0xe6, 0xe9, 0xd6, 0xf4, 0x0f, 0x67, 0x62,
0xea, 0x48, 0xfa, 0xc5, 0xa1, 0x82, 0x50, 0x2e, 0xf9, 0x40, 0xd3, 0xd4, 0xc8,
0x6a, 0x48, 0xf7, 0xe7, 0xc2, 0xc8, 0xd6, 0xba, 0x55, 0xdd, 0xb3, 0x27, 0x69,
0xdc, 0xc3, 0x10, 0x10, 0xc0, 0x44, 0xa8, 0x0f, 0xa4, 0x4f, 0x20, 0x1c, 0xd8,
0x72, 0x62, 0x70, 0xd6, 0xf7, 0x0e, 0xd0, 0x13, 0x78, 0xab, 0xe1, 0xd9, 0x34,
0xd4, 0x83, 0x53, 0x6b, 0xa6, 0x08, 0xe7, 0x23, 0x81, 0xfd, 0x15, 0x23, 0x4a,
0xd1, 0x4c, 0x5c, 0x2c, 0xf0, 0x83, 0xc0, 0xd0, 0x70, 0x22, 0x83, 0x42, 0x96,
0x8f, 0xf5, 0xf6, 0x6d, 0x6f, 0x23, 0x95, 0x1c, 0x54, 0xc7, 0xf7, 0x9c, 0x82,
0xe2, 0x36, 0xe5, 0x51, 0x7f, 0x26, 0x4e, 0xaa, 0xc0, 0x93, 0xc5
},
.plaintext = (unsigned char []) {
0x90, 0x89, 0x42, 0x5f, 0x0d, 0x9d, 0x17, 0x51, 0x64, 0x65, 0x99, 0x00, 0x49,
0xf0, 0x1a, 0x2e, 0x8e, 0x5f, 0x91, 0xc2, 0xfa, 0xea, 0x84, 0x68, 0x97, 0x36,
0x06, 0xc1, 0x2b, 0x50, 0x7c, 0xa0, 0x72, 0x81, 0x8b, 0x4f, 0xe2, 0xd6, 0x11,
0x70, 0x96, 0x38, 0xd2, 0x1e, 0x03, 0x4d, 0xbc, 0xdf, 0x47, 0xcf, 0x59
},
.aad = (unsigned char []){
0xd5, 0xf7, 0xc1, 0xf6, 0x89, 0x65, 0xd9, 0x13, 0xab, 0xc6, 0xbb, 0x01, 0xb3,
0x3d, 0x35, 0xa3, 0x17, 0xa7, 0xfb, 0xe7, 0xba, 0x2e, 0x53, 0xc7, 0x3d, 0x6c,
0x44, 0xab, 0xe1, 0xc2, 0x16, 0x0d, 0x54, 0x5d, 0x26, 0x24, 0xf2, 0x54, 0x50,
0xad, 0x75, 0x13, 0xb2, 0xd3, 0x2f, 0xfa, 0x85, 0x78, 0xc3, 0x0e, 0x96, 0xd7,
0xba, 0x49, 0xb3, 0x63, 0x4d, 0x99, 0xa1, 0x5e, 0xbf, 0xb3, 0x7d, 0xec, 0xd0,
0xf8, 0xef, 0xc3, 0x2a, 0xab, 0x40, 0xa0, 0x59, 0x4b, 0xeb, 0xa0, 0xda, 0xc6,
0xf8, 0xda, 0xa3, 0xac, 0xe9, 0x1b, 0xd0, 0x95, 0x49, 0xa1, 0xf5, 0xe1
},
.tag = (unsigned char []){
0xcc, 0x4d, 0x95, 0x0b, 0x8c, 0x33, 0x30, 0xf0, 0x48, 0xe3, 0xaa, 0x3e, 0x3c,
0xfb, 0xce, 0x21
},
.ciphertext = (unsigned char []){
0x02, 0x10, 0x1f, 0x84, 0x82, 0x05, 0x2c, 0xb7, 0x70, 0x68, 0x00, 0xe8, 0x31,
0x0f, 0x5d, 0x44, 0x4d, 0x67, 0x2b, 0x65, 0x59, 0xde, 0x0b, 0x87, 0x92, 0xd7,
0xc6, 0x4c, 0xb1, 0xbc, 0x58, 0x7d, 0xed, 0xd6, 0x31, 0x59, 0x2f, 0xf6, 0xe9,
0xc8, 0xfa, 0x28, 0x89, 0x5d, 0x36, 0x85, 0x9b, 0x97, 0x3a, 0x72, 0x24
},
.rv = 0,
},
};

const struct aes_ccm_tv AES_CCM_TV[] = {
{
.keylen = AES128_KEYLEN,
.noncelen = 13,
.adatalen = 32,
.payloadlen = 24,
.taglen = 16,
.key = (unsigned char []){
0x26, 0x51, 0x1f, 0xb5, 0x1f, 0xcf, 0xa7, 0x5c, 0xb4, 0xb4, 0x4d, 0xa7, 0x5a,
0x6e, 0x5a, 0x0e
},
.nonce = (unsigned char []){
0xea, 0x98, 0xec, 0x44, 0xf5, 0xa8, 0x67, 0x15, 0x01, 0x47, 0x83, 0x17, 0x2e
},
.adata = (unsigned char []){
0xe4, 0x69, 0x2b, 0x9f, 0x06, 0xb6, 0x66, 0xc7, 0x45, 0x1b, 0x14, 0x6c, 0x8a,
0xeb, 0x07, 0xa6, 0xe3, 0x0c, 0x62, 0x9d, 0x28, 0x06, 0x5c, 0x3d, 0xde, 0x59,
0x40, 0x32, 0x5b, 0x14, 0xb8, 0x10
},
.payload = (unsigned char []){
0x4d, 0xa4, 0x0b, 0x80, 0x57, 0x9c, 0x1d, 0x9a, 0x53, 0x09, 0xf7, 0xef, 0xec,
0xb7, 0xc0, 0x59, 0xa2, 0xf9, 0x14, 0x51, 0x1c, 0xa5, 0xfc, 0x10
},
.ciphertext = (unsigned char []){
0x1b, 0xf0, 0xba, 0x0e, 0xbb, 0x20, 0xd8, 0xed, 0xba, 0x59, 0xf2, 0x9a, 0x93,
0x71, 0x75, 0x0c, 0x9c, 0x71, 0x40, 0x78, 0xf7, 0x3c, 0x33, 0x5d, 0x2f, 0x13,
0x22, 0xac, 0x69, 0xb8, 0x48, 0xb0, 0x01, 0x47, 0x63, 0x23, 0xae, 0xd8, 0x4c,
0x47
},
.rv = 0,
},
{
.keylen = AES128_KEYLEN,
.noncelen = 13,
.adatalen = 32,
.payloadlen = 24,
.taglen = 16,
.key = (unsigned char []){
0x26, 0x51, 0x1f, 0xb5, 0x1f, 0xcf, 0xa7, 0x5c, 0xb4, 0xb4, 0x4d, 0xa7, 0x5a,
0x6e, 0x5a, 0x0e
},
.nonce = (unsigned char []){
0x5a, 0x16, 0xa8, 0x90, 0x2b, 0xd7, 0x0f, 0xa0, 0x6c, 0xfe, 0x18, 0x4c, 0x57
},
.adata = (unsigned char []){
0x39, 0x9d, 0x6b, 0x06, 0x52, 0x83, 0x64, 0x57, 0xec, 0x4f, 0x70, 0x1f, 0x0d,
0xc0, 0xe5, 0xae, 0xd7, 0x3d, 0x16, 0x58, 0x5d, 0x61, 0xcb, 0x1b, 0xb5, 0xb7,
0xee, 0x82, 0x4f, 0xc2, 0x87, 0xc8
},
.payload = NULL,
.ciphertext = (unsigned char []){
0x9d, 0x99, 0x3b, 0x94, 0x54, 0x76, 0xac, 0xe0, 0xb9, 0xca, 0x93, 0x29, 0x63,
0xac, 0x88, 0x35, 0xe1, 0xbd, 0x02, 0xe8, 0x06, 0x5d, 0xa2, 0xd8, 0x16, 0x78,
0x6c, 0x4d, 0x8c, 0xf1, 0x4c, 0x03, 0xb0, 0x31, 0xff, 0x72, 0x33, 0x11, 0xb3,
0xc4
},
.rv = EFAULT,
},
{
.keylen = AES192_KEYLEN,
.noncelen = 13,
.adatalen = 32,
.payloadlen = 24,
.taglen = 16,
.key = (unsigned char []){
0x26, 0x51, 0x1f, 0xb5, 0x1f, 0xcf, 0xa7, 0x5c, 0xb4, 0xb4, 0x4d, 0xa7, 0x5a,
0x6e, 0x5a, 0x0e, 0xb8, 0xd9, 0xc8, 0xf3, 0xb9, 0x06, 0xf8, 0x86
},
.nonce = (unsigned char []){
0xea, 0x98, 0xec, 0x44, 0xf5, 0xa8, 0x67, 0x15, 0x01, 0x47, 0x83, 0x17, 0x2e
},
.adata = (unsigned char []){
0xe4, 0x69, 0x2b, 0x9f, 0x06, 0xb6, 0x66, 0xc7, 0x45, 0x1b, 0x14, 0x6c, 0x8a,
0xeb, 0x07, 0xa6, 0xe3, 0x0c, 0x62, 0x9d, 0x28, 0x06, 0x5c, 0x3d, 0xde, 0x59,
0x40, 0x32, 0x5b, 0x14, 0xb8, 0x10
},
.payload = (unsigned char []){
0x4d, 0xa4, 0x0b, 0x80, 0x57, 0x9c, 0x1d, 0x9a, 0x53, 0x09, 0xf7, 0xef, 0xec,
0xb7, 0xc0, 0x59, 0xa2, 0xf9, 0x14, 0x51, 0x1c, 0xa5, 0xfc, 0x10
},
.ciphertext = (unsigned char []){
0x30, 0xc1, 0x54, 0xc6, 0x16, 0x94, 0x6e, 0xcc, 0xc2, 0xe2, 0x41, 0xd3, 0x36,
0xad, 0x33, 0x72, 0x09, 0x53, 0xe4, 0x49, 0xa0, 0xe6, 0xb0, 0xf0, 0xdb, 0xf8,
0xe9, 0x46, 0x49, 0x09, 0xbd, 0xf3, 0x37, 0xe4, 0x80, 0x93, 0xc0, 0x82, 0xa1,
0x0b
},
.rv = 0,
},
{
.keylen = AES192_KEYLEN,
.noncelen = 13,
.adatalen = 32,
.payloadlen = 24,
.taglen = 16,
.key = (unsigned char []){
0x26, 0x51, 0x1f, 0xb5, 0x1f, 0xcf, 0xa7, 0x5c, 0xb4, 0xb4, 0x4d, 0xa7, 0x5a,
0x6e, 0x5a, 0x0e, 0xb8, 0xd9, 0xc8, 0xf3, 0xb9, 0x06, 0xf8, 0x86
},
.nonce = (unsigned char []){
0x5a, 0x16, 0xa8, 0x90, 0x2b, 0xd7, 0x0f, 0xa0, 0x6c, 0xfe, 0x18, 0x4c, 0x57
},
.adata = (unsigned char []){
0x39, 0x9d, 0x6b, 0x06, 0x52, 0x83, 0x64, 0x57, 0xec, 0x4f, 0x70, 0x1f, 0x0d,
0xc0, 0xe5, 0xae, 0xd7, 0x3d, 0x16, 0x58, 0x5d, 0x61, 0xcb, 0x1b, 0xb5, 0xb7,
0xee, 0x82, 0x4f, 0xc2, 0x87, 0xc8
},
.payload = NULL,
.ciphertext = (unsigned char []){
0x0c, 0x95, 0xb6, 0x92, 0xb0, 0x7b, 0x39, 0x03, 0x9b, 0x40, 0xc8, 0x0c, 0xf5,
0x2f, 0xf7, 0x16, 0x08, 0xae, 0x87, 0xc9, 0x73, 0xac, 0x9c, 0xcb, 0x88, 0xbb,
0xa8, 0xf2, 0x04, 0xbb, 0x98, 0xb1, 0x7c, 0xb3, 0xc8, 0x64, 0x4e, 0x47, 0x2b,
0x1e
},
.rv = EFAULT,
},
{
.keylen = AES256_KEYLEN,
.noncelen = 13,
.adatalen = 32,
.payloadlen = 24,
.taglen = 16,
.key = (unsigned char []){
0x31, 0x4a, 0x20, 0x2f, 0x83, 0x6f, 0x9f, 0x25, 0x7e, 0x22, 0xd8, 0xc1, 0x17,
0x57, 0x83, 0x2a, 0xe5, 0x13, 0x1d, 0x35, 0x7a, 0x72, 0xdf, 0x88, 0xf3, 0xef,
0xf0, 0xff, 0xce, 0xe0, 0xda, 0x4e
},
.nonce = (unsigned char []){
0x35, 0x42, 0xfb, 0xe0, 0xf5, 0x9a, 0x6d, 0x5f, 0x3a, 0xbf, 0x61, 0x9b, 0x7d
},
.adata = (unsigned char []){
0xdd, 0x45, 0x31, 0xf1, 0x58, 0xa2, 0xfa, 0x3b, 0xc8, 0xa3, 0x39, 0xf7, 0x70,
0x59, 0x50, 0x48, 0xf4, 0xa4, 0x2b, 0xc1, 0xb0, 0x3f, 0x2e, 0x82, 0x4e, 0xfc,
0x6b, 0xa4, 0x98, 0x51, 0x19, 0xd8
},
.payload = (unsigned char []){
0xc5, 0xb3, 0xd7, 0x13, 0x12, 0xea, 0x14, 0xf2, 0xf8, 0xfa, 0xe5, 0xbd, 0x1a,
0x45, 0x31, 0x92, 0xb6, 0x60, 0x4a, 0x45, 0xdb, 0x75, 0xc5, 0xed
},
.ciphertext = (unsigned char []){
0x39, 0xc2, 0xe8, 0xf6, 0xed, 0xfe, 0x66, 0x3b, 0x90, 0x96, 0x3b, 0x98, 0xeb,
0x79, 0xe2, 0xd4, 0xf7, 0xf2, 0x8a, 0x50, 0x53, 0xae, 0x88, 0x81, 0x56, 0x7a,
0x6b, 0x44, 0x26, 0xf1, 0x66, 0x71, 0x36, 0xbe, 0xd4, 0xa5, 0xe3, 0x2a, 0x2b,
0xc1
},
.rv = 0,
},
{
.keylen = AES256_KEYLEN,
.noncelen = 13,
.adatalen = 32,
.payloadlen = 24,
.taglen = 16,
.key = (unsigned char []){
0x31, 0x4a, 0x20, 0x2f, 0x83, 0x6f, 0x9f, 0x25, 0x7e, 0x22, 0xd8, 0xc1, 0x17,
0x57, 0x83, 0x2a, 0xe5, 0x13, 0x1d, 0x35, 0x7a, 0x72, 0xdf, 0x88, 0xf3, 0xef,
0xf0, 0xff, 0xce, 0xe0, 0xda, 0x4e
},
.nonce = (unsigned char []){
0x48, 0xf2, 0xd4, 0xc0, 0xb1, 0x70, 0x72, 0xe0, 0xa9, 0xc3, 0x00, 0xd9, 0x0b
},
.adata = (unsigned char []){
0xc5, 0x61, 0x75, 0xe2, 0xcf, 0xe0, 0xd3, 0x74, 0x54, 0xd9, 0x89, 0xaf, 0xcc,
0x36, 0x68, 0x6f, 0xb3, 0x4c, 0x01, 0x54, 0x39, 0x60, 0x15, 0x67, 0x50, 0x6a,
0x4d, 0x00, 0x03, 0x18, 0x2b, 0xe7
},
.payload = NULL,
.ciphertext = (unsigned char []){
0x27, 0xc5, 0x75, 0xbe, 0x0b, 0x99, 0xaf, 0x9b, 0x10, 0x6f, 0x53, 0xf4, 0x71,
0xc3, 0x1c, 0xac, 0x4d, 0x54, 0xea, 0x0b, 0xcb, 0x60, 0x2a, 0x33, 0xfb, 0x67,
0xbb, 0x60, 0x92, 0xcd, 0x57, 0x9f, 0x72, 0x2a, 0xe9, 0xb6, 0x80, 0xda, 0x08,
0x3d
},
.rv = EFAULT,
},
};

const struct aes_ecb_tv AES_ECB_TV[] = {
{
.keylen = AES128_KEYLEN,
.len = 160,
.key = (unsigned char []){
0x44, 0xf0, 0xee, 0x62, 0x6d, 0x04, 0x46, 0xe0, 0xa3, 0x92, 0x4c, 0xfb, 0x07,
0x89, 0x44, 0xbb
},
.plaintext = (unsigned char []){
0x9c, 0x29, 0xee, 0xcb, 0x2d, 0xe0, 0x42, 0x54, 0xfa, 0xfb, 0x89, 0x6a, 0x99,
0x41, 0x02, 0xd1, 0xda, 0x30, 0xdd, 0xb4, 0x9d, 0x82, 0x72, 0x8e, 0xb2, 0x3d,
0xbd, 0x02, 0x99, 0x01, 0xe9, 0xb7, 0x5b, 0x3d, 0x0a, 0xee, 0x03, 0xf7, 0xa0,
0x5f, 0x6c, 0x85, 0x2d, 0x8f, 0xad, 0xa0, 0xb5, 0xc2, 0x8e, 0x8c, 0x9a, 0xed,
0x33, 0x4f, 0xad, 0x11, 0x82, 0x9d, 0xf3, 0xdf, 0xad, 0xc5, 0xc2, 0xe4, 0x71,
0xeb, 0x41, 0xaf, 0x9e, 0x48, 0xa8, 0xa4, 0x65, 0xe0, 0x3d, 0x5e, 0xbd, 0xb0,
0x21, 0x69, 0x15, 0x08, 0x1f, 0x3b, 0x5a, 0x0e, 0xbb, 0x23, 0x08, 0xdf, 0xc2,
0xd2, 0x8e, 0x5a, 0x8b, 0xa3, 0xf3, 0x2a, 0xda, 0xe4, 0xc3, 0x57, 0x59, 0x21,
0xbc, 0x65, 0x7b, 0x63, 0xd4, 0x6b, 0xa5, 0xa6, 0x18, 0x88, 0x0e, 0xe9, 0xad,
0x8a, 0xf3, 0xfb, 0xa5, 0x64, 0x3a, 0x50, 0x26, 0xfa, 0xcd, 0x7d, 0x66, 0x7c,
0xe5, 0x99, 0x32, 0x7f, 0x93, 0x6c, 0xdd, 0xa7, 0xe1, 0xbb, 0x74, 0x2a, 0x33,
0xa0, 0x19, 0x99, 0x0b, 0x76, 0xbe, 0x64, 0x8a, 0x6e, 0xc7, 0x25, 0xda, 0xed,
0x54, 0x0e, 0xd9, 0xe7
},
.ciphertext = (unsigned char []){
0x93, 0x1b, 0x2f, 0x5f, 0x3a, 0x58, 0x20, 0xd5, 0x3a, 0x6b, 0xea, 0xaa, 0x64,
0x31, 0x08, 0x3a, 0x34, 0x88, 0xf4, 0xeb, 0x03, 0xb0, 0xf5, 0xb5, 0x7e, 0xf8,
0x38, 0xe1, 0x57, 0x96, 0x23, 0x10, 0x3b, 0xd6, 0xe6, 0x80, 0x03, 0x77, 0x53,
0x8b, 0x2e, 0x51, 0xef, 0x70, 0x8f, 0x3c, 0x49, 0x56, 0x43, 0x2e, 0x8a, 0x8e,
0xe6, 0xa3, 0x4e, 0x19, 0x06, 0x42, 0xb2, 0x6a, 0xd8, 0xbd, 0xae, 0x6c, 0x2a,
0xf9, 0xa6, 0xc7, 0x99, 0x6f, 0x3b, 0x60, 0x04, 0xd2, 0x67, 0x1e, 0x41, 0xf1,
0xc9, 0xf4, 0x0e, 0xe0, 0x3d, 0x1c, 0x4a, 0x52, 0xb0, 0xa0, 0x65, 0x4a, 0x33,
0x1f, 0x15, 0xf3, 0x4d, 0xce, 0x4a, 0xcb, 0x96, 0xbd, 0x65, 0x07, 0x81, 0x5c,
0xa4, 0x34, 0x7a, 0x3d, 0xe1, 0x1a, 0x31, 0x1b, 0x7d, 0xe5, 0x35, 0x1c, 0x97,
0x87, 0xc4, 0x53, 0x81, 0x58, 0xe2, 0x89, 0x74, 0xff, 0xa8, 0x3d, 0x82, 0x96,
0xdf, 0xe9, 0xcd, 0x09, 0xcd, 0x87, 0xf7, 0xbf, 0x4f, 0x54, 0xd9, 0x7d, 0x28,
0xd4, 0x78, 0x87, 0x99, 0x16, 0x34, 0x08, 0x32, 0x39, 0x43, 0xb3, 0xe7, 0x2f,
0x5e, 0xab, 0x66, 0xc1
},
},
{
.keylen = AES192_KEYLEN,
.len = 160,
.key = (unsigned char []){
0x9c, 0xc2, 0x4e, 0xa1, 0xf1, 0x95, 0x9d, 0x9a, 0x97, 0x2e, 0x71, 0x82, 0xef,
0x3b, 0x4e, 0x22, 0xa9, 0x7a, 0x87, 0xd0, 0xda, 0x7f, 0xf6, 0x4b
},
.plaintext = (unsigned char []){
0xb8, 0xbb, 0x5c, 0xe5, 0x3a, 0x15, 0xaa, 0x6d, 0xfd, 0xf2, 0xcb, 0x61, 0xbc,
0x8e, 0x36, 0x17, 0xd1, 0xd0, 0xfe, 0xfe, 0x9b, 0xa5, 0xd1, 0x75, 0x55, 0x04,
0x70, 0xe3, 0x23, 0x97, 0xf6, 0xf3, 0xb3, 0xe6, 0x5b, 0x43, 0xbd, 0xed, 0x2b,
0x21, 0xe5, 0xc1, 0x81, 0xd3, 0xc4, 0xc4, 0xc5, 0x26, 0xc4, 0x1c, 0xea, 0xb0,
0x44, 0x28, 0x95, 0x08, 0x45, 0x80, 0x48, 0xb6, 0x33, 0x52, 0xdf, 0xc3, 0x79,
0xde, 0x37, 0x3f, 0xd1, 0x9a, 0x2c, 0x90, 0x0c, 0x43, 0x52, 0x4b, 0x75, 0x94,
0x9e, 0x67, 0x7c, 0xce, 0xda, 0x86, 0x6f, 0x7f, 0x2b, 0xcc, 0x48, 0x44, 0xef,
0x2e, 0x5d, 0xac, 0x5b, 0x80, 0x4b, 0x40, 0x45, 0xe6, 0x57, 0xc8, 0x15, 0x6d,
0x1d, 0xcd, 0xb4, 0x3c, 0xbf, 0x2f, 0x5e, 0x00, 0xa4, 0xf9, 0x25, 0x5e, 0x3b,
0xe2, 0x43, 0x94, 0x36, 0xc4, 0xd0, 0x44, 0x9a, 0x8d, 0x2c, 0x4c, 0x1a, 0x56,
0xbe, 0xce, 0x98, 0xea, 0x0f, 0xd6, 0x8a, 0xba, 0xf1, 0x23, 0x98, 0x03, 0x99,
0x94, 0xae, 0xbf, 0xfc, 0x69, 0x2b, 0x90, 0x00, 0xe5, 0x80, 0x47, 0x9b, 0x4f,
0x4b, 0x28, 0xb5, 0xfe
},
.ciphertext = (unsigned char []){
0x95, 0x2f, 0x45, 0x46, 0xa8, 0xbf, 0x71, 0x66, 0x96, 0x49, 0x17, 0xec, 0xe0,
0x1b, 0xda, 0x3c, 0x68, 0x57, 0xe4, 0x27, 0xce, 0xf5, 0xda, 0x0f, 0xf9, 0x0b,
0x0e, 0x4b, 0xf4, 0x4c, 0xf7, 0xcc, 0xfc, 0xcf, 0xdf, 0x01, 0xd7, 0x13, 0xdc,
0xf9, 0x67, 0x3f, 0x01, 0xc8, 0x7e, 0xae, 0xd5, 0x2b, 0xf4, 0xaa, 0x04, 0x6f,
0xf7, 0x78, 0x55, 0x8e, 0xa3, 0x96, 0xdc, 0x9c, 0xd2, 0x40, 0x71, 0x61, 0x36,
0x38, 0x61, 0x48, 0xa5, 0xc7, 0x63, 0x78, 0xb3, 0xff, 0xcd, 0x40, 0x86, 0x44,
0x07, 0xb8, 0xe6, 0x0b, 0x40, 0xa5, 0x94, 0xe0, 0x61, 0x9e, 0xdd, 0xae, 0x3f,
0x6d, 0x6e, 0x3b, 0x15, 0xb8, 0x6a, 0xf2, 0x31, 0xe1, 0xba, 0xe5, 0xed, 0x2a,
0xa5, 0x12, 0xe1, 0x1d, 0xa0, 0xe5, 0x57, 0x2b, 0x67, 0xff, 0xff, 0x93, 0x4c,
0x36, 0xe5, 0x85, 0xcf, 0xdd, 0x9f, 0x87, 0x70, 0x45, 0xcb, 0x19, 0xc1, 0x83,
0xb9, 0x94, 0xbf, 0x74, 0x64, 0x58, 0x62, 0xff, 0xa7, 0x26, 0x73, 0x9a, 0xad,
0xcb, 0x9e, 0x10, 0xaa, 0xff, 0xc8, 0x81, 0xc8, 0x8c, 0xa3, 0xaa, 0x65, 0xb3,
0x7f, 0x66, 0x7b, 0xcb
},
},
{
.keylen = AES256_KEYLEN,
.len = 160,
.key = (unsigned char []){
0xc4, 0xa7, 0x1e, 0x05, 0x5a, 0x72, 0x54, 0xdd, 0xa3, 0x60, 0x69, 0x3f, 0xe1,
0xbe, 0x49, 0xf1, 0x0f, 0xaa, 0x67, 0x31, 0xc3, 0x6d, 0xba, 0xa6, 0x59, 0x0b,
0x05, 0x97, 0x4e, 0x18, 0x5c, 0x5b
},
.plaintext = (unsigned char []){
0x31, 0xfd, 0x5a, 0x30, 0x7e, 0x27, 0x9b, 0x2f, 0x34, 0x58, 0x1e, 0x2c, 0x43,
0x23, 0x79, 0xdf, 0x8e, 0xcc, 0xba, 0xf7, 0x95, 0x32, 0x93, 0x89, 0x16, 0x71,
0x1c, 0xd3, 0x77, 0x54, 0x0b, 0x90, 0x45, 0x37, 0x3e, 0x47, 0xf2, 0x21, 0x4b,
0x8f, 0x87, 0x60, 0x40, 0xaf, 0x73, 0x3f, 0x6c, 0x9d, 0x8f, 0x03, 0xa7, 0xc5,
0x8f, 0x87, 0x14, 0xd2, 0xfb, 0xb4, 0xc1, 0x4a, 0xf5, 0x9c, 0x75, 0xb4, 0x83,
0xad, 0xc7, 0x18, 0x94, 0x6e, 0xe9, 0x07, 0xa1, 0x82, 0x86, 0xcc, 0x4e, 0xfd,
0x20, 0x67, 0x89, 0x06, 0x4b, 0x6f, 0x1b, 0x19, 0x5f, 0x0d, 0x0d, 0x23, 0x44,
0x68, 0xe4, 0xf0, 0x0e, 0x6f, 0x1c, 0xad, 0x5c, 0xd3, 0xb9, 0xc0, 0xa6, 0x43,
0xb3, 0xc0, 0xdd, 0x09, 0x28, 0x0f, 0xf2, 0xe2, 0xa5, 0x92, 0x91, 0x83, 0x40,
0x93, 0x84, 0xdd, 0x72, 0xdc, 0x94, 0xe3, 0x96, 0x87, 0xea, 0x2b, 0x62, 0x3d,
0x5d, 0x77, 0x67, 0x00, 0xbd, 0x8b, 0x36, 0xe6, 0x13, 0x0f, 0xfd, 0xe9, 0x66,
0xf1, 0x34, 0xc4, 0xb1, 0xf3, 0x5f, 0x29, 0xc5, 0xcc, 0x4a, 0x03, 0x29, 0x7e,
0x1c, 0xcc, 0x95, 0x39
},
.ciphertext = (unsigned char []){
0x2c, 0x48, 0x7f, 0xa9, 0x6f, 0x40, 0x90, 0xc5, 0x6a, 0xa1, 0xb5, 0xbe, 0x81,
0x91, 0x8a, 0x93, 0x4c, 0x94, 0x92, 0x87, 0x8f, 0xb0, 0xcd, 0x68, 0x6d, 0xcf,
0x8d, 0x17, 0xd8, 0x64, 0x85, 0x45, 0x4c, 0x51, 0x23, 0x7b, 0xbd, 0x09, 0x20,
0x5d, 0xce, 0xf1, 0x55, 0x2f, 0x43, 0x0d, 0xd0, 0x98, 0xb9, 0xd8, 0x27, 0xa6,
0x94, 0x73, 0x0c, 0x13, 0x3a, 0x02, 0x22, 0xc7, 0x7f, 0x54, 0x0f, 0x9d, 0x5f,
0xc2, 0xd3, 0x6a, 0xf3, 0x59, 0x58, 0x3c, 0x9e, 0x3b, 0x49, 0xdf, 0x88, 0x42,
0x28, 0xa6, 0x4d, 0xe7, 0x9b, 0x67, 0xf6, 0x62, 0x07, 0xc8, 0x28, 0x13, 0x60,
0xb9, 0x9b, 0x21, 0x40, 0x42, 0xce, 0x61, 0x36, 0x7f, 0xf9, 0x79, 0x60, 0xe9,
0x44, 0x45, 0x3c, 0xd6, 0x36, 0x79, 0xbb, 0x44, 0x70, 0x88, 0x97, 0xd2, 0x9b,
0xc5, 0xe7, 0x0f, 0x9f, 0xc8, 0xf1, 0xf7, 0x15, 0x14, 0x3f, 0xbb, 0x00, 0xf7,
0xf5, 0xc1, 0xb7, 0xb1, 0x61, 0xec, 0x26, 0xd8, 0xd4, 0x1d, 0x36, 0xfa, 0xb0,
0xfa, 0x8a, 0x85, 0xc3, 0xee, 0x6c, 0xe4, 0xd3, 0x70, 0x07, 0xeb, 0x7a, 0x89,
0xd6, 0x75, 0x35, 0x90
},
},
};

const struct aes_cbc_tv AES_CBC_TV[] = {
{
.keylen = AES128_KEYLEN,
.len = 160,
.key = (unsigned char []){
0x97, 0xa1, 0x02, 0x55, 0x29, 0xb9, 0x92, 0x5e, 0x25, 0xbb, 0xe7, 0x87, 0x70,
0xca, 0x2f, 0x99
},
.iv = {
0xd4, 0xb4, 0xea, 0xb9, 0x2a, 0xa9, 0x63, 0x7e, 0x87, 0xd3, 0x66, 0x38, 0x4e,
0xd6, 0x91, 0x5c
},
.plaintext = (unsigned char []){
0xe8, 0xb8, 0x91, 0x50, 0xd8, 0x43, 0x8b, 0xf5, 0xb1, 0x74, 0x49, 0xd6, 0xed,
0x26, 0xbd, 0x72, 0x12, 0x7e, 0x10, 0xe4, 0xaa, 0x57, 0xca, 0xd8, 0x52, 0x83,
0xe8, 0x35, 0x9e, 0x08, 0x92, 0x08, 0xe8, 0x49, 0x21, 0x64, 0x9f, 0x5b, 0x60,
0xea, 0x21, 0xf7, 0x86, 0x7c, 0xbc, 0x96, 0x20, 0x56, 0x0c, 0x4c, 0x62, 0x38,
0xdb, 0x02, 0x12, 0x16, 0xdb, 0x45, 0x3c, 0x99, 0x43, 0xf1, 0xf1, 0xa6, 0x05,
0x46, 0x17, 0x3d, 0xae, 0xf2, 0x55, 0x7c, 0x3c, 0xdd, 0x85, 0x50, 0x31, 0xb3,
0x53, 0xd4, 0xbf, 0x17, 0x6f, 0x28, 0x43, 0x9e, 0x48, 0x78, 0x5c, 0x37, 0xd3,
0x8f, 0x27, 0x0a, 0xa4, 0xa6, 0xfa, 0xad, 0x2b, 0xaa, 0xbc, 0xb0, 0xc0, 0xb2,
0xd1, 0xdd, 0x53, 0x22, 0x93, 0x74, 0x98, 0xce, 0x80, 0x3b, 0xa1, 0x14, 0x84,
0x40, 0xa5, 0x2e, 0x22, 0x7d, 0xdb, 0xa4, 0x87, 0x2f, 0xe4, 0xd8, 0x1d, 0x2d,
0x76, 0xa9, 0x39, 0xd2, 0x47, 0x55, 0xad, 0xb8, 0xa7, 0xb8, 0x45, 0x2c, 0xee,
0xd2, 0xd1, 0x79, 0xe1, 0xa5, 0x84, 0x8f, 0x31, 0x6f, 0x5c, 0x01, 0x63, 0x00,
0xa3, 0x90, 0xbf, 0xa7
},
.ciphertext = (unsigned char []){
0x22, 0xcd, 0xc3, 0x30, 0x6f, 0xcd, 0x4d, 0x31, 0xcc, 0xd3, 0x27, 0x20, 0xcb,
0xb6, 0x1b, 0xad, 0x28, 0xd8, 0x55, 0x67, 0x06, 0x57, 0xc4, 0x8c, 0x7b, 0x88,
0xc3, 0x1f, 0x4f, 0xa1, 0xf9, 0x3c, 0x01, 0xb5, 0x7d, 0xa9, 0x0b, 0xe6, 0x3e,
0xad, 0x67, 0xd6, 0xa3, 0x25, 0x52, 0x5e, 0x6e, 0xd4, 0x50, 0x83, 0xe6, 0xfb,
0x70, 0xa5, 0x35, 0x29, 0xd1, 0xfa, 0x0f, 0x55, 0x65, 0x3b, 0x94, 0x2a, 0xf5,
0x9d, 0x78, 0xa2, 0x66, 0x03, 0x61, 0xd6, 0x3a, 0x72, 0x90, 0x15, 0x5a, 0xc5,
0xc4, 0x33, 0x12, 0xa2, 0x5b, 0x23, 0x5d, 0xac, 0xbb, 0xc8, 0x63, 0xfa, 0xf0,
0x09, 0x40, 0xc9, 0x96, 0x24, 0x07, 0x6d, 0xfa, 0x44, 0x06, 0x8e, 0x7c, 0x55,
0x4c, 0x90, 0x38, 0x17, 0x69, 0x53, 0xe5, 0x71, 0x75, 0x1d, 0xfc, 0x09, 0x54,
0xd4, 0x1d, 0x11, 0x37, 0x71, 0xb0, 0x64, 0x66, 0xb1, 0xc8, 0xd1, 0x3e, 0x0d,
0x4c, 0xb6, 0x75, 0xed, 0x58, 0xd1, 0xa6, 0x19, 0xe1, 0x54, 0x09, 0x70, 0x98,
0x37, 0x81, 0xdc, 0x11, 0xd2, 0xdd, 0x85, 0x25, 0xab, 0x57, 0x45, 0x95, 0x8d,
0x61, 0x5d, 0xef, 0xda
},
},
{
.keylen = AES192_KEYLEN,
.len = 160,
.key = (unsigned char []){
0x50, 0x9b, 0xaf, 0x46, 0xfb, 0x9d, 0xe3, 0x42, 0x81, 0xda, 0xfc, 0xc3, 0xdb,
0x79, 0x59, 0x3b, 0xff, 0xa8, 0x42, 0x69, 0x04, 0x30, 0x26, 0x88
},
.iv = {
0xd6, 0xd8, 0x6e, 0x0c, 0x82, 0xdd, 0x87, 0x88, 0xf4, 0x14, 0x7a, 0x26, 0xf9,
0xa7, 0x1c, 0x74
},
.plaintext = (unsigned char []){
0x67, 0xd2, 0xdd, 0xa6, 0xda, 0x26, 0xe2, 0x13, 0x07, 0x97, 0x34, 0x00, 0x60,
0x07, 0x25, 0x72, 0x7a, 0xe8, 0x14, 0x15, 0x51, 0x17, 0x72, 0xf4, 0xa0, 0x9a,
0xd9, 0x90, 0x3b, 0xcf, 0x90, 0xcc, 0x2c, 0x0d, 0xac, 0x58, 0xba, 0x55, 0x9a,
0x01, 0x09, 0xc5, 0x4a, 0x9d, 0x61, 0x17, 0xb1, 0x5b, 0xb5, 0x74, 0xca, 0x47,
0x3e, 0x84, 0x80, 0x47, 0xe9, 0xa5, 0x4e, 0xe4, 0xab, 0xde, 0x76, 0xaf, 0xf9,
0x84, 0x9c, 0x44, 0x10, 0x9d, 0x16, 0x1f, 0x46, 0x44, 0x2e, 0x16, 0x10, 0xd8,
0xb0, 0x15, 0xcf, 0x36, 0xa0, 0x10, 0xed, 0x8e, 0xfa, 0x32, 0x07, 0xfd, 0xfc,
0x8f, 0xcc, 0x54, 0x8f, 0x14, 0x5c, 0x02, 0x7e, 0x44, 0xc5, 0xb0, 0xec, 0x35,
0xc9, 0x88, 0x6f, 0x4b, 0x9d, 0x65, 0x13, 0xa5, 0xbc, 0x10, 0xd0, 0xea, 0x6b,
0xbb, 0xc2, 0x6f, 0x54, 0xb1, 0x83, 0xbc, 0xae, 0x27, 0xfb, 0x79, 0x9d, 0x88,
0x72, 0xff, 0x74, 0x8f, 0xc4, 0x59, 0xd5, 0x5c, 0xfa, 0x25, 0x5a, 0xae, 0x29,
0xd7, 0x1b, 0x07, 0x6d, 0x9b, 0x44, 0xc1, 0x4d, 0x5c, 0xeb, 0xa9, 0x33, 0x2a,
0x76, 0x3d, 0x9c, 0x94
},
.ciphertext = (unsigned char []){
0x69, 0x28, 0x29, 0x9c, 0x52, 0xb4, 0xf0, 0x47, 0x92, 0x6f, 0x8a, 0x54, 0x15,
0x29, 0xda, 0x2d, 0x6b, 0xba, 0xa3, 0x99, 0x14, 0x3c, 0xed, 0x8e, 0xfb, 0x77,
0xab, 0x47, 0x40, 0x9d, 0x9a, 0x95, 0x3a, 0x38, 0x6c, 0x7a, 0xbd, 0x60, 0x26,
0xf4, 0x98, 0x31, 0xc7, 0x17, 0x62, 0x7c, 0x2a, 0x5e, 0x77, 0xbd, 0x2d, 0x43,
0x3d, 0x4d, 0x13, 0x0d, 0xac, 0xd9, 0x27, 0xea, 0x0d, 0x13, 0xa2, 0x3d, 0x01,
0xa7, 0xcf, 0x39, 0xc6, 0x71, 0x6d, 0xaf, 0xb6, 0xed, 0x55, 0x24, 0x10, 0xef,
0x5d, 0x27, 0xfb, 0x94, 0x7b, 0xe2, 0xc8, 0x78, 0x2e, 0xee, 0x78, 0x29, 0x19,
0x6c, 0x7e, 0xdc, 0xf1, 0x51, 0xc6, 0x5f, 0x9a, 0x01, 0xf5, 0x4f, 0x8d, 0x20,
0xf3, 0x8b, 0x7d, 0xa4, 0xa7, 0xe8, 0x3a, 0x2f, 0x01, 0x27, 0xd5, 0x9d, 0x3e,
0x24, 0x05, 0xd8, 0x67, 0x4f, 0xc9, 0xf4, 0x1b, 0x60, 0x4f, 0x78, 0x8f, 0x47,
0x15, 0xf9, 0xd3, 0x62, 0x4e, 0xee, 0x57, 0xf3, 0x87, 0xbf, 0xad, 0xd1, 0x8a,
0x1f, 0x90, 0x5e, 0x83, 0x9c, 0x26, 0xb8, 0x61, 0x74, 0x82, 0x34, 0x7f, 0xab,
0x6d, 0x08, 0x84, 0x5a
},
},
{
.keylen = AES256_KEYLEN,
.len = 160,
.key = (unsigned char []){
0x87, 0x72, 0x5b, 0xd4, 0x3a, 0x45, 0x60, 0x88, 0x14, 0x18, 0x07, 0x73, 0xf0,
0xe7, 0xab, 0x95, 0xa3, 0xc8, 0x59, 0xd8, 0x3a, 0x21, 0x30, 0xe8, 0x84, 0x19,
0x0e, 0x44, 0xd1, 0x4c, 0x69, 0x96
},
.iv = {
0xe4, 0x96, 0x51, 0x98, 0x8e, 0xbb, 0xb7, 0x2e, 0xb8, 0xbb, 0x80, 0xbb, 0x9a,
0xbb, 0xca, 0x34
},
.plaintext = (unsigned char []){
0xbf, 0xe5, 0xc6, 0x35, 0x4b, 0x7a, 0x3f, 0xf3, 0xe1, 0x92, 0xe0, 0x57, 0x75,
0xb9, 0xb7, 0x58, 0x07, 0xde, 0x12, 0xe3, 0x8a, 0x62, 0x6b, 0x8b, 0xf0, 0xe1,
0x2d, 0x5f, 0xff, 0x78, 0xe4, 0xf1, 0x77, 0x5a, 0xa7, 0xd7, 0x92, 0xd8, 0x85,
0x16, 0x2e, 0x66, 0xd8, 0x89, 0x30, 0xf9, 0xc3, 0xb2, 0xcd, 0xf8, 0x65, 0x4f,
0x56, 0x97, 0x25, 0x04, 0x80, 0x31, 0x90, 0x38, 0x62, 0x70, 0xf0, 0xaa, 0x43,
0x64, 0x5d, 0xb1, 0x87, 0xaf, 0x41, 0xfc, 0xea, 0x63, 0x9b, 0x1f, 0x80, 0x26,
0xcc, 0xdd, 0x0c, 0x23, 0xe0, 0xde, 0x37, 0x09, 0x4a, 0x8b, 0x94, 0x1e, 0xcb,
0x76, 0x02, 0x99, 0x8a, 0x4b, 0x26, 0x04, 0xe6, 0x9f, 0xc0, 0x42, 0x19, 0x58,
0x5d, 0x85, 0x46, 0x00, 0xe0, 0xad, 0x6f, 0x99, 0xa5, 0x3b, 0x25, 0x04, 0x04,
0x3c, 0x08, 0xb1, 0xc3, 0xe2, 0x14, 0xd1, 0x7c, 0xde, 0x05, 0x3c, 0xbd, 0xf9,
0x1d, 0xaa, 0x99, 0x9e, 0xd5, 0xb4, 0x7c, 0x37, 0x98, 0x3b, 0xa3, 0xee, 0x25,
0x4b, 0xc5, 0xc7, 0x93, 0x83, 0x7d, 0xaa, 0xa8, 0xc8, 0x5c, 0xfc, 0x12, 0xf7,
0xf5, 0x4f, 0x69, 0x9f
},
.ciphertext = (unsigned char []){
0x5b, 0x97, 0xa9, 0xd4, 0x23, 0xf4, 0xb9, 0x74, 0x13, 0xf3, 0x88, 0xd9, 0xa3,
0x41, 0xe7, 0x27, 0xbb, 0x33, 0x9f, 0x8e, 0x18, 0xa3, 0xfa, 0xc2, 0xf2, 0xfb,
0x85, 0xab, 0xdc, 0x8f, 0x13, 0x5d, 0xeb, 0x30, 0x05, 0x4a, 0x1a, 0xfd, 0xc9,
0xb6, 0xed, 0x7d, 0xa1, 0x6c, 0x55, 0xeb, 0xa6, 0xb0, 0xd4, 0xd1, 0x0c, 0x74,
0xe1, 0xd9, 0xa7, 0xcf, 0x8e, 0xdf, 0xae, 0xaa, 0x68, 0x4a, 0xc0, 0xbd, 0x9f,
0x9d, 0x24, 0xba, 0x67, 0x49, 0x55, 0xc7, 0x9d, 0xc6, 0xbe, 0x32, 0xae, 0xe1,
0xc2, 0x60, 0xb5, 0x58, 0xff, 0x07, 0xe3, 0xa4, 0xd4, 0x9d, 0x24, 0x16, 0x20,
0x11, 0xff, 0x25, 0x4d, 0xb8, 0xbe, 0x07, 0x8e, 0x8a, 0xd0, 0x7e, 0x64, 0x8e,
0x6b, 0xf5, 0x67, 0x93, 0x76, 0xcb, 0x43, 0x21, 0xa5, 0xef, 0x01, 0xaf, 0xe6,
0xad, 0x88, 0x16, 0xfc, 0xc7, 0x63, 0x46, 0x69, 0xc8, 0xc4, 0x38, 0x92, 0x95,
0xc9, 0x24, 0x1e, 0x45, 0xff, 0xf3, 0x9f, 0x32, 0x25, 0xf7, 0x74, 0x50, 0x32,
0xda, 0xee, 0xbe, 0x99, 0xd4, 0xb1, 0x9b, 0xcb, 0x21, 0x5d, 0x1b, 0xfd, 0xb3,
0x6e, 0xda, 0x2c, 0x24
},
},
};

const struct aes_cbc_cs_tv AES_CBC_CS_TV[] = {
{
.keylen = AES128_KEYLEN,
.len = 47,
.key = (unsigned char []){
0x63, 0x68, 0x69, 0x63, 0x6b, 0x65, 0x6e, 0x20, 0x74, 0x65, 0x72, 0x69, 0x79,
0x61, 0x6b, 0x69
},
.iv = {0},
.iv_out = {
0xb3, 0xff, 0xfd, 0x94, 0x0c, 0x16, 0xa1, 0x8c, 0x1b, 0x55, 0x49, 0xd2, 0xf8,
0x38, 0x02, 0x9e
},
.plaintext = (unsigned char []){
0x49, 0x20, 0x77, 0x6f, 0x75, 0x6c, 0x64, 0x20, 0x6c, 0x69, 0x6b, 0x65, 0x20,
0x74, 0x68, 0x65, 0x20, 0x47, 0x65, 0x6e, 0x65, 0x72, 0x61, 0x6c, 0x20, 0x47,
0x61, 0x75, 0x27, 0x73, 0x20, 0x43, 0x68, 0x69, 0x63, 0x6b, 0x65, 0x6e, 0x2c,
0x20, 0x70, 0x6c, 0x65, 0x61, 0x73, 0x65, 0x2c
},
.ciphertext = (unsigned char []){
0x97, 0x68, 0x72, 0x68, 0xd6, 0xec, 0xcc, 0xc0, 0xc0, 0x7b, 0x25, 0xe2, 0x5e,
0xcf, 0xe5, 0x84, 0x39, 0x31, 0x25, 0x23, 0xa7, 0x86, 0x62, 0xd5, 0xbe, 0x7f,
0xcb, 0xcc, 0x98, 0xeb, 0xf5, 0xb3, 0xff, 0xfd, 0x94, 0x0c, 0x16, 0xa1, 0x8c,
0x1b, 0x55, 0x49, 0xd2, 0xf8, 0x38, 0x02, 0x9e
},
.variant = 1,
},
{
.keylen = AES128_KEYLEN,
.len = 47,
.key = (unsigned char []){
0x63, 0x68, 0x69, 0x63, 0x6b, 0x65, 0x6e, 0x20, 0x74, 0x65, 0x72, 0x69, 0x79,
0x61, 0x6b, 0x69
},
.iv = {0},
.iv_out = {
0xb3, 0xff, 0xfd, 0x94, 0x0c, 0x16, 0xa1, 0x8c, 0x1b, 0x55, 0x49, 0xd2, 0xf8,
0x38, 0x02, 0x9e
},
.plaintext = (unsigned char []){
0x49, 0x20, 0x77, 0x6f, 0x75, 0x6c, 0x64, 0x20, 0x6c, 0x69, 0x6b, 0x65, 0x20,
0x74, 0x68, 0x65, 0x20, 0x47, 0x65, 0x6e, 0x65, 0x72, 0x61, 0x6c, 0x20, 0x47,
0x61, 0x75, 0x27, 0x73, 0x20, 0x43, 0x68, 0x69, 0x63, 0x6b, 0x65, 0x6e, 0x2c,
0x20, 0x70, 0x6c, 0x65, 0x61, 0x73, 0x65, 0x2c
},
.ciphertext = (unsigned char []){
0x97, 0x68, 0x72, 0x68, 0xd6, 0xec, 0xcc, 0xc0, 0xc0, 0x7b, 0x25, 0xe2, 0x5e,
0xcf, 0xe5, 0x84, 0xb3, 0xff, 0xfd, 0x94, 0x0c, 0x16, 0xa1, 0x8c, 0x1b, 0x55,
0x49, 0xd2, 0xf8, 0x38, 0x02, 0x9e, 0x39, 0x31, 0x25, 0x23, 0xa7, 0x86, 0x62,
0xd5, 0xbe, 0x7f, 0xcb, 0xcc, 0x98, 0xeb, 0xf5
},
.variant = 2,
},
{
.keylen = AES128_KEYLEN,
.len = 47,
.key = (unsigned char []){
0x63, 0x68, 0x69, 0x63, 0x6b, 0x65, 0x6e, 0x20, 0x74, 0x65, 0x72, 0x69, 0x79,
0x61, 0x6b, 0x69
},
.iv = {0},
.iv_out = {
0xb3, 0xff, 0xfd, 0x94, 0x0c, 0x16, 0xa1, 0x8c, 0x1b, 0x55, 0x49, 0xd2, 0xf8,
0x38, 0x02, 0x9e
},
.plaintext = (unsigned char []){
0x49, 0x20, 0x77, 0x6f, 0x75, 0x6c, 0x64, 0x20, 0x6c, 0x69, 0x6b, 0x65, 0x20,
0x74, 0x68, 0x65, 0x20, 0x47, 0x65, 0x6e, 0x65, 0x72, 0x61, 0x6c, 0x20, 0x47,
0x61, 0x75, 0x27, 0x73, 0x20, 0x43, 0x68, 0x69, 0x63, 0x6b, 0x65, 0x6e, 0x2c,
0x20, 0x70, 0x6c, 0x65, 0x61, 0x73, 0x65, 0x2c
},
.ciphertext = (unsigned char []){
0x97, 0x68, 0x72, 0x68, 0xd6, 0xec, 0xcc, 0xc0, 0xc0, 0x7b, 0x25, 0xe2, 0x5e,
0xcf, 0xe5, 0x84, 0xb3, 0xff, 0xfd, 0x94, 0x0c, 0x16, 0xa1, 0x8c, 0x1b, 0x55,
0x49, 0xd2, 0xf8, 0x38, 0x02, 0x9e, 0x39, 0x31, 0x25, 0x23, 0xa7, 0x86, 0x62,
0xd5, 0xbe, 0x7f, 0xcb, 0xcc, 0x98, 0xeb, 0xf5
},
.variant = 3,
},
{
.keylen = AES128_KEYLEN,
.len = 64,
.key = (unsigned char []){
0x63, 0x68, 0x69, 0x63, 0x6b, 0x65, 0x6e, 0x20, 0x74, 0x65, 0x72, 0x69, 0x79,
0x61, 0x6b, 0x69
},
.iv = {0},
.iv_out = {
0x48, 0x07, 0xef, 0xe8, 0x36, 0xee, 0x89, 0xa5, 0x26, 0x73, 0x0d, 0xbc, 0x2f,
0x7b, 0xc8, 0x40
},
.plaintext = (unsigned char []){
0x49, 0x20, 0x77, 0x6f, 0x75, 0x6c, 0x64, 0x20, 0x6c, 0x69, 0x6b, 0x65, 0x20,
0x74, 0x68, 0x65, 0x20, 0x47, 0x65, 0x6e, 0x65, 0x72, 0x61, 0x6c, 0x20, 0x47,
0x61, 0x75, 0x27, 0x73, 0x20, 0x43, 0x68, 0x69, 0x63, 0x6b, 0x65, 0x6e, 0x2c,
0x20, 0x70, 0x6c, 0x65, 0x61, 0x73, 0x65, 0x2c, 0x20, 0x61, 0x6e, 0x64, 0x20,
0x77, 0x6f, 0x6e, 0x74, 0x6f, 0x6e, 0x20, 0x73, 0x6f, 0x75, 0x70, 0x2e
},
.ciphertext = (unsigned char []){
0x97, 0x68, 0x72, 0x68, 0xd6, 0xec, 0xcc, 0xc0, 0xc0, 0x7b, 0x25, 0xe2, 0x5e,
0xcf, 0xe5, 0x84, 0x39, 0x31, 0x25, 0x23, 0xa7, 0x86, 0x62, 0xd5, 0xbe, 0x7f,
0xcb, 0xcc, 0x98, 0xeb, 0xf5, 0xa8, 0x9d, 0xad, 0x8b, 0xbb, 0x96, 0xc4, 0xcd,
0xc0, 0x3b, 0xc1, 0x03, 0xe1, 0xa1, 0x94, 0xbb, 0xd8, 0x48, 0x07, 0xef, 0xe8,
0x36, 0xee, 0x89, 0xa5, 0x26, 0x73, 0x0d, 0xbc, 0x2f, 0x7b, 0xc8, 0x40
},
.variant = 1,
},
{
.keylen = AES128_KEYLEN,
.len = 64,
.key = (unsigned char []){
0x63, 0x68, 0x69, 0x63, 0x6b, 0x65, 0x6e, 0x20, 0x74, 0x65, 0x72, 0x69, 0x79,
0x61, 0x6b, 0x69
},
.iv = {0},
.iv_out = {
0x48, 0x07, 0xef, 0xe8, 0x36, 0xee, 0x89, 0xa5, 0x26, 0x73, 0x0d, 0xbc, 0x2f,
0x7b, 0xc8, 0x40
},
.plaintext = (unsigned char []){
0x49, 0x20, 0x77, 0x6f, 0x75, 0x6c, 0x64, 0x20, 0x6c, 0x69, 0x6b, 0x65, 0x20,
0x74, 0x68, 0x65, 0x20, 0x47, 0x65, 0x6e, 0x65, 0x72, 0x61, 0x6c, 0x20, 0x47,
0x61, 0x75, 0x27, 0x73, 0x20, 0x43, 0x68, 0x69, 0x63, 0x6b, 0x65, 0x6e, 0x2c,
0x20, 0x70, 0x6c, 0x65, 0x61, 0x73, 0x65, 0x2c, 0x20, 0x61, 0x6e, 0x64, 0x20,
0x77, 0x6f, 0x6e, 0x74, 0x6f, 0x6e, 0x20, 0x73, 0x6f, 0x75, 0x70, 0x2e
},
.ciphertext = (unsigned char []){
0x97, 0x68, 0x72, 0x68, 0xd6, 0xec, 0xcc, 0xc0, 0xc0, 0x7b, 0x25, 0xe2, 0x5e,
0xcf, 0xe5, 0x84, 0x39, 0x31, 0x25, 0x23, 0xa7, 0x86, 0x62, 0xd5, 0xbe, 0x7f,
0xcb, 0xcc, 0x98, 0xeb, 0xf5, 0xa8, 0x9d, 0xad, 0x8b, 0xbb, 0x96, 0xc4, 0xcd,
0xc0, 0x3b, 0xc1, 0x03, 0xe1, 0xa1, 0x94, 0xbb, 0xd8, 0x48, 0x07, 0xef, 0xe8,
0x36, 0xee, 0x89, 0xa5, 0x26, 0x73, 0x0d, 0xbc, 0x2f, 0x7b, 0xc8, 0x40
},
.variant = 2,
},
{
.keylen = AES128_KEYLEN,
.len = 64,
.key = (unsigned char []){
0x63, 0x68, 0x69, 0x63, 0x6b, 0x65, 0x6e, 0x20, 0x74, 0x65, 0x72, 0x69, 0x79,
0x61, 0x6b, 0x69
},
.iv = {0},
.iv_out = {
0x48, 0x07, 0xef, 0xe8, 0x36, 0xee, 0x89, 0xa5, 0x26, 0x73, 0x0d, 0xbc, 0x2f,
0x7b, 0xc8, 0x40
},
.plaintext = (unsigned char []){
0x49, 0x20, 0x77, 0x6f, 0x75, 0x6c, 0x64, 0x20, 0x6c, 0x69, 0x6b, 0x65, 0x20,
0x74, 0x68, 0x65, 0x20, 0x47, 0x65, 0x6e, 0x65, 0x72, 0x61, 0x6c, 0x20, 0x47,
0x61, 0x75, 0x27, 0x73, 0x20, 0x43, 0x68, 0x69, 0x63, 0x6b, 0x65, 0x6e, 0x2c,
0x20, 0x70, 0x6c, 0x65, 0x61, 0x73, 0x65, 0x2c, 0x20, 0x61, 0x6e, 0x64, 0x20,
0x77, 0x6f, 0x6e, 0x74, 0x6f, 0x6e, 0x20, 0x73, 0x6f, 0x75, 0x70, 0x2e
},
.ciphertext = (unsigned char []){
0x97, 0x68, 0x72, 0x68, 0xd6, 0xec, 0xcc, 0xc0, 0xc0, 0x7b, 0x25, 0xe2, 0x5e,
0xcf, 0xe5, 0x84, 0x39, 0x31, 0x25, 0x23, 0xa7, 0x86, 0x62, 0xd5, 0xbe, 0x7f,
0xcb, 0xcc, 0x98, 0xeb, 0xf5, 0xa8, 0x48, 0x07, 0xef, 0xe8, 0x36, 0xee, 0x89,
0xa5, 0x26, 0x73, 0x0d, 0xbc, 0x2f, 0x7b, 0xc8, 0x40, 0x9d, 0xad, 0x8b, 0xbb,
0x96, 0xc4, 0xcd, 0xc0, 0x3b, 0xc1, 0x03, 0xe1, 0xa1, 0x94, 0xbb, 0xd8
},
.variant = 3,
},
};

const struct des3_cbc_cs_tv DES3_CBC_CS_TV[] = {
{
.len = 71,
.key = {
0x2a, 0xe6, 0xae, 0x5d, 0x75, 0x34, 0x8f, 0x9d, 0x34, 0xc4, 0xfd, 0x45, 0x85,
0xb6, 0xba, 0x8c, 0x9d, 0xef, 0x6e, 0xbf, 0xc7, 0x7f, 0xe0, 0x46
},
.iv = {
0xa5, 0x2b, 0x66, 0xc3, 0x34, 0x38, 0x59, 0xe0
},
.plaintext = (unsigned char []){
0x24, 0x19, 0x26, 0x42, 0xf7, 0xc7, 0x0e, 0x7d, 0x75, 0x69, 0xde, 0x8b, 0x86,
0x20, 0x4a, 0x10, 0xa0, 0x76, 0xe3, 0xf3, 0x79, 0xbc, 0x7f, 0x8a, 0x52, 0x25,
0xec, 0x4d, 0x76, 0xc0, 0x8a, 0xca, 0x40, 0xcd, 0xe8, 0x86, 0xf5, 0xd9, 0x9b,
0xc0, 0xb6, 0x65, 0xa1, 0x79, 0x56, 0x37, 0x22, 0x19, 0xcd, 0x52, 0xcb, 0x73,
0xec, 0x48, 0x62, 0xdf, 0x24, 0x2f, 0xb7, 0x65, 0xb1, 0x85, 0x3b, 0xc6, 0x25,
0x82, 0x41, 0xe4, 0x1a, 0x1a, 0xd7
},
.ciphertext = (unsigned char []){
0x95, 0x0e, 0x26, 0xc3, 0xf2, 0x46, 0xd7, 0xed, 0xff, 0xbd, 0xd3, 0x0e, 0x9a,
0x2c, 0x08, 0x2b, 0x48, 0x53, 0x3f, 0xbf, 0x26, 0x29, 0xa9, 0x00, 0x23, 0x35,
0xbb, 0x63, 0x6d, 0x3c, 0x3c, 0x62, 0x29, 0x1c, 0x9d, 0x7d, 0x67, 0xf8, 0x88,
0x7f, 0x55, 0x95, 0xbb, 0x02, 0xb2, 0x04, 0x17, 0x35, 0x86, 0x82, 0xf4, 0xdd,
0x11, 0x9c, 0xb8, 0x9a, 0x01, 0x97, 0xa8, 0x6c, 0xe8, 0x0e, 0xfe, 0xe7, 0x96,
0x5f, 0x36, 0xfc, 0xc3, 0xd9, 0xa5
},
.variant = 1,
},
{
.len = 71,
.key = {
0x2a, 0xe6, 0xae, 0x5d, 0x75, 0x34, 0x8f, 0x9d, 0x34, 0xc4, 0xfd, 0x45, 0x85,
0xb6, 0xba, 0x8c, 0x9d, 0xef, 0x6e, 0xbf, 0xc7, 0x7f, 0xe0, 0x46
},
.iv = {
0xa5, 0x2b, 0x66, 0xc3, 0x34, 0x38, 0x59, 0xe0
},
.plaintext = (unsigned char []){
0x24, 0x19, 0x26, 0x42, 0xf7, 0xc7, 0x0e, 0x7d, 0x75, 0x69, 0xde, 0x8b, 0x86,
0x20, 0x4a, 0x10, 0xa0, 0x76, 0xe3, 0xf3, 0x79, 0xbc, 0x7f, 0x8a, 0x52, 0x25,
0xec, 0x4d, 0x76, 0xc0, 0x8a, 0xca, 0x40, 0xcd, 0xe8, 0x86, 0xf5, 0xd9, 0x9b,
0xc0, 0xb6, 0x65, 0xa1, 0x79, 0x56, 0x37, 0x22, 0x19, 0xcd, 0x52, 0xcb, 0x73,
0xec, 0x48, 0x62, 0xdf, 0x24, 0x2f, 0xb7, 0x65, 0xb1, 0x85, 0x3b, 0xc6, 0x25,
0x82, 0x41, 0xe4, 0x1a, 0x1a, 0xd7
},
.ciphertext = (unsigned char []){
0x95, 0x0e, 0x26, 0xc3, 0xf2, 0x46, 0xd7, 0xed, 0xff, 0xbd, 0xd3, 0x0e, 0x9a,
0x2c, 0x08, 0x2b, 0x48, 0x53, 0x3f, 0xbf, 0x26, 0x29, 0xa9, 0x00, 0x23, 0x35,
0xbb, 0x63, 0x6d, 0x3c, 0x3c, 0x62, 0x29, 0x1c, 0x9d, 0x7d, 0x67, 0xf8, 0x88,
0x7f, 0x55, 0x95, 0xbb, 0x02, 0xb2, 0x04, 0x17, 0x35, 0x86, 0x82, 0xf4, 0xdd,
0x11, 0x9c, 0xb8, 0x9a, 0xe7, 0x96, 0x5f, 0x36, 0xfc, 0xc3, 0xd9, 0xa5,
0x01, 0x97, 0xa8, 0x6c, 0xe8, 0x0e, 0xfe
},
.variant = 2,
},
{
.len = 71,
.key = {
0x2a, 0xe6, 0xae, 0x5d, 0x75, 0x34, 0x8f, 0x9d, 0x34, 0xc4, 0xfd, 0x45, 0x85,
0xb6, 0xba, 0x8c, 0x9d, 0xef, 0x6e, 0xbf, 0xc7, 0x7f, 0xe0, 0x46
},
.iv = {
0xa5, 0x2b, 0x66, 0xc3, 0x34, 0x38, 0x59, 0xe0
},
.plaintext = (unsigned char []){
0x24, 0x19, 0x26, 0x42, 0xf7, 0xc7, 0x0e, 0x7d, 0x75, 0x69, 0xde, 0x8b, 0x86,
0x20, 0x4a, 0x10, 0xa0, 0x76, 0xe3, 0xf3, 0x79, 0xbc, 0x7f, 0x8a, 0x52, 0x25,
0xec, 0x4d, 0x76, 0xc0, 0x8a, 0xca, 0x40, 0xcd, 0xe8, 0x86, 0xf5, 0xd9, 0x9b,
0xc0, 0xb6, 0x65, 0xa1, 0x79, 0x56, 0x37, 0x22, 0x19, 0xcd, 0x52, 0xcb, 0x73,
0xec, 0x48, 0x62, 0xdf, 0x24, 0x2f, 0xb7, 0x65, 0xb1, 0x85, 0x3b, 0xc6, 0x25,
0x82, 0x41, 0xe4, 0x1a, 0x1a, 0xd7
},
.ciphertext = (unsigned char []){
0x95, 0x0e, 0x26, 0xc3, 0xf2, 0x46, 0xd7, 0xed, 0xff, 0xbd, 0xd3, 0x0e, 0x9a,
0x2c, 0x08, 0x2b, 0x48, 0x53, 0x3f, 0xbf, 0x26, 0x29, 0xa9, 0x00, 0x23, 0x35,
0xbb, 0x63, 0x6d, 0x3c, 0x3c, 0x62, 0x29, 0x1c, 0x9d, 0x7d, 0x67, 0xf8, 0x88,
0x7f, 0x55, 0x95, 0xbb, 0x02, 0xb2, 0x04, 0x17, 0x35, 0x86, 0x82, 0xf4, 0xdd,
0x11, 0x9c, 0xb8, 0x9a, 0xe7, 0x96, 0x5f, 0x36, 0xfc, 0xc3, 0xd9, 0xa5, 0x01,
0x97, 0xa8, 0x6c, 0xe8, 0x0e, 0xfe
},
.variant = 3,
},
{
.len = 72,
.key = {
0x2a, 0xe6, 0xae, 0x5d, 0x75, 0x34, 0x8f, 0x9d, 0x34, 0xc4, 0xfd, 0x45, 0x85,
0xb6, 0xba, 0x8c, 0x9d, 0xef, 0x6e, 0xbf, 0xc7, 0x7f, 0xe0, 0x46
},
.iv = {
0xa5, 0x2b, 0x66, 0xc3, 0x34, 0x38, 0x59, 0xe0
},
.plaintext = (unsigned char []){
0x24, 0x19, 0x26, 0x42, 0xf7, 0xc7, 0x0e, 0x7d, 0x75, 0x69, 0xde, 0x8b, 0x86,
0x20, 0x4a, 0x10, 0xa0, 0x76, 0xe3, 0xf3, 0x79, 0xbc, 0x7f, 0x8a, 0x52, 0x25,
0xec, 0x4d, 0x76, 0xc0, 0x8a, 0xca, 0x40, 0xcd, 0xe8, 0x86, 0xf5, 0xd9, 0x9b,
0xc0, 0xb6, 0x65, 0xa1, 0x79, 0x56, 0x37, 0x22, 0x19, 0xcd, 0x52, 0xcb, 0x73,
0xec, 0x48, 0x62, 0xdf, 0x24, 0x2f, 0xb7, 0x65, 0xb1, 0x85, 0x3b, 0xc6, 0x25,
0x82, 0x41, 0xe4, 0x1a, 0x1a, 0xd7, 0x15
},
.ciphertext = (unsigned char []){
0x95, 0x0e, 0x26, 0xc3, 0xf2, 0x46, 0xd7, 0xed, 0xff, 0xbd, 0xd3, 0x0e, 0x9a,
0x2c, 0x08, 0x2b, 0x48, 0x53, 0x3f, 0xbf, 0x26, 0x29, 0xa9, 0x00, 0x23, 0x35,
0xbb, 0x63, 0x6d, 0x3c, 0x3c, 0x62, 0x29, 0x1c, 0x9d, 0x7d, 0x67, 0xf8, 0x88,
0x7f, 0x55, 0x95, 0xbb, 0x02, 0xb2, 0x04, 0x17, 0x35, 0x86, 0x82, 0xf4, 0xdd,
0x11, 0x9c, 0xb8, 0x9a, 0x01, 0x97, 0xa8, 0x6c, 0xe8, 0x0e, 0xfe, 0x75, 0xec,
0xdd, 0x3b, 0x01, 0xc1, 0x3e, 0xbe, 0xc3
},
.variant = 1,
},
{
.len = 72,
.key = {
0x2a, 0xe6, 0xae, 0x5d, 0x75, 0x34, 0x8f, 0x9d, 0x34, 0xc4, 0xfd, 0x45, 0x85,
0xb6, 0xba, 0x8c, 0x9d, 0xef, 0x6e, 0xbf, 0xc7, 0x7f, 0xe0, 0x46
},
.iv = {
0xa5, 0x2b, 0x66, 0xc3, 0x34, 0x38, 0x59, 0xe0
},
.plaintext = (unsigned char []){
0x24, 0x19, 0x26, 0x42, 0xf7, 0xc7, 0x0e, 0x7d, 0x75, 0x69, 0xde, 0x8b, 0x86,
0x20, 0x4a, 0x10, 0xa0, 0x76, 0xe3, 0xf3, 0x79, 0xbc, 0x7f, 0x8a, 0x52, 0x25,
0xec, 0x4d, 0x76, 0xc0, 0x8a, 0xca, 0x40, 0xcd, 0xe8, 0x86, 0xf5, 0xd9, 0x9b,
0xc0, 0xb6, 0x65, 0xa1, 0x79, 0x56, 0x37, 0x22, 0x19, 0xcd, 0x52, 0xcb, 0x73,
0xec, 0x48, 0x62, 0xdf, 0x24, 0x2f, 0xb7, 0x65, 0xb1, 0x85, 0x3b, 0xc6, 0x25,
0x82, 0x41, 0xe4, 0x1a, 0x1a, 0xd7, 0x15
},
.ciphertext = (unsigned char []){
0x95, 0x0e, 0x26, 0xc3, 0xf2, 0x46, 0xd7, 0xed, 0xff, 0xbd, 0xd3, 0x0e, 0x9a,
0x2c, 0x08, 0x2b, 0x48, 0x53, 0x3f, 0xbf, 0x26, 0x29, 0xa9, 0x00, 0x23, 0x35,
0xbb, 0x63, 0x6d, 0x3c, 0x3c, 0x62, 0x29, 0x1c, 0x9d, 0x7d, 0x67, 0xf8, 0x88,
0x7f, 0x55, 0x95, 0xbb, 0x02, 0xb2, 0x04, 0x17, 0x35, 0x86, 0x82, 0xf4, 0xdd,
0x11, 0x9c, 0xb8, 0x9a, 0x01, 0x97, 0xa8, 0x6c, 0xe8, 0x0e, 0xfe, 0x75, 0xec,
0xdd, 0x3b, 0x01, 0xc1, 0x3e, 0xbe, 0xc3
},
.variant = 2,
},
{
.len = 72,
.key = {
0x2a, 0xe6, 0xae, 0x5d, 0x75, 0x34, 0x8f, 0x9d, 0x34, 0xc4, 0xfd, 0x45, 0x85,
0xb6, 0xba, 0x8c, 0x9d, 0xef, 0x6e, 0xbf, 0xc7, 0x7f, 0xe0, 0x46
},
.iv = {
0xa5, 0x2b, 0x66, 0xc3, 0x34, 0x38, 0x59, 0xe0
},
.plaintext = (unsigned char []){
0x24, 0x19, 0x26, 0x42, 0xf7, 0xc7, 0x0e, 0x7d, 0x75, 0x69, 0xde, 0x8b, 0x86,
0x20, 0x4a, 0x10, 0xa0, 0x76, 0xe3, 0xf3, 0x79, 0xbc, 0x7f, 0x8a, 0x52, 0x25,
0xec, 0x4d, 0x76, 0xc0, 0x8a, 0xca, 0x40, 0xcd, 0xe8, 0x86, 0xf5, 0xd9, 0x9b,
0xc0, 0xb6, 0x65, 0xa1, 0x79, 0x56, 0x37, 0x22, 0x19, 0xcd, 0x52, 0xcb, 0x73,
0xec, 0x48, 0x62, 0xdf, 0x24, 0x2f, 0xb7, 0x65, 0xb1, 0x85, 0x3b, 0xc6, 0x25,
0x82, 0x41, 0xe4, 0x1a, 0x1a, 0xd7, 0x15
},
.ciphertext = (unsigned char []){
0x95, 0x0e, 0x26, 0xc3, 0xf2, 0x46, 0xd7, 0xed, 0xff, 0xbd, 0xd3, 0x0e, 0x9a,
0x2c, 0x08, 0x2b, 0x48, 0x53, 0x3f, 0xbf, 0x26, 0x29, 0xa9, 0x00, 0x23, 0x35,
0xbb, 0x63, 0x6d, 0x3c, 0x3c, 0x62, 0x29, 0x1c, 0x9d, 0x7d, 0x67, 0xf8, 0x88,
0x7f, 0x55, 0x95, 0xbb, 0x02, 0xb2, 0x04, 0x17, 0x35, 0x86, 0x82, 0xf4, 0xdd,
0x11, 0x9c, 0xb8, 0x9a, 0xec, 0xdd, 0x3b, 0x01, 0xc1, 0x3e, 0xbe, 0xc3, 0x01,
0x97, 0xa8, 0x6c, 0xe8, 0x0e, 0xfe, 0x75,
},
.variant = 3,
},
};

const struct aes_cfb_tv AES_CFB_TV[] = {
{
.keylen = AES128_KEYLEN,
.len = 10,
.lcfb = 1,
.key = (unsigned char []){
0x63, 0x99, 0xc1, 0xdc, 0x06, 0x8b, 0xa3, 0x50, 0x98, 0x45, 0x62, 0x8f, 0xa9,
0xed, 0x1a, 0x96
},
.iv = {
0x11, 0x57, 0xc2, 0x76, 0x6c, 0x86, 0xb7, 0x54, 0xdf, 0x48, 0x5b, 0xe9, 0xdd,
0x58, 0x51, 0xdf
},
.plaintext = (unsigned char []){
0xfe, 0xff, 0x4e, 0x2e, 0x24, 0x58, 0xad, 0xdf, 0x2a, 0x54
},
.ciphertext = (unsigned char []){
0xc9, 0xc2, 0x84, 0xe9, 0xab, 0xbf, 0xe6, 0xfb, 0x11, 0xfe
},
},
{
.keylen = AES192_KEYLEN,
.len = 10,
.lcfb = 1,
.key = (unsigned char []){
0xba, 0xf0, 0x8b, 0x76, 0x31, 0x7a, 0x65, 0xc5, 0xf0, 0x7a, 0xe6, 0xf5, 0x7e,
0xb0, 0xe6, 0x54, 0x88, 0x65, 0x93, 0x24, 0xd2, 0x97, 0x09, 0xe3
},
.iv = {
0x0a, 0x02, 0x84, 0x6b, 0x62, 0xab, 0xb6, 0x93, 0xef, 0x31, 0xd7, 0x54, 0x84,
0x2e, 0xed, 0x29
},
.plaintext = (unsigned char []){
0x98, 0x95, 0x93, 0x24, 0x02, 0x39, 0x3d, 0xc3, 0x3a, 0x60
},
.ciphertext = (unsigned char []){
0x72, 0x9c, 0x0b, 0x6d, 0xeb, 0x75, 0xfa, 0x6e, 0xb5, 0xe8
},
},
{
.keylen = AES256_KEYLEN,
.len = 10,
.lcfb = 1,
.key = (unsigned char []){
0xec, 0x13, 0x06, 0x25, 0x51, 0xe4, 0xd7, 0x29, 0x1e, 0x32, 0x0f, 0x56, 0x5b,
0x74, 0x9e, 0xea, 0x18, 0x09, 0xb6, 0x63, 0xb2, 0x6f, 0x2c, 0x4d, 0x53, 0xb5,
0x20, 0x58, 0xb8, 0x33, 0xe0, 0xad
},
.iv = {
0xfb, 0xfa, 0x5a, 0x52, 0x8e, 0x20, 0x86, 0x30, 0x12, 0x79, 0x0c, 0x2a, 0xba,
0xfb, 0x5a, 0x0c
},
.plaintext = (unsigned char []){
0x54, 0x7b, 0xfd, 0x64, 0x2c, 0xf6, 0xe1, 0x2e, 0xd9, 0x42
},
.ciphertext = (unsigned char []){
0x2b, 0xfc, 0x3f, 0x02, 0x09, 0x30, 0x71, 0x40, 0x10, 0x1a
},
},
{
.keylen = AES128_KEYLEN,
.len = 160,
.lcfb = 16,
.key = (unsigned char []){
0xae, 0xf4, 0x9d, 0xa3, 0x3f, 0x53, 0x8e, 0xe6, 0x6e, 0x17, 0x8d, 0x4b, 0x61,
0x21, 0x05, 0x5d
},
.iv = {
0x84, 0x25, 0x66, 0xe6, 0x8b, 0x61, 0xff, 0x7b, 0xf0, 0x01, 0xf2, 0x64, 0x2d,
0xa6, 0x2f, 0x64
},
.plaintext = (unsigned char []){
0x41, 0x59, 0x91, 0xf6, 0x5e, 0x1a, 0x95, 0x04, 0x0c, 0xef, 0x99, 0x60, 0x55,
0x6f, 0x61, 0xe6, 0x17, 0x82, 0x7c, 0x30, 0xc7, 0x4b, 0xf3, 0x53, 0xcd, 0xd8,
0x61, 0x73, 0xdb, 0xe4, 0xcc, 0x98, 0x3a, 0x2e, 0xe6, 0xbc, 0x8c, 0xa6, 0xcf,
0xb7, 0x11, 0x21, 0xe7, 0xb0, 0xd0, 0x17, 0x8f, 0x2e, 0x13, 0x44, 0x5c, 0x71,
0x0d, 0xcc, 0x17, 0x6b, 0x78, 0x12, 0x01, 0x97, 0x11, 0x71, 0xf7, 0x48, 0x9f,
0x18, 0xfa, 0xf1, 0x10, 0xf3, 0x9a, 0xcc, 0xd1, 0xcf, 0x08, 0xc8, 0x5a, 0x95,
0x8d, 0x76, 0x98, 0xb1, 0x16, 0xf1, 0xc0, 0xd7, 0x58, 0x12, 0xac, 0x9b, 0x0b,
0x39, 0xae, 0xe7, 0xf7, 0x15, 0x9c, 0xca, 0xd8, 0xfd, 0xae, 0x9b, 0x99, 0xf2,
0xd6, 0x95, 0xea, 0xcf, 0x12, 0xc6, 0x46, 0x9d, 0x5b, 0x51, 0xa3, 0x4d, 0xe2,
0x6e, 0xac, 0x73, 0x61, 0x3d, 0xcb, 0x2f, 0x77, 0x12, 0x2c, 0xb1, 0xf8, 0xdd,
0x51, 0x62, 0x78, 0x6a, 0x12, 0x05, 0x2d, 0xc7, 0xb6, 0xde, 0xa6, 0xac, 0xc4,
0x98, 0x9d, 0xcc, 0x7e, 0xaf, 0xd9, 0x37, 0x4f, 0x6c, 0x29, 0x69, 0x7c, 0x74,
0x74, 0x9e, 0xf1, 0x6d
},
.ciphertext = (unsigned char []){
0x66, 0x25, 0x81, 0x14, 0x19, 0xbd, 0xee, 0x71, 0x53, 0x5f, 0x59, 0x7f, 0x7c,
0x22, 0x8b, 0xaf, 0xd8, 0x90, 0xfd, 0x69, 0xb8, 0x05, 0xa6, 0x99, 0xed, 0x58,
0x11, 0x6a, 0x82, 0xbd, 0xb2, 0x51, 0xab, 0xea, 0x7a, 0x4e, 0xf8, 0x79, 0xa9,
0x6f, 0xce, 0x8e, 0xe4, 0x95, 0x18, 0xb9, 0x87, 0x7a, 0x3a, 0x1e, 0x3c, 0xf3,
0x46, 0xd3, 0xcd, 0x73, 0x73, 0x89, 0x36, 0xd1, 0xcb, 0x6f, 0xff, 0x4b, 0x23,
0x53, 0xc8, 0xca, 0x50, 0x0a, 0x26, 0x68, 0x98, 0x13, 0xad, 0x2f, 0x67, 0x77,
0x4e, 0x23, 0x43, 0xf3, 0xe4, 0x83, 0x02, 0x59, 0x09, 0x4d, 0x3b, 0x34, 0x2e,
0x00, 0xfa, 0xab, 0xeb, 0xa5, 0xb8, 0xa8, 0x93, 0x10, 0x8a, 0x39, 0x0c, 0x64,
0x98, 0x36, 0xdd, 0xd5, 0xd1, 0x24, 0x89, 0xb2, 0xdd, 0x59, 0x1c, 0xa2, 0x53,
0x61, 0x03, 0x2e, 0x2d, 0xa1, 0x20, 0x7f, 0x79, 0x3a, 0x1e, 0x69, 0x51, 0x30,
0x02, 0xa9, 0x0c, 0xcc, 0x03, 0x6b, 0xb6, 0x3e, 0x9c, 0x10, 0xbe, 0x87, 0xdf,
0x2d, 0xef, 0x96, 0x0c, 0xd7, 0xa1, 0xb1, 0x62, 0x1e, 0x31, 0x17, 0x35, 0xd7,
0xae, 0xe4, 0x41, 0x9f
},
},
{
.keylen = AES192_KEYLEN,
.len = 160,
.lcfb = 16,
.key = (unsigned char []){
0xdc, 0xc8, 0x70, 0x21, 0x42, 0xb2, 0x9e, 0x15, 0x29, 0xf2, 0x3c, 0x5c, 0x37,
0x66, 0x46, 0x4f, 0x6b, 0xe0, 0xd2, 0xfb, 0xb1, 0x6e, 0x46, 0x82
},
.iv = {
0xe0, 0x81, 0x87, 0x69, 0xd7, 0x7f, 0x23, 0x15, 0x92, 0x4c, 0xf8, 0x1a, 0x36,
0x91, 0xe2, 0x75
},
.plaintext = (unsigned char []){
0xcb, 0x55, 0x01, 0x11, 0xbf, 0x0a, 0x03, 0xea, 0xf4, 0xf4, 0x9a, 0xf2, 0x14,
0xfc, 0xc0, 0x5a, 0x32, 0x97, 0x2d, 0x7b, 0x4d, 0xed, 0xe3, 0xe9, 0x81, 0x2a,
0x27, 0xef, 0x80, 0xd6, 0x80, 0x18, 0x81, 0x19, 0xea, 0xd5, 0x62, 0x31, 0x3e,
0x40, 0x0f, 0xe0, 0xa0, 0xff, 0xfb, 0xb8, 0x8c, 0x55, 0xa4, 0x2b, 0xd6, 0x81,
0xd5, 0xc9, 0x3a, 0x8a, 0x61, 0xba, 0x90, 0x90, 0x58, 0xe6, 0x2d, 0x99, 0xfa,
0x10, 0x9c, 0xfd, 0x49, 0x93, 0x5b, 0x15, 0x08, 0x62, 0xa8, 0xaa, 0xb2, 0xc3,
0x01, 0xb9, 0xb0, 0xa9, 0x15, 0x7c, 0x83, 0x84, 0x91, 0xcd, 0x73, 0x7a, 0xf4,
0x38, 0xcb, 0x66, 0xb1, 0xf2, 0x04, 0x20, 0x20, 0x0d, 0xbc, 0x56, 0xaa, 0x66,
0x55, 0x2c, 0xe4, 0xbe, 0x04, 0xca, 0xce, 0x4a, 0xce, 0x5b, 0xfb, 0xc6, 0x17,
0xe3, 0xb2, 0x7f, 0x40, 0xec, 0x6d, 0xbc, 0x85, 0xa4, 0x2b, 0x41, 0x0d, 0xcb,
0x7e, 0xa0, 0xb7, 0x8d, 0x47, 0x22, 0x97, 0xd9, 0xb9, 0x88, 0x75, 0xd6, 0x36,
0xb8, 0xef, 0x08, 0xc2, 0x54, 0xec, 0x9b, 0xd0, 0x5b, 0xfd, 0xa0, 0x1b, 0xb3,
0x8e, 0x8b, 0xeb, 0x6a
},
.ciphertext = (unsigned char []){
0x1a, 0x22, 0xb1, 0xbf, 0x57, 0x75, 0xd4, 0x3c, 0x66, 0xf1, 0xa7, 0x30, 0x84,
0xee, 0xef, 0xdb, 0x3e, 0xd2, 0x4c, 0xeb, 0xd9, 0xe2, 0xbe, 0xf2, 0xf0, 0x58,
0x67, 0x16, 0x5f, 0xb5, 0x93, 0x0b, 0x60, 0x58, 0xf5, 0x3e, 0xf4, 0x50, 0x33,
0x53, 0x85, 0x6f, 0xa6, 0xd2, 0xc9, 0x9f, 0x5b, 0x1d, 0xe9, 0x79, 0x5d, 0xa6,
0xe3, 0x14, 0x36, 0x5e, 0x2d, 0x1b, 0xb3, 0x71, 0x9b, 0x23, 0xe8, 0x30, 0x82,
0x3b, 0x74, 0x4e, 0x1e, 0xc4, 0x06, 0x50, 0x31, 0x83, 0x20, 0x3f, 0xed, 0xf4,
0x1b, 0xa0, 0x14, 0xe1, 0x6c, 0xa6, 0x5e, 0x34, 0x25, 0xa5, 0x1b, 0x0a, 0xbf,
0xca, 0x19, 0x08, 0x16, 0x0a, 0xc8, 0xf2, 0xb5, 0x58, 0x9c, 0x79, 0x54, 0x1b,
0xb3, 0x55, 0x9f, 0xdf, 0xb8, 0x94, 0x39, 0x4a, 0x07, 0x32, 0x01, 0x52, 0x11,
0xe9, 0x94, 0xae, 0x02, 0x4a, 0x13, 0x8a, 0xa2, 0x0d, 0x26, 0x7f, 0x79, 0xa6,
0x40, 0xc2, 0x37, 0x19, 0x25, 0x9c, 0x53, 0x0e, 0xaa, 0x1a, 0xf1, 0x28, 0xbc,
0x05, 0x09, 0x93, 0xa4, 0x14, 0xc6, 0xdc, 0x89, 0x61, 0x2c, 0x06, 0x37, 0x1a,
0xfe, 0xda, 0x1f, 0x79
},
},
{
.keylen = AES256_KEYLEN,
.len = 160,
.lcfb = 16,
.key = (unsigned char []){
0x4d, 0x3e, 0x4c, 0xec, 0x63, 0xed, 0xaf, 0xbb, 0x4d, 0x60, 0x00, 0x07, 0xe9,
0x51, 0x24, 0xf5, 0x54, 0xb3, 0x52, 0xad, 0xa4, 0x96, 0x6a, 0x60, 0xda, 0x4c,
0x89, 0x89, 0x12, 0xca, 0xda, 0x73
},
.iv = {
0x05, 0xae, 0xf5, 0xcc, 0xf4, 0x62, 0x98, 0xe0, 0xfe, 0xb5, 0x8d, 0x77, 0x12,
0x2b, 0x58, 0xd9
},
.plaintext = (unsigned char []){
0x95, 0xd6, 0xb8, 0x3e, 0x1c, 0x10, 0xc7, 0x21, 0xe0, 0xf0, 0xc3, 0x59, 0x07,
0xb3, 0xa4, 0xe3, 0x5c, 0x27, 0x94, 0xa6, 0xa8, 0x23, 0x48, 0x74, 0x44, 0x0b,
0xe7, 0xa7, 0x95, 0xdc, 0x8e, 0x2f, 0x7e, 0xc5, 0xcf, 0x73, 0x9d, 0x0b, 0xb1,
0x3b, 0x1f, 0xc5, 0x1c, 0xf5, 0xd4, 0xd2, 0x7d, 0x2e, 0xd4, 0xb9, 0x3c, 0x11,
0x89, 0x3c, 0x7b, 0x9a, 0x64, 0x9b, 0x22, 0xcb, 0xbc, 0x96, 0xa8, 0xcd, 0x58,
0x47, 0xd1, 0x35, 0xc4, 0x3d, 0x1a, 0x11, 0x85, 0x58, 0x11, 0xb8, 0x2c, 0xff,
0xbd, 0x22, 0x87, 0xe6, 0xc5, 0x5f, 0x45, 0xd1, 0x24, 0xd4, 0x7d, 0x54, 0x92,
0x18, 0xc1, 0xea, 0x00, 0x49, 0x28, 0x1d, 0xd5, 0x39, 0xa6, 0x0c, 0xdb, 0xb8,
0x05, 0x49, 0xdb, 0x3a, 0xf3, 0xb9, 0xf8, 0xd4, 0xca, 0x12, 0x7e, 0xfc, 0xb5,
0xcd, 0xe7, 0xec, 0xc9, 0x8e, 0x00, 0x8f, 0x1e, 0xde, 0xff, 0x69, 0x80, 0xf1,
0x72, 0x65, 0x28, 0x06, 0xff, 0x93, 0x95, 0xaf, 0x7a, 0x62, 0xf8, 0x8a, 0xba,
0xaf, 0x89, 0x74, 0xeb, 0xef, 0x1a, 0x02, 0xd7, 0x8e, 0x4b, 0xd5, 0x21, 0x49,
0xfa, 0x1e, 0xe1, 0x83
},
.ciphertext = (unsigned char []){
0xfe, 0xdc, 0x27, 0x08, 0xce, 0x2e, 0x24, 0x71, 0xab, 0x8e, 0x66, 0xc6, 0x9a,
0x34, 0x51, 0xa9, 0x43, 0x80, 0xda, 0x0e, 0x5e, 0x99, 0x98, 0xfa, 0x68, 0x54,
0x1d, 0x89, 0x9a, 0x57, 0x86, 0x36, 0x1b, 0x7e, 0x51, 0x57, 0x75, 0x7d, 0x6f,
0xe7, 0x46, 0xc7, 0x9a, 0x88, 0x38, 0xaf, 0xe9, 0xc8, 0x32, 0xcd, 0xa2, 0xa4,
0xd0, 0xa4, 0x4f, 0x4b, 0x11, 0x81, 0x14, 0x24, 0x50, 0xa6, 0x3f, 0x11, 0x76,
0xa8, 0x21, 0xf6, 0x6d, 0x16, 0x1d, 0x75, 0xd8, 0x5b, 0xfe, 0xfc, 0x01, 0xe6,
0x8d, 0x02, 0x12, 0x88, 0x64, 0x8d, 0x89, 0x1d, 0xfd, 0xf8, 0xe6, 0x6e, 0x0e,
0xf3, 0xa6, 0x56, 0x19, 0xcb, 0x75, 0x24, 0x3e, 0xec, 0xa0, 0x41, 0x55, 0xa4,
0xc9, 0x13, 0x39, 0x29, 0xde, 0x20, 0x66, 0xde, 0x4c, 0x77, 0xc7, 0xd2, 0x6f,
0x4c, 0xc9, 0x89, 0x4d, 0xe2, 0xb4, 0x00, 0x85, 0xae, 0x3b, 0xeb, 0x82, 0xb9,
0x52, 0x41, 0xf4, 0x46, 0x3f, 0xfa, 0x81, 0xb5, 0xf4, 0x18, 0xb7, 0xa7, 0x9c,
0xe4, 0x46, 0x63, 0x74, 0x7a, 0x6c, 0x78, 0xdc, 0x87, 0xb0, 0xa4, 0xae, 0x52,
0xd3, 0xf5, 0xce, 0xf9
},
},
};

const struct aes_ofb_tv AES_OFB_TV[] = {
{
.keylen = AES128_KEYLEN,
.len = 160,
.key = (unsigned char []){
0xe3, 0x0b, 0x4c, 0x87, 0x4c, 0x4c, 0x4f, 0x6e, 0x0c, 0xf1, 0xf8, 0xef, 0x58,
0xe5, 0xd3, 0x75
},
.iv = {
0x7e, 0x26, 0xf0, 0x7f, 0x80, 0x24, 0x34, 0x3c, 0xec, 0x35, 0x40, 0x9e, 0x71,
0xe0, 0xcd, 0x8c
},
.plaintext = (unsigned char []){
0x8c, 0xec, 0xa4, 0xdc, 0x34, 0x6c, 0xfd, 0x6b, 0x15, 0x77, 0x4e, 0x08, 0x2d,
0xb1, 0xa8, 0x94, 0x97, 0xb7, 0xd8, 0x5d, 0x6b, 0x5b, 0x71, 0x02, 0xe7, 0x74,
0x17, 0xf7, 0xa2, 0x43, 0xfa, 0xfe, 0x17, 0x11, 0x8b, 0x7a, 0x3b, 0xb4, 0x9d,
0x16, 0x57, 0xcf, 0x61, 0xb8, 0x66, 0xda, 0x39, 0x5a, 0x5b, 0x3f, 0x34, 0x91,
0x83, 0xa5, 0x3d, 0xfa, 0x11, 0xfc, 0x0a, 0xc0, 0x53, 0xbd, 0xdf, 0xf4, 0x9d,
0xd4, 0x72, 0xee, 0x55, 0xf5, 0xe4, 0x3a, 0x2f, 0x8b, 0xc7, 0x85, 0xe2, 0xbc,
0x42, 0x03, 0x00, 0x69, 0x49, 0x19, 0xff, 0x7b, 0xb4, 0x3f, 0xeb, 0x75, 0xa9,
0xca, 0xc4, 0x4e, 0xce, 0x96, 0xf6, 0x79, 0xe6, 0x18, 0xdb, 0x5d, 0x74, 0x33,
0xaf, 0x12, 0xdc, 0xc7, 0xe0, 0x96, 0x3f, 0xf1, 0x0b, 0x45, 0xd8, 0x35, 0xf9,
0xa8, 0xf4, 0x26, 0x27, 0xe7, 0xf3, 0xfd, 0x50, 0x38, 0x93, 0x26, 0x85, 0x96,
0x5a, 0xd0, 0xe1, 0x83, 0xf5, 0x95, 0x5e, 0x67, 0x1f, 0xc2, 0xb8, 0x78, 0xdd,
0x51, 0x05, 0x1e, 0xed, 0xaf, 0x85, 0x31, 0x0d, 0x1e, 0x4e, 0x8f, 0x75, 0xf2,
0xde, 0xcf, 0x36, 0xc7
},
.ciphertext = (unsigned char []){
0x5d, 0xca, 0xa1, 0x73, 0xed, 0xe1, 0x4f, 0xd2, 0xd6, 0x58, 0x97, 0x39, 0x26,
0x16, 0x8f, 0xf3, 0x4f, 0xd6, 0xdf, 0x9b, 0xce, 0x32, 0x80, 0xd4, 0x0b, 0x00,
0xc4, 0x3b, 0x80, 0xe2, 0x97, 0x9a, 0x1e, 0x19, 0x04, 0x5f, 0xec, 0x9a, 0xfb,
0x4c, 0xf2, 0x64, 0x51, 0x6f, 0x55, 0x10, 0x08, 0x55, 0xc3, 0xaa, 0xd1, 0x7b,
0x11, 0xbf, 0xcf, 0x05, 0x23, 0xb7, 0x9e, 0xb2, 0x0d, 0x65, 0x94, 0x10, 0x77,
0xdd, 0x46, 0xec, 0x46, 0x86, 0x4e, 0x0d, 0x79, 0x70, 0x4c, 0x22, 0x50, 0xe7,
0x2b, 0xf8, 0xb4, 0x48, 0xa6, 0xf0, 0xd3, 0x13, 0x0a, 0xb1, 0x0b, 0x42, 0x3d,
0x1a, 0x09, 0xd9, 0xff, 0x7a, 0x32, 0xbf, 0x70, 0x04, 0x41, 0xcc, 0xd2, 0x7d,
0x32, 0x23, 0x91, 0x38, 0x60, 0xc2, 0x80, 0x44, 0xea, 0x57, 0x66, 0xe4, 0x5a,
0x55, 0xb9, 0x3f, 0x89, 0x48, 0xa9, 0x59, 0xbd, 0x66, 0x61, 0x42, 0x15, 0x66,
0x89, 0x8e, 0x27, 0x95, 0x0f, 0x04, 0xe7, 0x26, 0x27, 0x9b, 0xcb, 0xc9, 0x90,
0xa2, 0x2c, 0x80, 0x19, 0x3e, 0xf0, 0xae, 0x65, 0x19, 0x66, 0x71, 0xeb, 0x59,
0x71, 0x32, 0x40, 0xcf
},
},
{
.keylen = AES192_KEYLEN,
.len = 160,
.key = (unsigned char []){
0x75, 0x3c, 0x1f, 0x7f, 0x39, 0xaf, 0xd2, 0x86, 0xff, 0x05, 0x13, 0x39, 0xc4,
0x5d, 0x57, 0xc8, 0x48, 0x85, 0x03, 0x93, 0xb4, 0x11, 0x2f, 0xe5
},
.iv = {
0xd3, 0xee, 0x0d, 0x8d, 0xe5, 0x08, 0x0e, 0xda, 0x4c, 0x26, 0xf5, 0xa3, 0xb9,
0xca, 0x8a, 0x9f
},
.plaintext = (unsigned char []){
0xad, 0x3c, 0xcd, 0xa2, 0x64, 0x34, 0x31, 0x30, 0xbf, 0x1d, 0xb3, 0x70, 0x3e,
0x27, 0x12, 0x71, 0x76, 0xfb, 0xaa, 0x7b, 0x6a, 0x5d, 0xa2, 0x71, 0x87, 0x83,
0xba, 0xf9, 0xf2, 0x8f, 0xcd, 0xd3, 0xed, 0x9c, 0xd3, 0x1a, 0xdc, 0xc7, 0x94,
0x27, 0xfe, 0x4d, 0xf0, 0x3f, 0x16, 0x72, 0xa5, 0xe5, 0x5c, 0xab, 0x0d, 0xb0,
0xf4, 0xd4, 0x34, 0xd8, 0x60, 0x34, 0x0d, 0x2f, 0xa0, 0x5b, 0xfe, 0xb0, 0x7e,
0x92, 0x41, 0x57, 0x06, 0x4a, 0x24, 0xd0, 0xf1, 0x0e, 0x32, 0x93, 0xf7, 0x8a,
0x26, 0x76, 0xe3, 0xc5, 0x37, 0x34, 0xf2, 0x2d, 0x4e, 0xa3, 0x3e, 0x89, 0x38,
0x4b, 0xd1, 0x7f, 0x4a, 0x0f, 0x59, 0x35, 0x41, 0x79, 0xce, 0x48, 0xa7, 0xd1,
0xc1, 0xba, 0x35, 0xe7, 0xf7, 0x77, 0x35, 0xf5, 0x86, 0x80, 0xf0, 0xe8, 0x9b,
0xf9, 0x24, 0x2a, 0x4f, 0x08, 0x73, 0x22, 0xd9, 0x9e, 0x50, 0x73, 0x36, 0xa8,
0xe9, 0x03, 0x7b, 0x6f, 0x1b, 0xfb, 0xe4, 0x56, 0x14, 0xab, 0xeb, 0x2f, 0x71,
0x51, 0x6b, 0x94, 0xca, 0xf6, 0x18, 0x85, 0x1d, 0xdc, 0xfa, 0xc7, 0x42, 0x9a,
0x21, 0x77, 0xbe, 0x40
},
.ciphertext = (unsigned char []){
0xa9, 0x14, 0x09, 0xfa, 0x69, 0x4a, 0x4a, 0xd3, 0x4a, 0x9c, 0xb1, 0xc1, 0x53,
0x4a, 0x94, 0x97, 0x9a, 0x5b, 0xeb, 0x39, 0x00, 0x83, 0xa6, 0xa6, 0x1b, 0x4a,
0x44, 0x36, 0x74, 0x6b, 0xa1, 0x20, 0xf3, 0x7a, 0x0f, 0x3e, 0x97, 0xca, 0xa9,
0x15, 0x6a, 0x8e, 0xe4, 0x10, 0xb5, 0x3e, 0x67, 0x0c, 0x70, 0x3d, 0x1d, 0x19,
0xfc, 0xb8, 0x88, 0x7f, 0x15, 0xd1, 0x58, 0x00, 0x02, 0xfd, 0x3c, 0x5d, 0x0e,
0xaa, 0xf8, 0x1c, 0x3c, 0x26, 0xb8, 0x87, 0x37, 0xbc, 0xa3, 0xf8, 0x88, 0x20,
0xb4, 0x54, 0x0b, 0x0d, 0xc5, 0xd6, 0xa4, 0x2c, 0xed, 0x0e, 0x2e, 0x38, 0x0c,
0x29, 0xb4, 0x60, 0xa4, 0x72, 0xe4, 0xec, 0xeb, 0x19, 0xc0, 0x24, 0x1e, 0x33,
0x97, 0x6d, 0x17, 0x03, 0x34, 0xb6, 0x22, 0x78, 0x55, 0x12, 0x0d, 0xf6, 0x5f,
0xa6, 0x7e, 0x5a, 0x4f, 0xc6, 0x89, 0x38, 0xf4, 0x5b, 0x82, 0x66, 0x5c, 0x88,
0x10, 0xbf, 0x8c, 0x8d, 0x83, 0x21, 0x73, 0xeb, 0x68, 0xb8, 0x21, 0xc9, 0x5a,
0x7f, 0x35, 0x96, 0x1b, 0x03, 0x94, 0xf0, 0xe3, 0x6b, 0x61, 0xc4, 0xf7, 0xb5,
0x29, 0xb7, 0x7e, 0x42
},
},
{
.keylen = AES256_KEYLEN,
.len = 160,
.key = (unsigned char []){
0x98, 0xa9, 0x97, 0x1e, 0x86, 0x80, 0x6c, 0xcc, 0x34, 0x95, 0x11, 0x6f, 0xd0,
0x6d, 0xc9, 0xd1, 0x52, 0x2f, 0xe8, 0x80, 0x60, 0xfd, 0xdd, 0xc3, 0x6e, 0x84,
0x6f, 0xd3, 0x29, 0xd2, 0x47, 0x48
},
.iv = {
0x99, 0x83, 0xce, 0x04, 0x8f, 0x19, 0xef, 0x40, 0x43, 0x05, 0x4c, 0x03, 0xaa,
0x01, 0x0b, 0xee
},
.plaintext = (unsigned char []){
0x00, 0xca, 0xa2, 0x33, 0x19, 0x8f, 0x51, 0xbb, 0xf5, 0x93, 0x40, 0x4f, 0x59,
0x82, 0x69, 0x97, 0xb4, 0xea, 0x38, 0x73, 0x85, 0xcf, 0x74, 0x4c, 0x93, 0xcf,
0xa0, 0x0e, 0x70, 0x2e, 0x8f, 0x16, 0xff, 0x5a, 0xa7, 0xe1, 0x7a, 0x9a, 0x60,
0x20, 0xdf, 0x0f, 0x0d, 0xe4, 0xea, 0x6a, 0xbb, 0x38, 0xbc, 0xf1, 0xd7, 0x77,
0x81, 0x0a, 0x83, 0x18, 0xf6, 0x9b, 0x5e, 0x83, 0x05, 0xf6, 0xd7, 0x27, 0xf0,
0x6f, 0x00, 0x8b, 0x4b, 0xec, 0x2d, 0x65, 0xcd, 0x4c, 0x51, 0x6c, 0xa4, 0x9f,
0x62, 0xfb, 0x2f, 0x91, 0x6f, 0x27, 0x3c, 0x45, 0xbb, 0x72, 0x2b, 0xec, 0x78,
0xc3, 0x16, 0xf9, 0x0b, 0x5e, 0xd5, 0xde, 0x6e, 0xf1, 0xd3, 0x66, 0x60, 0x3c,
0xed, 0x30, 0x3c, 0x10, 0xe3, 0x3d, 0xd5, 0xc9, 0x9e, 0xb0, 0xf9, 0x94, 0xdb,
0x5a, 0x78, 0x67, 0xda, 0x9b, 0x53, 0x0f, 0xc4, 0xd0, 0xb9, 0xce, 0x22, 0x4c,
0x6e, 0xab, 0x78, 0x10, 0x35, 0x9c, 0x97, 0x33, 0xcf, 0x93, 0x3c, 0x57, 0x36,
0x11, 0xd3, 0x1f, 0xcd, 0xf3, 0xf1, 0xdb, 0x87, 0xcf, 0xd1, 0x7b, 0xe7, 0xf4,
0xa4, 0x70, 0xa0, 0xb4
},
.ciphertext = (unsigned char []){
0x4b, 0x07, 0x67, 0xfe, 0xc0, 0xd4, 0xbd, 0x07, 0xc7, 0x9c, 0x0f, 0x56, 0x52,
0xda, 0xfa, 0xdf, 0x10, 0xcf, 0xce, 0x89, 0xe3, 0x25, 0x9d, 0xea, 0x94, 0xd3,
0x92, 0x52, 0xc6, 0x40, 0x84, 0x0c, 0x28, 0xab, 0xcb, 0xe0, 0xef, 0xca, 0x53,
0xdc, 0x84, 0xb1, 0xef, 0x55, 0x79, 0xf6, 0xef, 0x28, 0xc2, 0x13, 0xf4, 0x45,
0x22, 0x0b, 0x03, 0x6f, 0xe3, 0x51, 0xd9, 0x3d, 0xca, 0xe5, 0x7e, 0x65, 0x4a,
0xc0, 0x1d, 0x39, 0xf8, 0x72, 0x13, 0x72, 0x3f, 0x0a, 0x46, 0x2d, 0x55, 0x36,
0xb8, 0x33, 0x6d, 0xab, 0x5c, 0x7d, 0x2f, 0xc7, 0x28, 0xf8, 0x65, 0x75, 0x6b,
0x85, 0xf7, 0x52, 0x61, 0x44, 0x19, 0x0e, 0x04, 0x12, 0xc3, 0x14, 0x26, 0x50,
0x61, 0x6d, 0xbd, 0xe7, 0xcd, 0xe1, 0x7e, 0x88, 0x7a, 0x60, 0xad, 0x39, 0xf2,
0xa1, 0x33, 0x0d, 0x82, 0x09, 0xf1, 0x32, 0x33, 0xce, 0x54, 0x31, 0xfd, 0x5c,
0x29, 0x72, 0x38, 0xf8, 0xb3, 0xff, 0x53, 0xa3, 0xfb, 0x89, 0xc8, 0x41, 0x68,
0xb0, 0x4f, 0xfa, 0x8f, 0x7f, 0x53, 0xe1, 0x4c, 0x36, 0xa2, 0xd3, 0x12, 0x4d,
0x68, 0xa2, 0x7f, 0xb5
},
},
};

const struct des3_ecb_tv DES3_ECB_TV[] = {
{
.len = 72,
.key = {
0xbc, 0x51, 0xa2, 0x29, 0x1a, 0xf4, 0xbc, 0x10, 0x31, 0x57, 0x26, 0x9e, 0x7a,
0x4c, 0x15, 0xb5, 0xa1, 0x31, 0x97, 0xd9, 0xad, 0x97, 0x0d, 0xb9
},
.plaintext = (unsigned char []){
0x2e, 0xdd, 0x7f, 0x0f, 0xf0, 0x86, 0xab, 0xd0, 0xa1, 0xb1, 0x21, 0xd0, 0x53,
0xf7, 0xfb, 0x09, 0x59, 0xba, 0x89, 0xfd, 0xcf, 0xb9, 0x7e, 0xb4, 0x22, 0x81,
0xfd, 0x18, 0x85, 0xa9, 0x5c, 0xf3, 0xd8, 0x4e, 0xc0, 0xb0, 0xfe, 0x2e, 0xbe,
0x55, 0x50, 0xad, 0xa9, 0x49, 0xa1, 0x52, 0x47, 0x66, 0x03, 0xdf, 0x9b, 0xd0,
0x9a, 0x48, 0xd0, 0x01, 0xc9, 0xb9, 0xdc, 0x20, 0xd0, 0x82, 0xd1, 0xe7, 0xc2,
0x73, 0x85, 0xf0, 0xcc, 0xcf, 0xfe, 0xc1
},
.ciphertext = (unsigned char []){
0x54, 0x15, 0xca, 0x2d, 0x2c, 0xab, 0xe7, 0x75, 0xb8, 0x17, 0xb2, 0x2b, 0x8f,
0xe4, 0x97, 0xde, 0x96, 0xb3, 0xc0, 0xdb, 0x73, 0xe0, 0x2e, 0xf3, 0x44, 0x62,
0x09, 0x7f, 0x90, 0xf3, 0x34, 0xa1, 0x7d, 0x72, 0x25, 0xcc, 0x44, 0x85, 0xf4,
0x86, 0xe3, 0x82, 0xb1, 0x6c, 0x12, 0x90, 0xfb, 0x77, 0x77, 0x09, 0xce, 0x7f,
0x1d, 0x23, 0x9d, 0xfd, 0x23, 0xd0, 0xa1, 0x15, 0xb1, 0xe2, 0x70, 0xd4, 0x30,
0x73, 0xd1, 0xb6, 0xbb, 0x84, 0xcf, 0x6b
},
},
{
.len = 80,
.key = {
0xb5, 0xcd, 0x0e, 0x1f, 0x1f, 0x86, 0x45, 0x8f, 0x34, 0x2f, 0x26, 0x83, 0xcd,
0x3b, 0xbf, 0xb9, 0x10, 0x85, 0xe3, 0x8f, 0x67, 0xc2, 0xb9, 0xa1
},
.plaintext = (unsigned char []){
0x94, 0x4c, 0xb9, 0x7a, 0x22, 0x68, 0x39, 0x36, 0x6d, 0xfe, 0xaf, 0x46, 0x80,
0xb4, 0xad, 0xad, 0x05, 0x37, 0x2c, 0x6f, 0x9c, 0xa4, 0x60, 0xf6, 0x87, 0xa9,
0x4c, 0xf2, 0x86, 0x48, 0x32, 0xff, 0x5d, 0x57, 0x2e, 0xf1, 0xcf, 0x16, 0x41,
0xa7, 0xe0, 0x08, 0x2d, 0x6f, 0x48, 0x28, 0x5a, 0xf7, 0xe4, 0x1f, 0x1c, 0x6b,
0x10, 0x0e, 0x67, 0xe1, 0x37, 0x93, 0x66, 0x0c, 0x34, 0x38, 0x8f, 0xbe, 0xf0,
0x43, 0xf0, 0xbb, 0x87, 0xd7, 0xaa, 0x8a, 0xb2, 0xcc, 0x98, 0x77, 0x90, 0xc1,
0x51, 0xcb
},
.ciphertext = (unsigned char []){
0xbd, 0x79, 0x1a, 0xa9, 0xd3, 0x42, 0xdc, 0x4b, 0x7d, 0xda, 0x97, 0xcf, 0x0a,
0x1f, 0x41, 0xab, 0x30, 0xc3, 0x52, 0xbe, 0x5c, 0x4f, 0xc1, 0x7b, 0x47, 0x67,
0x30, 0x4a, 0xca, 0x4f, 0x64, 0x2b, 0xf3, 0xae, 0xf7, 0x9d, 0x58, 0x92, 0x51,
0xbe, 0xab, 0xb5, 0xd4, 0xb3, 0x8f, 0xff, 0x57, 0x4a, 0xaa, 0x4b, 0xd6, 0xde,
0xfc, 0x6e, 0x67, 0x7c, 0x6f, 0x75, 0x70, 0x43, 0xaf, 0x2f, 0x18, 0x11, 0x3b,
0xec, 0xfb, 0x5c, 0xbd, 0x83, 0x22, 0x5d, 0x97, 0x9a, 0x32, 0x74, 0xbe, 0x21,
0xe3, 0xc8
},
},
};

const struct des3_cbc_tv DES3_CBC_TV[] = {
{
.len = 72,
.key = {
0x2a, 0xe6, 0xae, 0x5d, 0x75, 0x34, 0x8f, 0x9d, 0x34, 0xc4, 0xfd, 0x45, 0x85,
0xb6, 0xba, 0x8c, 0x9d, 0xef, 0x6e, 0xbf, 0xc7, 0x7f, 0xe0, 0x46
},
.iv = {
0xa5, 0x2b, 0x66, 0xc3, 0x34, 0x38, 0x59, 0xe0
},
.plaintext = (unsigned char []){
0x24, 0x19, 0x26, 0x42, 0xf7, 0xc7, 0x0e, 0x7d, 0x75, 0x69, 0xde, 0x8b, 0x86,
0x20, 0x4a, 0x10, 0xa0, 0x76, 0xe3, 0xf3, 0x79, 0xbc, 0x7f, 0x8a, 0x52, 0x25,
0xec, 0x4d, 0x76, 0xc0, 0x8a, 0xca, 0x40, 0xcd, 0xe8, 0x86, 0xf5, 0xd9, 0x9b,
0xc0, 0xb6, 0x65, 0xa1, 0x79, 0x56, 0x37, 0x22, 0x19, 0xcd, 0x52, 0xcb, 0x73,
0xec, 0x48, 0x62, 0xdf, 0x24, 0x2f, 0xb7, 0x65, 0xb1, 0x85, 0x3b, 0xc6, 0x25,
0x82, 0x41, 0xe4, 0x1a, 0x1a, 0xd7, 0x15
},
.ciphertext = (unsigned char []){
0x95, 0x0e, 0x26, 0xc3, 0xf2, 0x46, 0xd7, 0xed, 0xff, 0xbd, 0xd3, 0x0e, 0x9a,
0x2c, 0x08, 0x2b, 0x48, 0x53, 0x3f, 0xbf, 0x26, 0x29, 0xa9, 0x00, 0x23, 0x35,
0xbb, 0x63, 0x6d, 0x3c, 0x3c, 0x62, 0x29, 0x1c, 0x9d, 0x7d, 0x67, 0xf8, 0x88,
0x7f, 0x55, 0x95, 0xbb, 0x02, 0xb2, 0x04, 0x17, 0x35, 0x86, 0x82, 0xf4, 0xdd,
0x11, 0x9c, 0xb8, 0x9a, 0x01, 0x97, 0xa8, 0x6c, 0xe8, 0x0e, 0xfe, 0x75, 0xec,
0xdd, 0x3b, 0x01, 0xc1, 0x3e, 0xbe, 0xc3
},
},
{
.len = 80,
.key = {
0xb6, 0x04, 0xec, 0x25, 0x5e, 0x04, 0x54, 0x0b, 0x23, 0x0b, 0x4a, 0xce, 0xf1,
0x38, 0x40, 0xc1, 0xe9, 0x58, 0xb9, 0xf2, 0xe9, 0x13, 0x9d, 0x57
},
.iv = {
0x58, 0x6c, 0x47, 0x1f, 0x4e, 0xcd, 0xa1, 0xde
},
.plaintext = (unsigned char []){
0x90, 0x0c, 0x63, 0xa0, 0x8f, 0xd1, 0x42, 0xa9, 0xc1, 0x62, 0x63, 0x93, 0x91,
0xe0, 0xe2, 0xb6, 0x46, 0x97, 0xb7, 0xa5, 0x16, 0xd6, 0x9c, 0xd7, 0x77, 0x96,
0xd4, 0x90, 0x14, 0x87, 0x73, 0xfc, 0x98, 0x4f, 0x21, 0xa0, 0xc2, 0x86, 0xd7,
0x3d, 0x86, 0xd8, 0xfd, 0x80, 0x11, 0xaf, 0x8f, 0x81, 0x74, 0xe0, 0x3e, 0xad,
0x4c, 0xaf, 0xb9, 0x64, 0x12, 0x46, 0x61, 0x5a, 0xad, 0x9b, 0xa8, 0xb0, 0x2f,
0xde, 0x3e, 0xdf, 0xf3, 0x55, 0x26, 0xd3, 0xba, 0x21, 0x46, 0x5d, 0x2c, 0x99,
0xf8, 0x7c
},
.ciphertext = (unsigned char []){
0xf7, 0xc0, 0x8a, 0x2c, 0xf5, 0x80, 0x5b, 0xc1, 0x5c, 0x15, 0x7c, 0xff, 0xfe,
0x5b, 0x67, 0x49, 0x42, 0x6c, 0xde, 0x28, 0x19, 0xc3, 0x7e, 0x93, 0x34, 0xa2,
0x97, 0x6e, 0x75, 0xb0, 0x3b, 0x50, 0xa3, 0xe7, 0x95, 0x89, 0x61, 0xad, 0x7c,
0x64, 0x61, 0x41, 0x48, 0xa9, 0xd5, 0x59, 0xe5, 0x66, 0x26, 0x0a, 0x26, 0xef,
0xe9, 0xe7, 0x5b, 0x24, 0x0f, 0x6d, 0x25, 0xf2, 0x5b, 0x9f, 0x89, 0x1d, 0x1a,
0xed, 0x44, 0x3c, 0x0e, 0x5a, 0x5e, 0x05, 0xab, 0xde, 0x02, 0xca, 0x86, 0x09,
0x8d, 0x44
},
},
};

const struct des3_cfb_tv DES3_CFB_TV[] = {
{
.len = 10,
.lcfb = 1,
.key = {
0x29, 0x0d, 0xba, 0x49, 0xc1, 0x2a, 0xe9, 0x67, 0x6e, 0xf7, 0x19, 0xef, 0x6b,
0xec, 0xad, 0x0d, 0x04, 0x79, 0x80, 0x6e, 0x02, 0xe0, 0x70, 0x6e
},
.iv = {
0x2d, 0x5f, 0x4b, 0xed, 0x09, 0x21, 0x47, 0x79
},
.plaintext = (unsigned char []){
0xe8, 0xcb, 0xdd, 0x0d, 0x7e, 0x8a, 0xc9, 0x64, 0xae, 0x2d
},
.ciphertext = (unsigned char []){
0xec, 0xb1, 0x2f, 0xfe, 0xa8, 0x97, 0x76, 0x87, 0x37, 0xaf
},
},
{
.len = 80,
.lcfb = 8,
.key = {
0x26, 0xfe, 0x46, 0xf7, 0x75, 0xc2, 0x34, 0xb5, 0x29, 0x9e, 0xa1, 0xda, 0x16,
0x8c, 0xe6, 0x0d, 0xa1, 0xe5, 0x51, 0x57, 0x98, 0x46, 0x8f, 0xb9
},
.iv = {
0x62, 0x6c, 0x2d, 0xfb, 0x4e, 0x4e, 0xd3, 0x66
},
.plaintext = (unsigned char []){
0x0d, 0xc2, 0xc7, 0x5e, 0x4d, 0xdf, 0x8b, 0xbc, 0x36, 0x13, 0xfa, 0x64, 0x42,
0xd3, 0x9c, 0x2f, 0xa0, 0x31, 0x56, 0x1a, 0x5e, 0x98, 0xec, 0xbf, 0x36, 0x33,
0xc4, 0xa9, 0x4e, 0x96, 0xb9, 0x6e, 0x84, 0xfd, 0x42, 0x07, 0xe3, 0x84, 0x79,
0x8b, 0x1f, 0x95, 0x3d, 0xc2, 0x63, 0xc5, 0x1f, 0x86, 0xe6, 0x61, 0x96, 0xd5,
0x81, 0x7c, 0x3f, 0x04, 0xb8, 0x31, 0x92, 0xe2, 0x62, 0xe1, 0x6e, 0x5b, 0xe9,
0x14, 0x3b, 0x81, 0x87, 0x5b, 0x9d, 0x37, 0xcc, 0xe9, 0xbf, 0xb0, 0xee, 0xe0,
0x26, 0xfc
},
.ciphertext = (unsigned char []){
0x04, 0xbf, 0x26, 0x5b, 0xf3, 0x32, 0x51, 0x90, 0x6f, 0xb3, 0x31, 0x05, 0x10,
0x09, 0xc4, 0xcc, 0x76, 0xe3, 0x4d, 0x30, 0x8e, 0x0e, 0xb2, 0x8e, 0x12, 0xeb,
0x22, 0x67, 0x4f, 0x21, 0xfa, 0xb2, 0x2b, 0x8f, 0xea, 0x77, 0x66, 0x85, 0x54,
0x99, 0x20, 0x33, 0xf4, 0xf4, 0x9c, 0x61, 0xc6, 0xa7, 0x43, 0x61, 0x24, 0xb3,
0xec, 0x42, 0x2f, 0xc6, 0x5a, 0x47, 0x71, 0x51, 0x06, 0x97, 0xc1, 0xe0, 0x1f,
0x36, 0x66, 0xab, 0xd0, 0x34, 0x7f, 0x68, 0xc1, 0x25, 0xa2, 0x64, 0xe0, 0xd1,
0xc3, 0xd2
},
},
};

const struct des3_ofb_tv DES3_OFB_TV[] = {
{
.len = 72,
.key = {
0xfe, 0x80, 0xea, 0x91, 0xe5, 0x45, 0x7f, 0x75, 0x7f, 0x34, 0x2f, 0xdf, 0x1f,
0xd9, 0xf1, 0xbc, 0x32, 0x29, 0x8a, 0xc8, 0x58, 0x7c, 0x80, 0x54
},
.iv = {
0x07, 0xa1, 0xf8, 0xf1, 0xbb, 0x2e, 0x90, 0xaa
},
.plaintext = (unsigned char []){
0x83, 0x14, 0x3e, 0x9e, 0xc9, 0x91, 0x34, 0xc8, 0x5f, 0xae, 0x01, 0x3b, 0x69,
0x51, 0x47, 0x0c, 0xae, 0x79, 0xad, 0x7c, 0x8b, 0xbd, 0xef, 0xad, 0xc8, 0x18,
0x47, 0x77, 0x57, 0xab, 0xc6, 0x3a, 0x0d, 0x14, 0x1e, 0x64, 0x0b, 0x1f, 0x9d,
0xd4, 0xb0, 0x42, 0x90, 0x4e, 0xbc, 0x7a, 0x30, 0xb7, 0xde, 0x55, 0xca, 0x08,
0x51, 0x20, 0x26, 0xb5, 0x03, 0xfb, 0x29, 0xa9, 0x64, 0xc1, 0xbf, 0x68, 0xef,
0xfd, 0x2b, 0x96, 0x98, 0x1f, 0x40, 0x83
},
.ciphertext = (unsigned char []){
0xe5, 0x69, 0x72, 0x0a, 0xd5, 0xae, 0xab, 0x8a, 0x1a, 0xd1, 0xa6, 0xab, 0xbf,
0x06, 0xd7, 0xe2, 0xf8, 0x03, 0x2f, 0x2f, 0x55, 0x62, 0xee, 0xa0, 0x3b, 0x47,
0xb9, 0x16, 0x95, 0x42, 0xec, 0x4b, 0xc7, 0x77, 0x31, 0x14, 0xe3, 0xfc, 0x52,
0xe4, 0x20, 0x82, 0x4b, 0x92, 0x88, 0x38, 0xa4, 0x5c, 0xf0, 0xec, 0xfd, 0x2b,
0x2d, 0x73, 0xeb, 0x19, 0x85, 0x4c, 0x02, 0x30, 0x64, 0x80, 0x33, 0x74, 0x50,
0xcc, 0x59, 0x24, 0x1f, 0x02, 0x0e, 0x3e
},
},
{
.len = 80,
.key = {
0x75, 0xcd, 0xd3, 0x07, 0x45, 0xfb, 0xbc, 0x31, 0x04, 0x83, 0x54, 0x26, 0x52,
0x9b, 0x45, 0xe5, 0xfd, 0x10, 0x2c, 0x7c, 0x8a, 0x9d, 0x1f, 0x1c
},
.iv = {
0x09, 0x77, 0x3d, 0xa8, 0x62, 0xac, 0x17, 0xee
},
.plaintext = (unsigned char []){
0x57, 0xdd, 0xb8, 0x43, 0xff, 0x45, 0x18, 0x60, 0xef, 0x4a, 0xfc, 0x3e, 0x0e,
0x89, 0x97, 0x89, 0xb3, 0x7d, 0xeb, 0xc9, 0xf9, 0x53, 0x41, 0x5f, 0xb9, 0x68,
0x67, 0x6f, 0xfc, 0x52, 0x8f, 0x8c, 0x8e, 0x25, 0xba, 0x1c, 0xac, 0x21, 0x4d,
0xf4, 0xfd, 0x0c, 0xec, 0xfb, 0x19, 0x89, 0xe5, 0x99, 0x30, 0x92, 0x2f, 0xb6,
0xd0, 0xb7, 0x58, 0x83, 0xa5, 0x23, 0x46, 0x28, 0x47, 0x18, 0xf0, 0x7e, 0x71,
0x21, 0x10, 0x97, 0x74, 0xfe, 0x4e, 0x14, 0x0b, 0x19, 0x83, 0xc0, 0x36, 0xc9,
0xf3, 0x0c
},
.ciphertext = (unsigned char []){
0x59, 0x14, 0x41, 0x8e, 0x5d, 0x3b, 0x71, 0x55, 0x42, 0x18, 0x6f, 0xd2, 0xa5,
0xa8, 0xde, 0x6d, 0x59, 0x39, 0x5b, 0xa1, 0x01, 0xba, 0xb0, 0x5c, 0xde, 0x35,
0x4a, 0xa9, 0xe5, 0x9a, 0xee, 0x8c, 0x15, 0xbe, 0xa7, 0x2d, 0x07, 0xa3, 0x03,
0x39, 0xc7, 0xfa, 0x7e, 0x86, 0xe2, 0xe6, 0x3f, 0xf6, 0xc0, 0x03, 0x02, 0x9d,
0x34, 0xa7, 0x56, 0x26, 0x52, 0x68, 0x08, 0x72, 0x81, 0xde, 0xdf, 0x83, 0xd1,
0xaa, 0x89, 0x99, 0x8e, 0xb6, 0xd6, 0x9a, 0x15, 0xbf, 0x24, 0xb7, 0xe6, 0x11,
0x1b, 0x4d
},
},
};

const struct sha_tv SHA1_TV[] = {
{
.msg_len = 512 / 8,
.msg = (unsigned char []){
0x45, 0x92, 0x7e, 0x32, 0xdd, 0xf8, 0x01, 0xca, 0xf3, 0x5e, 0x18, 0xe7, 0xb5,
0x07, 0x8b, 0x7f, 0x54, 0x35, 0x27, 0x82, 0x12, 0xec, 0x6b, 0xb9, 0x9d, 0xf8,
0x84, 0xf4, 0x9b, 0x32, 0x7c, 0x64, 0x86, 0xfe, 0xae, 0x46, 0xba, 0x18, 0x7d,
0xc1, 0xcc, 0x91, 0x45, 0x12, 0x1e, 0x14, 0x92, 0xe6, 0xb0, 0x6e, 0x90, 0x07,
0x39, 0x4d, 0xc3, 0x3b, 0x77, 0x48, 0xf8, 0x6a, 0xc3, 0x20, 0x7c, 0xfe
},
.md = (unsigned char []){
0xa7, 0x0c, 0xfb, 0xfe, 0x75, 0x63, 0xdd, 0x0e, 0x66, 0x5c, 0x7c, 0x67, 0x15,
0xa9, 0x6a, 0x8d, 0x75, 0x69, 0x50, 0xc0
}
},
};

const struct sha_tv SHA224_TV[] = {
{
.msg_len = 512 / 8,
.msg = (unsigned char []){
0xa3, 0x31, 0x0b, 0xa0, 0x64, 0xbe, 0x2e, 0x14, 0xad, 0x32, 0x27, 0x6e, 0x18,
0xcd, 0x03, 0x10, 0xc9, 0x33, 0xa6, 0xe6, 0x50, 0xc3, 0xc7, 0x54, 0xd0, 0x24,
0x3c, 0x6c, 0x61, 0x20, 0x78, 0x65, 0xb4, 0xb6, 0x52, 0x48, 0xf6, 0x6a, 0x08,
0xed, 0xf6, 0xe0, 0x83, 0x26, 0x89, 0xa9, 0xdc, 0x3a, 0x2e, 0x5d, 0x20, 0x95,
0xee, 0xea, 0x50, 0xbd, 0x86, 0x2b, 0xac, 0x88, 0xc8, 0xbd, 0x31, 0x8d
},
.md = (unsigned char []){
0xb2, 0xa5, 0x58, 0x6d, 0x9c, 0xbf, 0x0b, 0xaa, 0x99, 0x91, 0x57, 0xb4, 0xaf,
0x06, 0xd8, 0x8a, 0xe0, 0x8d, 0x7c, 0x9f, 0xaa, 0xb4, 0xbc, 0x1a, 0x96, 0x82,
0x9d, 0x65
}
},
};

const struct sha_tv SHA256_TV[] = {
{
.msg_len = 512 / 8,
.msg = (unsigned char []){
0x5a, 0x86, 0xb7, 0x37, 0xea, 0xea, 0x8e, 0xe9, 0x76, 0xa0, 0xa2, 0x4d, 0xa6,
0x3e, 0x7e, 0xd7, 0xee, 0xfa, 0xd1, 0x8a, 0x10, 0x1c, 0x12, 0x11, 0xe2, 0xb3,
0x65, 0x0c, 0x51, 0x87, 0xc2, 0xa8, 0xa6, 0x50, 0x54, 0x72, 0x08, 0x25, 0x1f,
0x6d, 0x42, 0x37, 0xe6, 0x61, 0xc7, 0xbf, 0x4c, 0x77, 0xf3, 0x35, 0x39, 0x03,
0x94, 0xc3, 0x7f, 0xa1, 0xa9, 0xf9, 0xbe, 0x83, 0x6a, 0xc2, 0x85, 0x09
},
.md = (unsigned char []){
0x42, 0xe6, 0x1e, 0x17, 0x4f, 0xbb, 0x38, 0x97, 0xd6, 0xdd, 0x6c, 0xef, 0x3d,
0xd2, 0x80, 0x2f, 0xe6, 0x7b, 0x33, 0x19, 0x53, 0xb0, 0x61, 0x14, 0xa6, 0x5c,
0x77, 0x28, 0x59, 0xdf, 0xc1, 0xaa
}
},
};

const struct sha_tv SHA384_TV[] = {
{
.msg_len = 512 / 8,
.msg = (unsigned char []){
0x93, 0x03, 0x5d, 0x3a, 0x13, 0xae, 0x1b, 0x06, 0xdd, 0x03, 0x3e, 0x76, 0x4a,
0xca, 0x01, 0x24, 0x96, 0x1d, 0xa7, 0x9c, 0x36, 0x6c, 0x6c, 0x75, 0x6b, 0xc4,
0xbc, 0xc1, 0x18, 0x50, 0xa3, 0xa8, 0xd1, 0x20, 0x85, 0x4f, 0x34, 0x29, 0x0f,
0xff, 0x7c, 0x8d, 0x6d, 0x83, 0x53, 0x1d, 0xbd, 0xd1, 0xe8, 0x1c, 0xc4, 0xed,
0x42, 0x46, 0xe0, 0x0b, 0xd4, 0x11, 0x3e, 0xf4, 0x51, 0x33, 0x4d, 0xaa
},
.md = (unsigned char []){
0x8d, 0x46, 0xcc, 0x84, 0xb6, 0xc2, 0xde, 0xb2, 0x06, 0xaa, 0x5c, 0x86, 0x17,
0x98, 0x79, 0x87, 0x51, 0xa2, 0x6e, 0xe7, 0x4b, 0x1d, 0xaf, 0x3a, 0x55, 0x7c,
0x41, 0xae, 0xbd, 0x65, 0xad, 0xc0, 0x27, 0x55, 0x9f, 0x7c, 0xd9, 0x2b, 0x25,
0x5b, 0x37, 0x4c, 0x83, 0xbd, 0x55, 0x56, 0x8b, 0x45
}
},
};

const struct sha_tv SHA512_TV[] = {
{
.msg_len = 512 / 8,
.msg = (unsigned char []){
0xc1, 0xca, 0x70, 0xae, 0x12, 0x79, 0xba, 0x0b, 0x91, 0x81, 0x57, 0x55, 0x8b,
0x49, 0x20, 0xd6, 0xb7, 0xfb, 0xa8, 0xa0, 0x6b, 0xe5, 0x15, 0x17, 0x0f, 0x20,
0x2f, 0xaf, 0xd3, 0x6f, 0xb7, 0xf7, 0x9d, 0x69, 0xfa, 0xd7, 0x45, 0xdb, 0xa6,
0x15, 0x05, 0x68, 0xdb, 0x1e, 0x2b, 0x72, 0x85, 0x04, 0x11, 0x3e, 0xea, 0xc3,
0x4f, 0x52, 0x7f, 0xc8, 0x2f, 0x22, 0x00, 0xb4, 0x62, 0xec, 0xbf, 0x5d
},
.md = (unsigned char []){
0x04, 0x6e, 0x46, 0x62, 0x39, 0x12, 0xb3, 0x93, 0x2b, 0x8d, 0x66, 0x2a, 0xb4,
0x25, 0x83, 0x42, 0x38, 0x43, 0x20, 0x63, 0x01, 0xb5, 0x8b, 0xf2, 0x0a, 0xb6,
0xd7, 0x6f, 0xd4, 0x7f, 0x1c, 0xbb, 0xcf, 0x42, 0x1d, 0xf5, 0x36, 0xec, 0xd7,
0xe5, 0x6d, 0xb5, 0x35, 0x4e, 0x7e, 0x0f, 0x98, 0x82, 0x2d, 0x21, 0x29, 0xc1,
0x97, 0xf6, 0xf0, 0xf2, 0x22, 0xb8, 0xec, 0x52, 0x31, 0xf3, 0x96, 0x7d
}
},
};
#endif /* ICA_FIPS */

const struct drbg_sha512_tv DRBG_SHA512_TV[] = {
{
.no_reseed = true,
.pr = false,
.entropy_len = 256 / 8,
.nonce_len = 128 / 8,
.pers_len = 0,
.add_len = 0,
.prnd_len = 2048 / 8,

.inst.entropy = (unsigned char[]){
0x6b,0x50,0xa7,0xd8,0xf8,0xa5,0x5d,0x7a,0x3d,0xf8,0xbb,0x40,0xbc,0xc3,0xb7,0x22,
0xd8,0x70,0x8d,0xe6,0x7f,0xda,0x01,0x0b,0x03,0xc4,0xc8,0x4d,0x72,0x09,0x6f,0x8c,
},
.inst.nonce = (unsigned char[]){
0x3e,0xc6,0x49,0xcc,0x62,0x56,0xd9,0xfa,0x31,0xdb,0x7a,0x29,0x04,0xaa,0xf0,0x25,
},
.inst.v = (unsigned char[]){
0xad,0xe3,0x6f,0x9a,0x01,0xc7,0x76,0x61,0x34,0x35,0xf5,0x4e,0x24,0x74,0x22,0x21,
0x9a,0x29,0x89,0xc7,0x93,0x2e,0x60,0x1e,0xe8,0x14,0x24,0x8d,0xd5,0x03,0xf1,0x65,
0x5d,0x08,0x22,0x72,0xd5,0xad,0x95,0xe1,0x23,0x1e,0x8a,0xa7,0x13,0xd9,0x2b,0x5e,
0xbc,0xbb,0x80,0xab,0x8d,0xe5,0x79,0xab,0x5b,0x47,0x4e,0xdd,0xee,0x6b,0x03,0x8f,
0x0f,0x5c,0x5e,0xa9,0x1a,0x83,0xdd,0xd3,0x88,0xb2,0x75,0x4b,0xce,0x83,0x36,0x57,
0x4b,0xf1,0x5c,0xca,0x7e,0x09,0xc0,0xd3,0x89,0xc6,0xe0,0xda,0xc4,0x81,0x7e,0x5b,
0xf9,0xe1,0x01,0xc1,0x92,0x05,0xea,0xf5,0x2f,0xc6,0xc6,0xc7,0x8f,0xbc,0xf4,
},
.inst.c = (unsigned char[]){
0xf4,0xa3,0xe5,0xa0,0x72,0x63,0x95,0xc6,0x4f,0x48,0xd0,0x8b,0x5b,0x5f,0x8e,0x6b,
0x96,0x1f,0x16,0xed,0xbc,0x66,0x94,0x45,0x31,0xd7,0x47,0x73,0x22,0xa5,0x86,0xce,
0xc0,0x4c,0xac,0x63,0xb8,0x39,0x50,0xbf,0xe6,0x59,0x6c,0x38,0x58,0x99,0x1f,0x27,
0xa7,0x9d,0x71,0x2a,0xb3,0x7b,0xf9,0xfb,0x17,0x86,0xaa,0x99,0x81,0xaa,0x43,0xe4,
0x37,0xd3,0x1e,0x6e,0xe5,0xe6,0xee,0xc2,0xed,0x95,0x4f,0x53,0x0e,0x46,0x8a,0xcc,
0x45,0xa5,0xdb,0x69,0x0d,0x81,0xc9,0x32,0x92,0xbc,0x8f,0x33,0xe6,0xf6,0x09,0x7c,
0x8e,0x05,0x19,0x0d,0xf1,0xb6,0xcc,0xf3,0x02,0x21,0x90,0x25,0xec,0xed,0x0e,
},
.inst.reseed_ctr = 1,

.gen1.v =(unsigned char[]){
0xa2,0x87,0x55,0x3a,0x74,0x2b,0x0c,0x27,0x83,0x7e,0xc5,0xd9,0x7f,0xd3,0xb0,0x8d,
0x30,0x48,0xa0,0xb5,0x4f,0x94,0xf4,0x64,0x19,0xeb,0x6c,0x00,0xf7,0xa9,0x78,0x34,
0x1d,0x54,0xce,0xd6,0x8d,0xe6,0xe6,0xa1,0x09,0x77,0xf6,0xdf,0x6c,0x72,0x4b,0x26,
0x78,0x01,0xa8,0x2e,0x3a,0x96,0x7a,0x5f,0xb0,0x63,0x6e,0xdf,0xc9,0x16,0xc0,0x93,
0x52,0x6f,0x16,0x68,0xea,0x76,0x95,0x50,0x56,0xc9,0x54,0x09,0x44,0xce,0x67,0x73,
0xd6,0x55,0x30,0xce,0xae,0xa7,0xcf,0xf2,0x70,0xb3,0x57,0x27,0x86,0xe9,0x60,0x09,
0x73,0x23,0x2e,0xa5,0xdc,0xd3,0x75,0x77,0xc0,0xd9,0x84,0x37,0x19,0x9c,0x69,
} ,
.gen1.c = (unsigned char[]){
0xf4,0xa3,0xe5,0xa0,0x72,0x63,0x95,0xc6,0x4f,0x48,0xd0,0x8b,0x5b,0x5f,0x8e,0x6b,
0x96,0x1f,0x16,0xed,0xbc,0x66,0x94,0x45,0x31,0xd7,0x47,0x73,0x22,0xa5,0x86,0xce,
0xc0,0x4c,0xac,0x63,0xb8,0x39,0x50,0xbf,0xe6,0x59,0x6c,0x38,0x58,0x99,0x1f,0x27,
0xa7,0x9d,0x71,0x2a,0xb3,0x7b,0xf9,0xfb,0x17,0x86,0xaa,0x99,0x81,0xaa,0x43,0xe4,
0x37,0xd3,0x1e,0x6e,0xe5,0xe6,0xee,0xc2,0xed,0x95,0x4f,0x53,0x0e,0x46,0x8a,0xcc,
0x45,0xa5,0xdb,0x69,0x0d,0x81,0xc9,0x32,0x92,0xbc,0x8f,0x33,0xe6,0xf6,0x09,0x7c,
0x8e,0x05,0x19,0x0d,0xf1,0xb6,0xcc,0xf3,0x02,0x21,0x90,0x25,0xec,0xed,0x0e,
},
.gen1.reseed_ctr = 2,

.gen2.v = (unsigned char[]){
0x97,0x2b,0x3a,0xda,0xe6,0x8e,0xa1,0xed,0xd2,0xc7,0x96,0x64,0xdb,0x33,0x3e,0xf8,
0xc6,0x67,0xb7,0xa3,0x0b,0xfb,0x88,0xa9,0x4b,0xc2,0xb3,0x74,0x1a,0x4e,0xff,0x02,
0xdd,0xa1,0x7b,0x3a,0x46,0x20,0x37,0x60,0xef,0xd1,0x63,0x17,0xc5,0x0b,0x6a,0xb8,
0xbc,0x68,0x49,0x73,0xb3,0xa3,0x83,0x57,0x0e,0x00,0x53,0x78,0xdd,0x89,0xf1,0xf9,
0xa6,0xdc,0xcb,0x83,0xb6,0xe2,0x28,0x21,0x94,0x41,0x33,0x56,0xa7,0x9c,0x52,0x3d,
0x5e,0x38,0x1f,0x78,0xe1,0x22,0xa9,0xa2,0xd2,0xe6,0x00,0xdb,0xc1,0x61,0x85,0xf5,
0x1c,0xca,0x1c,0x70,0xf5,0x5d,0x27,0x6a,0x76,0x63,0x1b,0xac,0x4e,0xee,0x7a,
},
.gen2.c = (unsigned char[]){
0xf4,0xa3,0xe5,0xa0,0x72,0x63,0x95,0xc6,0x4f,0x48,0xd0,0x8b,0x5b,0x5f,0x8e,0x6b,
0x96,0x1f,0x16,0xed,0xbc,0x66,0x94,0x45,0x31,0xd7,0x47,0x73,0x22,0xa5,0x86,0xce,
0xc0,0x4c,0xac,0x63,0xb8,0x39,0x50,0xbf,0xe6,0x59,0x6c,0x38,0x58,0x99,0x1f,0x27,
0xa7,0x9d,0x71,0x2a,0xb3,0x7b,0xf9,0xfb,0x17,0x86,0xaa,0x99,0x81,0xaa,0x43,0xe4,
0x37,0xd3,0x1e,0x6e,0xe5,0xe6,0xee,0xc2,0xed,0x95,0x4f,0x53,0x0e,0x46,0x8a,0xcc,
0x45,0xa5,0xdb,0x69,0x0d,0x81,0xc9,0x32,0x92,0xbc,0x8f,0x33,0xe6,0xf6,0x09,0x7c,
0x8e,0x05,0x19,0x0d,0xf1,0xb6,0xcc,0xf3,0x02,0x21,0x90,0x25,0xec,0xed,0x0e,
},
.gen2.reseed_ctr = 3,

.prnd = (unsigned char[]){
0x95,0xb7,0xf1,0x7e,0x98,0x02,0xd3,0x57,0x73,0x92,0xc6,0xa9,0xc0,0x80,0x83,0xb6,
0x7d,0xd1,0x29,0x22,0x65,0xb5,0xf4,0x2d,0x23,0x7f,0x1c,0x55,0xbb,0x9b,0x10,0xbf,
0xcf,0xd8,0x2c,0x77,0xa3,0x78,0xb8,0x26,0x6a,0x00,0x99,0x14,0x3b,0x3c,0x2d,0x64,
0x61,0x1e,0xee,0xb6,0x9a,0xcd,0xc0,0x55,0x95,0x7c,0x13,0x9e,0x8b,0x19,0x0c,0x7a,
0x06,0x95,0x5f,0x2c,0x79,0x7c,0x27,0x78,0xde,0x94,0x03,0x96,0xa5,0x01,0xf4,0x0e,
0x91,0x39,0x6a,0xcf,0x8d,0x7e,0x45,0xeb,0xdb,0xb5,0x3b,0xbf,0x8c,0x97,0x52,0x30,
0xd2,0xf0,0xff,0x91,0x06,0xc7,0x61,0x19,0xae,0x49,0x8e,0x7f,0xbc,0x03,0xd9,0x0f,
0x8e,0x4c,0x51,0x62,0x7a,0xed,0x5c,0x8d,0x42,0x63,0xd5,0xd2,0xb9,0x78,0x87,0x3a,
0x0d,0xe5,0x96,0xee,0x6d,0xc7,0xf7,0xc2,0x9e,0x37,0xee,0xe8,0xb3,0x4c,0x90,0xdd,
0x1c,0xf6,0xa9,0xdd,0xb2,0x2b,0x4c,0xbd,0x08,0x6b,0x14,0xb3,0x5d,0xe9,0x3d,0xa2,
0xd5,0xcb,0x18,0x06,0x69,0x8c,0xbd,0x7b,0xbb,0x67,0xbf,0xe3,0xd3,0x1f,0xd2,0xd1,
0xdb,0xd2,0xa1,0xe0,0x58,0xa3,0xeb,0x99,0xd7,0xe5,0x1f,0x1a,0x93,0x8e,0xed,0x5e,
0x1c,0x1d,0xe2,0x3a,0x6b,0x43,0x45,0xd3,0x19,0x14,0x09,0xf9,0x2f,0x39,0xb3,0x67,
0x0d,0x8d,0xbf,0xb6,0x35,0xd8,0xe6,0xa3,0x69,0x32,0xd8,0x10,0x33,0xd1,0x44,0x8d,
0x63,0xb4,0x03,0xdd,0xf8,0x8e,0x12,0x1b,0x6e,0x81,0x9a,0xc3,0x81,0x22,0x6c,0x13,
0x21,0xe4,0xb0,0x86,0x44,0xf6,0x72,0x7c,0x36,0x8c,0x5a,0x9f,0x7a,0x4b,0x3e,0xe2,
},
},
{
.no_reseed = true,
.pr = false,
.entropy_len = 256 / 8,
.nonce_len = 128 / 8,
.pers_len = 0,
.add_len = 256 / 8,
.prnd_len = 2048 / 8,

.inst.entropy = (unsigned char[]){
0x9c,0x96,0xa3,0x4f,0x68,0x68,0x9b,0x8a,0xa8,0xd9,0xc1,0xf6,0xcd,0x0f,0xa7,0xc6,
0xf9,0x60,0x71,0xca,0xf1,0xbf,0x55,0x56,0xf4,0x5b,0xdb,0xf4,0x8c,0x6c,0xf0,0xc6,
},
.inst.nonce = (unsigned char[]){
0x88,0x5c,0x25,0x39,0x04,0x6a,0xfb,0x14,0x01,0xeb,0x7a,0x5c,0x84,0xdb,0xd9,0xc2,
},
.inst.v = (unsigned char[]){
0x11,0xb6,0xf9,0xc7,0x4e,0x77,0x32,0xa3,0x93,0xe6,0xd3,0x4f,0x4d,0x8d,0x5b,0xb9,
0x3d,0xf8,0xd0,0x7a,0x52,0xfe,0xb5,0x30,0xe0,0x9e,0x44,0xaa,0xdc,0x90,0x19,0x16,
0x5e,0x9c,0xf9,0x03,0x87,0xf8,0x0f,0x1a,0x41,0x62,0x16,0x90,0xff,0x29,0x08,0xb7,
0x28,0x21,0x37,0x09,0x27,0xcb,0xc1,0xd3,0xd6,0x6d,0x47,0xa9,0xc0,0x4c,0x8e,0xd8,
0x49,0x26,0x39,0xe3,0x8b,0xe7,0x6d,0x30,0x05,0xc1,0xc4,0xa8,0x83,0x73,0x0b,0x55,
0xa4,0x91,0x4a,0x87,0xc9,0x73,0x31,0xbd,0x06,0xba,0xf0,0x79,0x40,0x95,0x7f,0x62,
0x82,0xe3,0xcd,0x28,0xa3,0x7d,0xa7,0x55,0xde,0x25,0x0f,0xf1,0x08,0x38,0x85,},
.inst.c = (unsigned char[]){
0xb6,0xba,0xbd,0x1c,0xa5,0x4b,0x3c,0x3c,0x7c,0x5b,0xec,0xf6,0xc1,0xa0,0x52,0xa5,
0x37,0x6e,0x2a,0x63,0x7a,0xe9,0x25,0x1a,0x8a,0xee,0x90,0xa4,0x6e,0x9b,0x0a,0xbf,
0xe9,0xe1,0xda,0x50,0xd6,0x97,0x17,0x10,0x04,0x3a,0x3f,0xff,0x16,0xef,0x17,0x05,
0xe4,0x98,0x27,0xd7,0xe9,0x5c,0x4e,0xed,0x90,0x9e,0x34,0xee,0x00,0x39,0x9a,0x96,
0x22,0x10,0xcb,0x36,0x91,0x33,0xed,0x04,0xcf,0x50,0x33,0xc2,0xa8,0xda,0x69,0x2d,
0x41,0x8c,0x70,0xec,0x14,0x10,0x62,0x37,0x9c,0xd0,0x84,0x6f,0x1a,0x32,0x72,0x5e,
0x46,0x4d,0x7b,0x39,0xa1,0x69,0x9e,0x49,0xdc,0xc4,0x02,0x6a,0xe2,0xd8,0x7e,
},
.inst.reseed_ctr = 1,

.gen1.add = (unsigned char[]){
0xcb,0x61,0xc4,0xf7,0x5c,0x01,0xb5,0x78,0xaa,0x23,0x3a,0x0b,0xae,0x48,0x81,0xc0,
0xa1,0x15,0x27,0xc2,0x2f,0xe7,0xb3,0x4f,0xb6,0xae,0x62,0xee,0xbc,0xfe,0x60,0x85,
},
.gen1.v = (unsigned char[]){
0xc8,0x71,0xb6,0xe3,0xf3,0xc2,0x6e,0xe0,0x10,0x42,0xc0,0x46,0x0f,0x2d,0xae,0x5e,
0x75,0x66,0xfa,0xdd,0xcd,0xe7,0xda,0x4b,0x6b,0x8c,0xd5,0x4f,0x4b,0x2b,0x23,0xd6,
0x48,0x7e,0xd3,0x54,0x5e,0x8f,0x26,0x2a,0x45,0x9c,0x56,0x90,0x16,0x18,0x20,0xc4,
0xd7,0x95,0xe0,0xb1,0xd2,0xc5,0x25,0x5c,0x61,0xe5,0xa0,0x1b,0xa2,0xee,0x46,0x8e,
0xfe,0x9e,0xdf,0x2a,0x9d,0x68,0x0f,0x37,0x5d,0x37,0xd7,0x2f,0x43,0xf2,0x08,0x3e,
0x92,0x6d,0x6d,0x78,0xb2,0x21,0x95,0xdd,0xb6,0x4f,0x40,0xbc,0x0c,0x56,0xc4,0x5a,
0x0d,0x48,0x14,0x19,0xbf,0x2e,0x29,0xca,0xec,0xfc,0xbd,0xca,0x32,0xbc,0x2e,
},
.gen1.c = (unsigned char[]){
0xb6,0xba,0xbd,0x1c,0xa5,0x4b,0x3c,0x3c,0x7c,0x5b,0xec,0xf6,0xc1,0xa0,0x52,0xa5,
0x37,0x6e,0x2a,0x63,0x7a,0xe9,0x25,0x1a,0x8a,0xee,0x90,0xa4,0x6e,0x9b,0x0a,0xbf,
0xe9,0xe1,0xda,0x50,0xd6,0x97,0x17,0x10,0x04,0x3a,0x3f,0xff,0x16,0xef,0x17,0x05,
0xe4,0x98,0x27,0xd7,0xe9,0x5c,0x4e,0xed,0x90,0x9e,0x34,0xee,0x00,0x39,0x9a,0x96,
0x22,0x10,0xcb,0x36,0x91,0x33,0xed,0x04,0xcf,0x50,0x33,0xc2,0xa8,0xda,0x69,0x2d,
0x41,0x8c,0x70,0xec,0x14,0x10,0x62,0x37,0x9c,0xd0,0x84,0x6f,0x1a,0x32,0x72,0x5e,
0x46,0x4d,0x7b,0x39,0xa1,0x69,0x9e,0x49,0xdc,0xc4,0x02,0x6a,0xe2,0xd8,0x7e,
},
.gen1.reseed_ctr = 2,

.gen2.add = (unsigned char[]){
0xc0,0x66,0xfd,0x2e,0xb8,0xe4,0xae,0xa2,0xe7,0x14,0x5e,0xda,0x0c,0xfc,0x8b,0xef,
0x5e,0xed,0xcc,0x36,0x7b,0x1c,0xb4,0xde,0x7e,0xb2,0xc2,0x75,0x9f,0xa7,0x5b,0xf7,
},
.gen2.v = (unsigned char[]){
0x7f,0x2c,0x74,0x00,0x99,0x0d,0xab,0x1c,0x8c,0x9e,0xad,0x3c,0xd0,0xce,0x01,0x03,
0xac,0xd5,0x25,0x41,0x48,0xd0,0xff,0x65,0xf6,0x7b,0x65,0xf3,0xb9,0xc6,0x2e,0x96,
0x32,0x60,0xad,0xa5,0x35,0x26,0x3d,0x3a,0x49,0xd6,0x96,0x8f,0x2d,0x07,0x38,0xf1,
0xff,0x65,0xdf,0x49,0x81,0xe3,0xf6,0x71,0x0c,0x7e,0x69,0xec,0xc0,0x61,0x16,0xa8,
0x27,0xa5,0x38,0x37,0x38,0x52,0x56,0xb8,0x2b,0x25,0x7b,0x70,0x33,0x15,0x79,0xb2,
0x2a,0xbd,0x05,0x55,0x3e,0x2d,0x36,0x14,0x66,0xe4,0x83,0x97,0x2a,0xbe,0x31,0x10,
0xcd,0x4d,0x09,0x35,0x99,0x9a,0x21,0xd1,0xce,0x3d,0x98,0x0d,0x4a,0x31,0xec,
},
.gen2.c = (unsigned char[]){
0xb6,0xba,0xbd,0x1c,0xa5,0x4b,0x3c,0x3c,0x7c,0x5b,0xec,0xf6,0xc1,0xa0,0x52,0xa5,
0x37,0x6e,0x2a,0x63,0x7a,0xe9,0x25,0x1a,0x8a,0xee,0x90,0xa4,0x6e,0x9b,0x0a,0xbf,
0xe9,0xe1,0xda,0x50,0xd6,0x97,0x17,0x10,0x04,0x3a,0x3f,0xff,0x16,0xef,0x17,0x05,
0xe4,0x98,0x27,0xd7,0xe9,0x5c,0x4e,0xed,0x90,0x9e,0x34,0xee,0x00,0x39,0x9a,0x96,
0x22,0x10,0xcb,0x36,0x91,0x33,0xed,0x04,0xcf,0x50,0x33,0xc2,0xa8,0xda,0x69,0x2d,
0x41,0x8c,0x70,0xec,0x14,0x10,0x62,0x37,0x9c,0xd0,0x84,0x6f,0x1a,0x32,0x72,0x5e,
0x46,0x4d,0x7b,0x39,0xa1,0x69,0x9e,0x49,0xdc,0xc4,0x02,0x6a,0xe2,0xd8,0x7e,
},
.gen2.reseed_ctr = 3,

.prnd = (unsigned char[]){
0x78,0x2c,0x20,0x8e,0xd5,0x80,0x44,0xe7,0x8b,0x5b,0xbb,0xd8,0x77,0x2a,0x3c,0xaf,
0x25,0xb4,0x7d,0x36,0xaf,0xeb,0x0d,0x34,0x93,0xc4,0x3e,0x01,0xcc,0x66,0xa0,0xca,
0x2f,0xac,0xed,0x2a,0xb1,0x86,0xbc,0x46,0x82,0x5d,0x98,0x9c,0xf8,0xee,0x7c,0x95,
0xf8,0xc0,0xb0,0xd2,0xb7,0x6e,0x6c,0x85,0x90,0xe7,0x28,0x34,0xd4,0xc5,0x24,0x45,
0xae,0xce,0xeb,0x7b,0xf5,0xf5,0xd9,0xac,0x44,0xa1,0x2c,0xbd,0x3f,0xa7,0xf4,0x46,
0x2f,0x85,0x64,0x52,0xdc,0x4a,0x92,0x91,0x82,0xd2,0x38,0x8a,0xa7,0x63,0x5b,0x96,
0x98,0xa9,0x12,0x58,0x5d,0xf7,0xf5,0x60,0xad,0xc5,0x08,0x0d,0x53,0xb8,0x2b,0xbd,
0x7e,0x9e,0x48,0x0b,0x00,0xd1,0xda,0x5b,0xb2,0xd4,0x80,0xca,0xe2,0xba,0x8c,0x67,
0xd4,0xbf,0x3b,0xfd,0x14,0x6a,0x91,0xd6,0xaa,0xb3,0x9f,0xaa,0xe1,0x60,0x0a,0xf2,
0xce,0x32,0x04,0xca,0xbf,0x4c,0x1c,0xae,0xe4,0xcf,0xd5,0xe6,0xf8,0xdb,0x19,0x02,
0x03,0x3f,0x7f,0x8d,0x33,0xbc,0x6e,0x0e,0x5d,0x32,0xa3,0x20,0xba,0x73,0x5d,0x09,
0x1f,0x30,0x86,0x7b,0x7c,0xb7,0x88,0x0c,0x2e,0x3c,0xe6,0xaa,0xda,0x79,0x66,0x41,
0x91,0xdf,0x36,0x0d,0x35,0xfe,0x9a,0xe7,0xba,0xbc,0xa4,0x14,0x85,0xb0,0x6a,0xb4,
0x9d,0xff,0x52,0x87,0x82,0xfb,0xe6,0xf2,0xb0,0xe7,0x49,0x96,0xe9,0xce,0x92,0x72,
0xd1,0xef,0x39,0x2b,0xe5,0xc1,0x7c,0xc6,0x2c,0x74,0xbe,0x50,0x4e,0x6a,0x87,0x31,
0xdd,0x95,0x48,0xb0,0xdb,0x27,0xe0,0xb7,0xdb,0x48,0x86,0xf5,0x37,0x88,0x36,0x23,
},
},
{
.no_reseed = true,
.pr = false,
.entropy_len = 256 / 8,
.nonce_len = 128 / 8,
.pers_len = 256 / 8,
.add_len = 0,
.prnd_len = 2048 / 8,

.inst.entropy = (unsigned char[]){
0xb5,0x10,0xa6,0x0a,0x5a,0x19,0x00,0x7b,0xa8,0x08,0xe6,0x0b,0x16,0x28,0xed,0x99,
0x07,0xd1,0xbe,0xf9,0x92,0xd4,0xfc,0x0a,0xbe,0x51,0x50,0x01,0x31,0xcb,0x4c,0xf8,
},
.inst.nonce = (unsigned char[]){
0xeb,0x1b,0xd7,0x83,0xac,0xcc,0x85,0x5e,0xb3,0x20,0xcd,0xe1,0x0b,0x1b,0xe6,0xd0,
},
.inst.pers = (unsigned char[]){
0x7e,0xa1,0x0e,0x96,0xaf,0x90,0x0c,0x25,0xd3,0xbe,0x3b,0x50,0xa0,0xcc,0x71,0xa7,
0x9f,0xe4,0x14,0xbd,0x4c,0x37,0x39,0x80,0x3f,0x02,0xff,0xe5,0xb2,0x60,0xbf,0xbb,
},
.inst.v = (unsigned char[]){
0x6e,0x10,0x09,0xeb,0x74,0x25,0x4e,0x17,0xbc,0xa3,0x2a,0xb7,0xbc,0x01,0x91,0xba,
0x70,0x89,0xb7,0x8c,0x4b,0xa4,0xbb,0xab,0xab,0x03,0xfc,0x6c,0x1e,0xa5,0xb7,0xdb,
0xef,0x37,0xec,0xcd,0x5f,0x70,0x7f,0x99,0x11,0xab,0x57,0xe4,0xd1,0x8d,0x34,0x84,
0x53,0x42,0x42,0x90,0x07,0x33,0x61,0x5d,0x66,0xf2,0x2b,0x76,0x90,0x1e,0x08,0xcc,
0x0d,0x86,0xa3,0x89,0xc8,0x6f,0x27,0x12,0xab,0x34,0x05,0x23,0x66,0x36,0x14,0x1d,
0xdd,0xc8,0xae,0x83,0xaa,0x33,0x77,0x21,0xf5,0x7e,0x23,0x08,0xf8,0x15,0x0a,0x35,
0x0f,0x07,0xef,0x94,0x68,0x26,0xef,0x80,0xba,0x20,0xfe,0x20,0x23,0xe4,0x39,
},
.inst.c = (unsigned char[]){
0x61,0x67,0x19,0xd5,0xd6,0xa4,0xdf,0xd8,0x39,0x9e,0x41,0xf3,0x67,0x7e,0x12,0xcc,
0xbd,0x2e,0x3d,0x1e,0x1d,0xc5,0x7b,0xa0,0xb7,0x1d,0x1c,0x61,0x5a,0x50,0x14,0xd7,
0x33,0x63,0x7f,0x2c,0x6f,0x0a,0x5e,0x97,0xe7,0x78,0x31,0x59,0xbe,0x99,0xf7,0x47,
0x7a,0x31,0x7a,0x5c,0x72,0x56,0x52,0xcc,0xd6,0x31,0x95,0xd9,0x73,0x6d,0xf9,0xa9,
0x7e,0x6f,0x59,0x4e,0x57,0x0b,0xad,0x9f,0x99,0x7a,0xa3,0x8b,0xdf,0x51,0x44,0xda,
0x63,0x61,0xfe,0x9b,0xe2,0x82,0x88,0x20,0x5a,0xbc,0x9d,0x8b,0x41,0x1f,0x5f,0x38,
0x0a,0x56,0x15,0xce,0xfe,0x6d,0x3e,0x82,0x88,0xda,0x5f,0x43,0x01,0xfb,0x08,
},
.inst.reseed_ctr = 1,

.gen1.v = (unsigned char[]){
0xcf,0x77,0x23,0xc1,0x4a,0xca,0x2d,0xef,0xf6,0x41,0x6c,0xab,0x23,0x7f,0xa4,0x87,
0x2d,0xb7,0xf4,0xaa,0x69,0x6a,0x37,0x4c,0x62,0x21,0x18,0xcd,0x78,0xf5,0xcc,0xb3,
0x22,0x9b,0x6b,0xf9,0xce,0x7a,0xde,0x30,0xf9,0x23,0x89,0x3e,0x90,0x27,0x2c,0xa2,
0x36,0x72,0xbc,0x98,0xda,0xea,0x39,0xca,0x18,0x52,0x1a,0xcc,0xc2,0x56,0x41,0xff,
0x36,0x00,0xf4,0x40,0x89,0x45,0x24,0x63,0x29,0x31,0x87,0xf2,0x41,0x98,0xca,0x38,
0x45,0x39,0xdd,0xaa,0xbb,0x5c,0x4b,0x72,0xb5,0x33,0xf1,0x51,0xec,0xcf,0x82,0x53,
0x6d,0xf2,0xcf,0x82,0x9a,0x69,0x41,0xf7,0x0b,0x95,0x49,0xe4,0xc2,0xd9,0x6e,
},
.gen1.c = (unsigned char[]){
0x61,0x67,0x19,0xd5,0xd6,0xa4,0xdf,0xd8,0x39,0x9e,0x41,0xf3,0x67,0x7e,0x12,0xcc,
0xbd,0x2e,0x3d,0x1e,0x1d,0xc5,0x7b,0xa0,0xb7,0x1d,0x1c,0x61,0x5a,0x50,0x14,0xd7,
0x33,0x63,0x7f,0x2c,0x6f,0x0a,0x5e,0x97,0xe7,0x78,0x31,0x59,0xbe,0x99,0xf7,0x47,
0x7a,0x31,0x7a,0x5c,0x72,0x56,0x52,0xcc,0xd6,0x31,0x95,0xd9,0x73,0x6d,0xf9,0xa9,
0x7e,0x6f,0x59,0x4e,0x57,0x0b,0xad,0x9f,0x99,0x7a,0xa3,0x8b,0xdf,0x51,0x44,0xda,
0x63,0x61,0xfe,0x9b,0xe2,0x82,0x88,0x20,0x5a,0xbc,0x9d,0x8b,0x41,0x1f,0x5f,0x38,
0x0a,0x56,0x15,0xce,0xfe,0x6d,0x3e,0x82,0x88,0xda,0x5f,0x43,0x01,0xfb,0x08,
},
.gen1.reseed_ctr = 2,

.gen2.v = (unsigned char[]){
0x30,0xde,0x3d,0x97,0x21,0x6f,0x0d,0xc8,0x2f,0xdf,0xae,0x9e,0x8a,0xfd,0xb7,0x53,
0xea,0xe6,0x31,0xc8,0x87,0x2f,0xb2,0xed,0x19,0x3e,0x35,0x2e,0xd3,0x45,0xe1,0x8a,
0x55,0xfe,0xeb,0x26,0x3d,0x85,0x3c,0xc8,0xe0,0x9b,0xba,0x98,0x4e,0xc1,0x24,0x3f,
0x77,0x1a,0x2c,0x1a,0x65,0x07,0x99,0x3f,0x08,0xe9,0x48,0x16,0x77,0x03,0x0d,0xa7,
0x8b,0xc5,0x13,0xcf,0x4d,0xf3,0x3e,0xf5,0xe1,0x61,0x09,0x2e,0xee,0x86,0xfb,0x21,
0x40,0x70,0x1b,0x20,0xdf,0x82,0xde,0x4c,0x3b,0xf0,0xe7,0x67,0x36,0xf6,0x24,0x82,
0xc7,0x94,0xff,0xd2,0x99,0x73,0x5f,0x49,0x13,0x23,0x77,0x8e,0xa4,0x72,0x8f,
},
.gen2.c = (unsigned char[]){
0x61,0x67,0x19,0xd5,0xd6,0xa4,0xdf,0xd8,0x39,0x9e,0x41,0xf3,0x67,0x7e,0x12,0xcc,
0xbd,0x2e,0x3d,0x1e,0x1d,0xc5,0x7b,0xa0,0xb7,0x1d,0x1c,0x61,0x5a,0x50,0x14,0xd7,
0x33,0x63,0x7f,0x2c,0x6f,0x0a,0x5e,0x97,0xe7,0x78,0x31,0x59,0xbe,0x99,0xf7,0x47,
0x7a,0x31,0x7a,0x5c,0x72,0x56,0x52,0xcc,0xd6,0x31,0x95,0xd9,0x73,0x6d,0xf9,0xa9,
0x7e,0x6f,0x59,0x4e,0x57,0x0b,0xad,0x9f,0x99,0x7a,0xa3,0x8b,0xdf,0x51,0x44,0xda,
0x63,0x61,0xfe,0x9b,0xe2,0x82,0x88,0x20,0x5a,0xbc,0x9d,0x8b,0x41,0x1f,0x5f,0x38,
0x0a,0x56,0x15,0xce,0xfe,0x6d,0x3e,0x82,0x88,0xda,0x5f,0x43,0x01,0xfb,0x08,
},
.gen2.reseed_ctr = 3,

.prnd = (unsigned char[]){
0x5f,0x03,0xab,0x52,0x9b,0x9e,0xda,0xb5,0x5e,0x75,0xe4,0x89,0xbc,0xeb,0x0a,0xb6,
0xcb,0x9e,0xc6,0x25,0x5c,0x61,0x96,0xcd,0x07,0x53,0x97,0xb0,0x91,0xc3,0xeb,0x59,
0x8e,0x31,0x62,0x32,0x0c,0x71,0xe3,0x89,0xe1,0xd9,0xd6,0x2a,0x2d,0xe5,0x99,0x0c,
0x20,0x97,0x56,0x10,0x84,0xe1,0x3a,0xb4,0xb2,0x97,0x7f,0x99,0xc2,0x3d,0xaf,0x21,
0x4b,0xbf,0xeb,0x16,0x21,0x67,0xc9,0x33,0x86,0x0f,0xb2,0xcc,0x9f,0x80,0x07,0x9e,
0x17,0x89,0x39,0x62,0xff,0x03,0xa2,0x6b,0x2b,0x4b,0x83,0x8e,0x6d,0xe9,0xff,0x5e,
0x9d,0xcf,0xba,0x73,0xac,0xc6,0xee,0x76,0xc4,0x77,0xd1,0xf7,0x00,0x22,0x65,0xe6,
0x93,0x85,0x0a,0x26,0x0e,0x68,0x61,0x7f,0x13,0xb2,0x47,0x16,0xd8,0xdb,0x42,0xc9,
0xa8,0xf2,0xe0,0xbd,0x50,0x02,0x28,0x9a,0x86,0xb0,0x2a,0x84,0x5d,0x98,0x86,0xd3,
0xfd,0x05,0xd8,0xd8,0x5a,0xc7,0x7e,0xa6,0x01,0x92,0xa2,0x90,0x47,0x30,0x3a,0x03,
0xb5,0x1e,0x0e,0x70,0x8f,0xa0,0x02,0x05,0xb5,0xc0,0x3b,0x67,0xb7,0x78,0x74,0x05,
0x2c,0xe5,0x6d,0x5c,0xf3,0x8e,0x72,0x6d,0x60,0x8b,0xc0,0xef,0x8f,0xc2,0xa7,0xd2,
0x77,0x78,0xf4,0x7c,0x68,0x91,0x63,0x7b,0x63,0x1c,0xcc,0x6e,0x36,0xeb,0x89,0x3f,
0xe0,0x8a,0x5b,0x9f,0x55,0x71,0xbd,0x43,0x2c,0x82,0x6d,0x99,0x69,0x8f,0xd4,0x17,
0x9b,0xc5,0xcf,0xb0,0xd8,0xea,0x2d,0x55,0x5c,0x36,0xb7,0x05,0x68,0xc2,0x20,0x6b,
0x18,0xf3,0xbe,0xa1,0x4b,0x39,0x8d,0x2b,0x6a,0x00,0x12,0x2d,0x7a,0x44,0x7e,0xa3,
},
},
{
.no_reseed = true,
.pr = false,
.entropy_len = 256 / 8,
.nonce_len = 128 / 8,
.pers_len = 256 / 8,
.add_len = 256 / 8,
.prnd_len = 2048 / 8,

.inst.entropy = (unsigned char[]){
0xb1,0x48,0xb3,0x17,0xa2,0x68,0x62,0x8f,0x04,0xc9,0xa8,0x7a,0x0a,0xc5,0xf9,0x14,
0x98,0xb8,0x02,0x0f,0x4e,0x48,0x35,0x72,0x0c,0x1a,0x3c,0xc0,0x7b,0x05,0x0f,0xa2,
},
.inst.nonce = (unsigned char[]){
0x2f,0x35,0xe5,0xb6,0x22,0xae,0xef,0xe7,0x56,0x05,0xc2,0x27,0x4e,0xc8,0xc6,0x96,
},
.inst.pers = (unsigned char[]){
0xfc,0x52,0x7a,0x2f,0x16,0xb5,0x3c,0x51,0x3f,0x94,0x85,0x5b,0x35,0xce,0xa6,0x09,
0x0c,0x30,0x3d,0xcc,0x64,0x2e,0x98,0xed,0x5f,0x32,0x3a,0xba,0x0f,0x35,0xfa,0x27,
},
.inst.v = (unsigned char[]){
0xc0,0xcb,0x7d,0x99,0x92,0xeb,0x7d,0x59,0x7e,0xbe,0x77,0x6d,0x6a,0x88,0x68,0x81,
0xc1,0x79,0x15,0xeb,0x90,0xb7,0x67,0x2d,0xee,0x5c,0x18,0xcb,0x10,0xd4,0xb9,0xfb,
0x7b,0x66,0x1d,0x0b,0xa6,0x8f,0x1b,0xef,0x72,0xbe,0xeb,0xc0,0x40,0x98,0x2c,0xe7,
0xcb,0xb2,0xd0,0xb9,0x0b,0xfd,0x3a,0xce,0x08,0xea,0x37,0xa3,0x3e,0x71,0x41,0x27,
0x18,0xb1,0xc0,0xe6,0x85,0x63,0xad,0x5b,0x35,0x3a,0x20,0xa5,0x84,0xad,0xf0,0x4c,
0xfa,0x4a,0x19,0x91,0x7c,0xf3,0x42,0xa1,0x3f,0x9f,0x51,0xbb,0x12,0xf3,0xe2,0x4b,
0x3b,0x1d,0xf0,0xf7,0x03,0xcb,0xa2,0x09,0x9f,0x63,0x2c,0x63,0x99,0xe7,0x4c,
},
.inst.c = (unsigned char[]){
0x02,0xd6,0x77,0x70,0xe8,0x13,0x7f,0x9b,0x39,0x96,0x09,0x63,0xae,0xa6,0x82,0x73,
0xe2,0x6c,0x75,0x94,0x5b,0xe1,0x13,0x6a,0xc1,0x4a,0xdd,0xdc,0xa6,0x33,0x90,0x43,
0xdc,0xab,0x44,0xe7,0xd1,0x2b,0xf1,0x1d,0xc4,0x30,0xa9,0x67,0x26,0xa7,0xa0,0x29,
0xeb,0x0e,0xc0,0x39,0xe0,0x37,0x6a,0x12,0x45,0x48,0x1a,0xeb,0x7e,0x06,0x8a,0x12,
0xb7,0x5f,0x55,0xd5,0xad,0x5f,0xe6,0xe5,0xb2,0x37,0xad,0x50,0xbc,0x9e,0x2d,0x91,
0xb7,0xce,0xaa,0x3a,0x23,0xfb,0xbb,0x04,0x49,0xe1,0x6f,0xdb,0xab,0x57,0xde,0x0a,
0xef,0xd8,0xd1,0x7f,0x36,0x28,0x1c,0x74,0x25,0x90,0xda,0xf8,0x3c,0x14,0x3d,
},
.inst.reseed_ctr = 1,

.gen1.add = (unsigned char[]){
0x2c,0xc9,0xf1,0x37,0xfc,0xd8,0xc2,0xd5,0x26,0xd7,0x00,0x93,0xfe,0x11,0xf9,0x0a,
0x0a,0x36,0xbc,0x97,0x64,0xa4,0xc5,0x60,0x90,0x72,0xe1,0x81,0xa2,0x49,0x45,0x16,
},
.gen1.v =(unsigned char[]){
0xc3,0xa1,0xf5,0x0a,0x7a,0xfe,0xfc,0xf4,0xb8,0x54,0x80,0xd1,0x19,0x2e,0xea,0xf5,
0xa3,0xe5,0x8b,0x7f,0xec,0x98,0x7a,0x98,0xaf,0xa6,0xf6,0xa7,0xb7,0x08,0x4a,0x3f,
0x58,0x11,0x61,0xf3,0x77,0xbb,0x0d,0x0d,0x36,0xef,0x95,0x27,0x67,0x3f,0xcd,0xf7,
0xc0,0xa6,0x8e,0x23,0x93,0x29,0x5e,0xdc,0x31,0x2d,0x32,0xc1,0x89,0xf7,0x7c,0xbb,
0x60,0xbe,0xdc,0x1e,0x7c,0x7a,0x16,0x77,0x81,0x5f,0x06,0x6f,0x70,0x37,0x43,0xfb,
0xcc,0xbe,0x3b,0x59,0x24,0x35,0x4e,0x7a,0xd0,0x88,0xd1,0xde,0xd4,0x1a,0x70,0x39,
0xbb,0x52,0x6a,0x91,0xef,0x70,0xe1,0xb1,0xdf,0xc9,0x39,0xc3,0xcc,0x04,0x58,
} ,
.gen1.c = (unsigned char[]){
0x02,0xd6,0x77,0x70,0xe8,0x13,0x7f,0x9b,0x39,0x96,0x09,0x63,0xae,0xa6,0x82,0x73,
0xe2,0x6c,0x75,0x94,0x5b,0xe1,0x13,0x6a,0xc1,0x4a,0xdd,0xdc,0xa6,0x33,0x90,0x43,
0xdc,0xab,0x44,0xe7,0xd1,0x2b,0xf1,0x1d,0xc4,0x30,0xa9,0x67,0x26,0xa7,0xa0,0x29,
0xeb,0x0e,0xc0,0x39,0xe0,0x37,0x6a,0x12,0x45,0x48,0x1a,0xeb,0x7e,0x06,0x8a,0x12,
0xb7,0x5f,0x55,0xd5,0xad,0x5f,0xe6,0xe5,0xb2,0x37,0xad,0x50,0xbc,0x9e,0x2d,0x91,
0xb7,0xce,0xaa,0x3a,0x23,0xfb,0xbb,0x04,0x49,0xe1,0x6f,0xdb,0xab,0x57,0xde,0x0a,
0xef,0xd8,0xd1,0x7f,0x36,0x28,0x1c,0x74,0x25,0x90,0xda,0xf8,0x3c,0x14,0x3d,
},
.gen1.reseed_ctr = 2,

.gen2.add = (unsigned char[]){
0xe4,0x03,0x61,0x24,0x5b,0x91,0x88,0x0e,0x30,0x8f,0xb7,0x77,0xc2,0x8b,0xbf,0xae,
0xa5,0x98,0x2e,0x45,0xfe,0xcb,0x77,0x57,0xbb,0x1c,0x9d,0xe2,0xdf,0x9d,0xc6,0x12,
},
.gen2.v = (unsigned char[]){
0xc6,0x78,0x6c,0x7b,0x63,0x12,0x7c,0x8f,0xf1,0xea,0x8a,0x34,0xc7,0xd5,0x6d,0x69,
0x86,0x52,0x01,0x14,0x48,0x79,0x8e,0x03,0x70,0xf1,0xd4,0x84,0x5d,0x3b,0xda,0x83,
0x34,0xbc,0xa6,0xdb,0x48,0xe6,0xfe,0x2a,0xfb,0x20,0x3e,0x8e,0x8d,0xe7,0x6e,0xe5,
0x30,0x94,0x36,0x9e,0x68,0xb3,0x2d,0xb7,0xe1,0xf7,0x18,0x53,0xc1,0xa3,0x0b,0x9c,
0xb5,0x5b,0x06,0x7b,0x9a,0x19,0x87,0xe9,0x18,0x80,0xf2,0xc8,0xdc,0x5d,0x77,0xec,
0x26,0x7c,0x12,0xcb,0x99,0xeb,0xf4,0xc3,0xf0,0xe1,0x3d,0xa4,0x74,0x45,0xe0,0xec,
0x23,0x49,0xa7,0x1d,0x34,0x16,0x8e,0x33,0xfe,0xbb,0x65,0x7f,0x8d,0x12,0xc7,
},
.gen2.c = (unsigned char[]){
0x02,0xd6,0x77,0x70,0xe8,0x13,0x7f,0x9b,0x39,0x96,0x09,0x63,0xae,0xa6,0x82,0x73,
0xe2,0x6c,0x75,0x94,0x5b,0xe1,0x13,0x6a,0xc1,0x4a,0xdd,0xdc,0xa6,0x33,0x90,0x43,
0xdc,0xab,0x44,0xe7,0xd1,0x2b,0xf1,0x1d,0xc4,0x30,0xa9,0x67,0x26,0xa7,0xa0,0x29,
0xeb,0x0e,0xc0,0x39,0xe0,0x37,0x6a,0x12,0x45,0x48,0x1a,0xeb,0x7e,0x06,0x8a,0x12,
0xb7,0x5f,0x55,0xd5,0xad,0x5f,0xe6,0xe5,0xb2,0x37,0xad,0x50,0xbc,0x9e,0x2d,0x91,
0xb7,0xce,0xaa,0x3a,0x23,0xfb,0xbb,0x04,0x49,0xe1,0x6f,0xdb,0xab,0x57,0xde,0x0a,
0xef,0xd8,0xd1,0x7f,0x36,0x28,0x1c,0x74,0x25,0x90,0xda,0xf8,0x3c,0x14,0x3d,
},
.gen2.reseed_ctr = 3,

.prnd = (unsigned char[]){
0x66,0xad,0x04,0x8b,0x4d,0x2d,0x00,0x32,0x23,0xc6,0x4d,0xd9,0x82,0x7c,0xc2,0x2e,
0xd3,0xec,0x8f,0xcb,0x61,0x20,0x9d,0x19,0x96,0x19,0x17,0x75,0x92,0xe9,0xb8,0x92,
0x26,0xbe,0x30,0xb1,0x93,0x0b,0xdd,0x74,0x9f,0x30,0xed,0x09,0xda,0x52,0xab,0xaa,
0x2e,0x59,0x9a,0xfa,0xf9,0x19,0x03,0xe7,0xa2,0xb5,0x9f,0xfb,0x8f,0xd4,0x70,0xe6,
0x60,0x44,0x85,0xa2,0x7c,0x20,0x0d,0x37,0x5f,0xef,0xf6,0x21,0x11,0x85,0x95,0xa7,
0xa3,0x05,0x7b,0x7e,0x31,0xea,0xdc,0x06,0x87,0xb1,0x00,0x8c,0x3c,0xb2,0xc7,0x43,
0x5a,0x57,0x04,0xb1,0xa1,0xa6,0xa3,0x48,0x7d,0x60,0xfd,0x14,0x79,0x3c,0x31,0x48,
0x6a,0xf7,0x65,0xce,0x2c,0xe1,0x82,0xde,0x88,0x11,0x24,0x45,0xdd,0x5f,0xf1,0x1b,
0x25,0x6c,0xfd,0xa0,0x70,0x18,0xb9,0x5f,0x97,0xed,0xba,0xb4,0xe4,0xc3,0x9c,0xa0,
0x97,0xc4,0x2f,0x9d,0xce,0x80,0xcd,0x3f,0x32,0x67,0x7f,0x3c,0x22,0x4a,0x86,0xb3,
0x15,0xd0,0x2e,0x37,0x7d,0xca,0x8f,0x37,0x85,0xe9,0x74,0x8f,0xfd,0xbe,0x3f,0xca,
0xa3,0xb0,0xc6,0xbf,0x00,0x1b,0x63,0xb5,0x74,0x26,0x83,0x63,0x58,0xe9,0xb3,0x15,
0xc6,0x71,0x8e,0x0b,0x74,0xfb,0x82,0xb9,0xbf,0x3d,0xf7,0x00,0xa6,0x41,0xab,0x94,
0x11,0xd1,0xb9,0xfb,0xa4,0x23,0x09,0xa8,0x4b,0xef,0x67,0xa1,0x42,0x04,0xf3,0x16,
0x0e,0xd1,0x6a,0x54,0x97,0xfe,0x21,0x1a,0xa1,0xf5,0xd3,0xae,0x4b,0x85,0x8b,0x6d,
0x44,0x5f,0x1d,0x09,0x45,0x43,0xd0,0x10,0x7c,0xe0,0x4e,0xf1,0xd1,0xba,0x33,0xab,
},
},
{
.no_reseed = false,
.pr = true,
.entropy_len = 256 / 8,
.nonce_len = 128 / 8,
.pers_len = 0,
.add_len = 0,
.prnd_len = 2048 / 8,

.inst.entropy = (unsigned char[]){
0x73,0xc9,0xb1,0x15,0xb7,0xef,0xb0,0xa6,0x32,0x44,0xd7,0x49,0x3a,0xe5,0x82,0x05,
0x99,0xd7,0xce,0xe5,0xca,0x05,0x4d,0xb2,0xf7,0x26,0x9b,0xa7,0xf6,0x21,0xbd,0xca,
},
.inst.nonce = (unsigned char[]){
0xc2,0x04,0xe6,0xde,0x78,0x9b,0x03,0x94,0xfb,0xbe,0x66,0x63,0x46,0x6e,0xfc,0xea,
},
.inst.v = (unsigned char[]){
0xf7,0xba,0x3c,0xf1,0x65,0xbe,0x38,0x8f,0x1e,0x05,0x02,0xac,0x63,0x92,0x02,0x21,
0xd2,0x7c,0x05,0x1e,0x44,0x7e,0xc3,0xd3,0x08,0x91,0x4d,0xa7,0xfe,0x17,0x6c,0xc7,
0xfb,0x91,0x46,0xdc,0x86,0x0e,0x5a,0x5a,0xda,0x53,0xda,0x7d,0x8b,0xb8,0xf6,0x7a,
0xc8,0xe2,0x5f,0xe9,0x44,0xd4,0x53,0xf0,0x99,0x75,0x6b,0xc5,0xae,0x58,0xef,0xbf,
0x80,0x2d,0x07,0x83,0x01,0xf8,0xdf,0xad,0x9c,0x9e,0xd4,0xb3,0x3c,0x56,0x0f,0x43,
0xe4,0x9b,0xf4,0xfb,0xd3,0x53,0x48,0x00,0xac,0xeb,0xd1,0xf2,0xc3,0x81,0x77,0x2b,
0x21,0x33,0xe4,0xff,0x46,0x9e,0xa4,0x99,0x65,0xd9,0xc2,0x57,0x58,0xf2,0xab,
},
.inst.c = (unsigned char[]){
0x20,0xc0,0xbc,0xf1,0x5a,0x1e,0xd3,0xb5,0x89,0x0f,0xb3,0xc7,0xc7,0x55,0xfe,0x2f,
0x57,0x45,0x93,0x0d,0xa0,0xf6,0x53,0x03,0x04,0xe3,0xed,0xfa,0xec,0x45,0x69,0xc8,
0x79,0x60,0x64,0x7f,0xa1,0xda,0xc7,0x1c,0x31,0x61,0xe4,0xaf,0xde,0x6b,0x87,0xd4,
0x1f,0x1b,0x1b,0x9a,0xbf,0x7f,0x5b,0x5f,0x63,0xa6,0x7c,0xd6,0xa6,0x05,0xaf,0xf2,
0x31,0x1d,0x7b,0x46,0xea,0xdd,0xf9,0x75,0x29,0x9a,0x13,0x2c,0xc3,0xc2,0x72,0xcc,
0x4e,0x6f,0x48,0xe2,0x31,0xcf,0x1c,0x75,0x07,0xbf,0xaf,0x32,0x86,0xcd,0xc9,0x2a,
0xe9,0x67,0xe4,0x31,0xc5,0xef,0x70,0xb1,0x9e,0x06,0x6b,0x51,0x9f,0xbc,0x3a,
},
.inst.reseed_ctr = 1,

.gen1.entropy = (unsigned char[]){
0xcf,0xce,0xf3,0x77,0x6b,0x37,0x64,0x9a,0x7f,0x6d,0x2b,0x48,0xf4,0x43,0xda,0x79,
0xa2,0xf2,0xf8,0x1d,0x04,0xf3,0xaf,0x98,0x53,0xa9,0xe6,0x96,0xc4,0x48,0x74,0x40,
},
.gen1.v = (unsigned char[]){
0xda,0xd2,0x1f,0x2d,0x3f,0xda,0x88,0x4e,0x22,0xda,0x80,0x0f,0xaa,0xee,0x0e,0x2e,
0xb5,0xb0,0x21,0x2f,0x4a,0x37,0x4d,0x7c,0x6f,0x94,0x6e,0x4f,0x7f,0xad,0x8a,0x71,
0xe9,0x3e,0xb0,0xb8,0xd7,0xea,0xeb,0x55,0x44,0x7c,0x51,0x0c,0xe4,0x41,0xa9,0x9c,
0x8b,0xf8,0x2b,0xe1,0xa7,0x70,0x8b,0x43,0x43,0xaf,0x52,0xc3,0xad,0x8f,0xbc,0x2f,
0x29,0x6b,0x29,0xac,0x89,0xeb,0x46,0x9c,0x71,0xa4,0x1f,0x80,0x02,0xdb,0xe4,0x9f,
0xf2,0xac,0x70,0x93,0xb0,0x61,0xb7,0xf7,0x62,0xa3,0x3b,0xf8,0x4c,0xe7,0x19,0x02,
0x40,0xe8,0xa3,0xfd,0x9c,0x02,0x66,0x3b,0xcb,0x80,0xdd,0xac,0x63,0xac,0x98,
},
.gen1.c = (unsigned char[]){
0x5d,0x6d,0x9b,0xb3,0x5c,0xf2,0x29,0x19,0xda,0xc5,0x05,0x76,0x96,0x64,0x23,0xbf,
0xb4,0x62,0x32,0x3d,0x1a,0x04,0x3c,0x00,0xa8,0x56,0x50,0x63,0xfc,0x8a,0x8e,0x2d,
0x75,0x99,0x26,0xf7,0xb4,0xda,0x65,0x33,0xe0,0x9b,0xe2,0xbe,0xd5,0x33,0x51,0xe4,
0x68,0xec,0x32,0x50,0x63,0x0f,0x4d,0xd9,0x4e,0xc8,0xa6,0xbb,0x20,0xa2,0x74,0x9b,
0x18,0x32,0xfc,0x5e,0xa5,0x4f,0xa5,0xed,0xe8,0x70,0x31,0xa6,0xbe,0x21,0xfe,0xa5,
0x65,0x73,0x7f,0xb7,0xf5,0x07,0xac,0x22,0xbd,0xe3,0x83,0xad,0x69,0xcc,0xe9,0x59,
0xbd,0xf8,0xea,0xef,0x21,0xf3,0x9e,0xff,0xa5,0x25,0x59,0x6b,0x89,0xf1,0xda,
},
.gen1.reseed_ctr = 2,

.gen2.entropy = (unsigned char[]){
0xd0,0x63,0x8e,0x28,0xca,0xe8,0xd1,0xc0,0xf5,0x72,0x09,0xd6,0x77,0xd8,0x89,0xd1,
0x95,0xa6,0x72,0x02,0x3c,0xb8,0xad,0xe3,0x9f,0x79,0x49,0x89,0xe1,0xda,0xee,0x34,
},
.gen2.v = (unsigned char[]){
0xb3,0xc0,0x3b,0x5a,0x64,0x2f,0xc9,0x4f,0xb6,0x63,0xc4,0x7c,0xc7,0xbc,0xcf,0x61,
0x81,0x5d,0x8e,0x94,0xf8,0x2a,0x3b,0x59,0x03,0x17,0xed,0x7e,0xa6,0x02,0xb4,0xde,
0x8c,0x27,0xdc,0x9e,0x42,0x66,0xa7,0x98,0x57,0xc8,0xbd,0xb9,0x8c,0x84,0x9e,0xd0,
0x88,0x17,0x86,0x13,0xc4,0xda,0x20,0x99,0x93,0x20,0x42,0x33,0xbb,0xa7,0xc1,0xe8,
0x0a,0xe5,0xcb,0x6a,0xf9,0x0d,0x6b,0x7f,0x09,0x17,0xf8,0x0b,0x5b,0x0c,0xc8,0x54,
0x90,0xe3,0xa1,0xce,0x0e,0xe1,0x0f,0xb9,0x53,0x96,0x2a,0xd6,0x40,0x0f,0xf1,0x54,
0x28,0x6d,0xdf,0x29,0xd9,0xc0,0xa0,0x04,0x7b,0xf0,0x60,0x95,0xc8,0xa1,0x27,
},
.gen2.c = (unsigned char[]){
0x1f,0xff,0x76,0x2e,0x53,0x76,0x82,0x21,0xd1,0xc8,0x64,0xa1,0x8e,0xb4,0x84,0xd4,
0x69,0x30,0x50,0x11,0xf8,0x9d,0x1a,0x6c,0xca,0x4a,0x21,0x6f,0x64,0x01,0x03,0xe2,
0x19,0x65,0xbc,0xb2,0x5f,0xed,0x86,0xd4,0x0d,0xdb,0x2f,0xa9,0xbb,0xee,0x7f,0x81,
0x43,0x1c,0xe4,0xc9,0x98,0x53,0x09,0xd8,0x2a,0xc2,0xf1,0x33,0xa9,0x31,0x6f,0xd3,
0x42,0xb2,0x64,0x96,0xcb,0xe0,0x89,0x3f,0x53,0x6c,0xbc,0x52,0x8a,0x02,0x11,0x6e,
0xf7,0xf4,0x47,0x90,0x29,0x0a,0x4a,0x57,0x41,0x7e,0xfb,0xc5,0xd2,0xc7,0xb7,0x48,
0xad,0x1a,0x4e,0x6e,0x13,0xff,0x32,0xfb,0x83,0x12,0x5b,0x9c,0x0f,0xd6,0x47,
},
.gen2.reseed_ctr = 2,

.prnd = (unsigned char []){
0x04,0x74,0x4d,0x1d,0x42,0x60,0x19,0x95,0xfa,0x3b,0x10,0x1d,0xed,0x3d,0x25,0x31,
0xcb,0xf4,0x5a,0xfd,0x83,0x12,0x0d,0x58,0xeb,0x26,0x59,0x4a,0x86,0x3b,0xd8,0x31,
0x83,0x11,0xb0,0x8d,0x3d,0xf4,0xc5,0x71,0xa9,0xc2,0x6d,0xff,0x63,0xa3,0xe9,0x91,
0x3a,0x9a,0x17,0xa7,0xc4,0x55,0x18,0x6f,0xdf,0xdd,0x90,0xc6,0x64,0xa8,0x4b,0x73,
0xa1,0x10,0x6a,0x5a,0x82,0xf7,0x41,0xbd,0x4c,0x7a,0x48,0xbd,0x04,0x6c,0x26,0x8d,
0x89,0x19,0xef,0xc9,0x41,0xf8,0xb4,0x5a,0x3c,0x3d,0x89,0xcf,0x37,0x14,0x1b,0x5c,
0x41,0xb1,0x0f,0xf5,0x43,0xa6,0x92,0x62,0x72,0xd6,0x23,0xad,0x8e,0xcc,0xd0,0x26,
0x55,0x20,0x90,0xad,0xcf,0xac,0xb1,0x24,0xf4,0x7c,0x4a,0xd6,0x2b,0xe9,0x0e,0xa5,
0xa0,0xa7,0x08,0x7d,0x81,0x45,0x84,0x45,0x81,0x3a,0xf8,0x8f,0xfb,0x5a,0x8c,0x35,
0x19,0xf9,0x77,0x13,0x1c,0xc8,0x51,0xcb,0x44,0x54,0xb0,0xa7,0x56,0xc8,0x37,0x3f,
0x05,0x23,0x82,0x43,0x5a,0xb9,0x34,0x71,0x8c,0x95,0x51,0x77,0x36,0x33,0x89,0xc0,
0x6b,0x0b,0x50,0x73,0x47,0x8e,0x84,0xd2,0x53,0xff,0x02,0xa3,0xf1,0xbe,0xf1,0xbb,
0xf1,0x33,0x8f,0x77,0xf9,0x2f,0x02,0x9f,0x63,0x8a,0x46,0x91,0xc4,0x8c,0x47,0x0d,
0x30,0xd2,0x30,0xf0,0x07,0xf5,0x45,0xe0,0x22,0xf6,0x6c,0x78,0xa1,0x30,0x69,0x78,
0x14,0xaa,0x55,0xd2,0x00,0x0a,0x49,0x55,0x3b,0xef,0x35,0xfa,0xb5,0x80,0x8e,0x2f,
0x3c,0xbb,0x38,0xc4,0x05,0x61,0x1f,0xa8,0x14,0x44,0x12,0x4e,0x3f,0x89,0xe1,0xe8,
},
},
{
.no_reseed = false,
.pr = true,
.entropy_len = 256 / 8,
.nonce_len = 128 / 8,
.pers_len = 0,
.add_len = 256 / 8,
.prnd_len = 2048 / 8,

.inst.entropy = (unsigned char[]){
0x7b,0xa8,0x43,0x1c,0x16,0xfb,0x16,0xd1,0x12,0x15,0x09,0xf2,0x25,0xab,0xd5,0xb3,
0x7a,0x25,0xb0,0xc5,0xc0,0x69,0xb7,0x45,0x4c,0x22,0x66,0x26,0x9c,0xf2,0x6e,0x1b,
},
.inst.nonce = (unsigned char[]){
0x7f,0x04,0x48,0x2b,0x3a,0x14,0xd8,0xfc,0x7a,0x29,0x81,0x6a,0x27,0xca,0x66,0xa2,
},
.inst.v = (unsigned char[]){
0xa3,0x30,0x40,0xd0,0x3d,0x1a,0x6b,0x5e,0xd9,0x31,0x9a,0x37,0xa0,0xac,0x8b,0x63,
0x91,0x0c,0x2b,0xb0,0x27,0x79,0x5f,0xd7,0x81,0xfb,0x9b,0xbf,0x8a,0xa9,0xdc,0xac,
0x73,0xb3,0x27,0x6c,0xdf,0x04,0x8d,0x05,0xd7,0x33,0x2c,0x65,0xd0,0x63,0xaf,0xac,
0x34,0x55,0x88,0x1e,0xf7,0x78,0xfc,0xc5,0xcf,0x12,0xf2,0x5f,0xcf,0x5f,0xe6,0xb3,
0xd4,0x1c,0xc3,0xf3,0xfb,0x0d,0x9f,0xd9,0x56,0x7d,0xa8,0x10,0x87,0x51,0xe5,0xef,
0x5b,0x54,0xb9,0xae,0x1e,0x2f,0xb3,0x00,0x58,0x6b,0x9f,0xce,0x2c,0x00,0xaf,0xb0,
0xf5,0xb1,0x97,0xd0,0x8d,0x16,0x17,0xf5,0x49,0x6b,0x7a,0xdb,0x2c,0x9f,0x2c,
},
.inst.c = (unsigned char[]){
0x8d,0x06,0xa0,0x3f,0x5d,0x2f,0x9a,0x9c,0x40,0xd1,0x6b,0x34,0xa5,0xd3,0x4e,0xa8,
0xcd,0x00,0x1b,0x79,0x77,0xfc,0xcd,0x67,0x2d,0xde,0x8b,0xff,0xe4,0x87,0xf6,0x3f,
0x60,0x89,0x9a,0xbf,0x3e,0x57,0xc4,0x2d,0x57,0xba,0x3f,0x00,0xdb,0x41,0x1e,0x57,
0x13,0x7f,0x14,0x12,0xf9,0x47,0xf3,0xc3,0xb0,0x31,0x56,0x99,0x27,0x1c,0xbf,0xc3,
0xa8,0xa1,0xf3,0x03,0x26,0x21,0xa0,0xa7,0xdf,0xc4,0x82,0xa9,0xc3,0x6a,0x8c,0x6a,
0x5a,0x15,0xb0,0x89,0xca,0x69,0xea,0x8d,0xe9,0xa9,0xcb,0x86,0x44,0x79,0x72,0x52,
0x19,0x32,0xc0,0xe1,0x47,0x45,0x5f,0xf3,0xcc,0x07,0x06,0x05,0x81,0xa7,0xb2,
},
.inst.reseed_ctr = 1,

.gen1.add = (unsigned char[]){
0x64,0x65,0x3b,0x87,0x53,0x59,0xf3,0xab,0x5d,0x96,0xaa,0x49,0x4d,0x44,0x66,0x90,
0x22,0xb4,0x8c,0xa5,0xc1,0xc1,0xf7,0x6a,0x18,0x14,0xb7,0x65,0x51,0x99,0xdb,0xf3,
},
.gen1.entropy = (unsigned char[]){
0xd0,0x52,0x01,0x36,0x5d,0x2b,0xcc,0xf9,0xa6,0x58,0x04,0xb2,0x3b,0xba,0x82,0xbf,
0xb6,0x55,0x96,0xd4,0x2f,0xc5,0x1c,0xc5,0x29,0x22,0x08,0x1d,0x6e,0x50,0x74,0xa2,
},
.gen1.v = (unsigned char[]){
0x05,0x3b,0x46,0xee,0x13,0x54,0x25,0xf9,0x99,0x32,0xb0,0x06,0x57,0x22,0x11,0xf3,
0xb5,0xa1,0x68,0x11,0x11,0x12,0x76,0x99,0xb9,0x26,0x6e,0xc3,0x73,0xed,0xf7,0x83,
0x60,0xfa,0x5f,0x7e,0xbf,0x3d,0x8b,0x22,0xc3,0xfe,0x78,0x51,0x48,0x83,0x25,0x37,
0xe7,0x33,0x5f,0x02,0x0b,0x59,0x5a,0x97,0x35,0xeb,0xc5,0xb7,0x26,0x66,0xf6,0xbf,
0x2b,0x50,0xa2,0x0a,0xe1,0xc0,0x5a,0x68,0xb7,0xfa,0xd6,0x0f,0x36,0x52,0x1c,0x46,
0x27,0x2b,0xff,0x70,0x7f,0x66,0xbb,0xfa,0x2a,0xb4,0x66,0x6f,0xfc,0x2a,0x56,0x9d,
0x8e,0xfc,0x30,0x27,0xfb,0x3b,0x48,0x83,0x04,0xfb,0x38,0xfb,0xd6,0x2e,0x16,
},
.gen1.c = (unsigned char[]){
0x60,0xdc,0x33,0x8e,0x05,0x60,0xbe,0x50,0x39,0x6b,0x94,0x57,0xd4,0x55,0x83,0xb3,
0x03,0x7e,0x2f,0xd0,0x91,0xd4,0xb2,0x1d,0x25,0xbb,0x2d,0xb4,0xde,0x15,0xc5,0xdf,
0x11,0xe9,0x00,0x84,0x4c,0xca,0x4c,0xe5,0x3f,0x38,0x8e,0x4d,0x9d,0xcb,0xa4,0x3e,
0x91,0xa8,0xc1,0x15,0xdc,0x0f,0x89,0x9a,0xfc,0x15,0xa6,0xa9,0xda,0x9b,0x73,0x4c,
0x52,0x18,0x7c,0xf2,0x6a,0xe9,0xc7,0x60,0x93,0xce,0x1e,0x91,0x27,0xf7,0xb8,0x2a,
0x16,0x42,0x9b,0x4f,0x05,0xde,0xdb,0x89,0x4f,0x1e,0x3f,0x93,0x79,0x52,0x3a,0x6c,
0x0b,0x06,0xd5,0x4d,0xda,0x25,0x65,0x8e,0xe8,0x49,0x5a,0x0e,0x0a,0x9f,0x91,
},
.gen1.reseed_ctr = 2,

.gen2.add = (unsigned char[]){
0x1e,0x50,0x04,0xc4,0xc2,0xf8,0x50,0xfa,0x6b,0xe2,0xe0,0x17,0x10,0xa4,0xdf,0x0e,
0x57,0x63,0x00,0xff,0x4f,0xc4,0xde,0x40,0xe9,0xc9,0x13,0x58,0xf2,0x6f,0xd4,0x08,
},
.gen2.entropy = (unsigned char[]){
0x59,0x18,0x5a,0x07,0xb2,0x0d,0xec,0x58,0x57,0xb7,0x9d,0xda,0xb2,0x8d,0x46,0xa5,
0x2e,0x4b,0x0a,0xad,0x9d,0xf9,0xcd,0xb7,0x76,0x13,0x38,0x54,0xaa,0xc0,0xdb,0x3e,
},
.gen2.v = (unsigned char[]){
0x2e,0x54,0xe1,0x54,0x44,0xb8,0x43,0x9e,0x2e,0x06,0x6c,0xf2,0x3d,0x94,0x88,0x0b,
0x72,0xe6,0xfd,0x3e,0xb4,0x3c,0xfd,0xa7,0xd8,0x91,0x0a,0xcb,0x89,0xc0,0xbd,0x76,
0x31,0x21,0x52,0xe6,0x91,0x18,0x9d,0x44,0x75,0x78,0xc2,0x7c,0x7b,0xdf,0xb0,0x7c,
0x2c,0x4a,0x0e,0x49,0x87,0xe3,0xf7,0x96,0xcd,0xa9,0xa0,0x0f,0x99,0xb9,0x5e,0x3d,
0xb9,0xa1,0x9a,0xf2,0xdc,0x24,0x1b,0x71,0x4d,0xba,0x0a,0x86,0x14,0xa4,0xa7,0x28,
0xe5,0xbc,0xb9,0x92,0xf8,0x1e,0x49,0x6c,0xa4,0x68,0x8a,0xdd,0x84,0x37,0x58,0x89,
0x52,0xbf,0x8b,0x16,0x7c,0xc6,0xdb,0x30,0x81,0x00,0xd8,0xf6,0xa3,0x1c,0x77,
},
.gen2.c = (unsigned char[]){
0x06,0x3f,0x41,0xe9,0x7b,0xe2,0x0f,0xcf,0x7f,0x74,0xa3,0xad,0x28,0x3c,0x6c,0x4e,
0x77,0xe3,0x43,0x1f,0x4d,0x36,0x09,0x10,0x54,0xa7,0xd3,0xf6,0x4d,0x45,0x07,0xc5,
0x25,0x66,0x4e,0x2b,0xd4,0x45,0xe2,0xb8,0x2c,0x6a,0xf4,0x37,0xf7,0x4f,0xe8,0x02,
0x91,0x9c,0x4a,0x09,0x30,0xe8,0x65,0x5e,0xe2,0x0d,0x9c,0x6c,0x7d,0x9c,0x7c,0x2f,
0xb1,0xab,0xdc,0xa2,0x58,0xb7,0x5d,0x22,0x3e,0xb0,0xfd,0xbc,0xea,0x7e,0x68,0xb8,
0xa8,0xe6,0x21,0x5f,0xea,0xee,0x03,0xcf,0x1f,0xed,0x7e,0x7c,0x6b,0x2a,0x4f,0xb9,
0x3e,0x1b,0xdb,0x43,0x97,0x0d,0x6c,0xa2,0xfd,0x97,0x03,0x65,0x05,0x51,0x76,
},
.gen2.reseed_ctr = 2,

.prnd = (unsigned char[]){
0x16,0x23,0x2b,0x52,0x84,0xea,0x1a,0x8a,0xad,0xca,0x65,0x67,0x79,0x08,0xba,0x74,
0xc9,0x70,0xab,0x6c,0x34,0x6b,0xe4,0xe6,0x78,0x96,0x25,0xc5,0xdc,0x10,0xe5,0xac,
0xa1,0xe6,0xb8,0x53,0xfe,0x17,0x4b,0xc1,0x16,0xa0,0x58,0x2b,0xa7,0xe0,0x7b,0x20,
0xf6,0x98,0xea,0xbf,0xb9,0x1c,0x89,0xce,0x5b,0x87,0x23,0x26,0x36,0xbe,0xf1,0x6e,
0x5a,0x96,0x6d,0x00,0x73,0xd4,0xe9,0x2a,0x89,0x73,0x5c,0x39,0x7d,0xbb,0xdf,0x80,
0xbc,0xf9,0xc3,0x10,0xa5,0xae,0xc9,0x8a,0xd0,0x77,0x64,0x3a,0xde,0x1d,0x87,0x37,
0xd4,0x08,0xb7,0x88,0xd9,0x0c,0x6b,0x53,0x8d,0x10,0xda,0x03,0x91,0xd9,0xea,0x87,
0xe7,0x4a,0x7f,0x8d,0x05,0x27,0xb1,0xd0,0xed,0xf7,0x60,0xc6,0x9a,0xcf,0xac,0xca,
0xdd,0x29,0xbc,0x69,0xb2,0x74,0xcf,0x08,0xfd,0x2a,0x50,0x6d,0x8d,0x58,0x8b,0x4d,
0xee,0xfc,0x67,0xa3,0x1a,0xeb,0x43,0x83,0x1c,0x09,0x6e,0x3c,0xa7,0x06,0x6b,0x43,
0x45,0xb2,0x69,0x16,0xd3,0xc5,0xf8,0x31,0x82,0x40,0x6b,0x44,0x8c,0xea,0x0f,0xb0,
0xd1,0xc6,0x20,0xd8,0x75,0xa1,0x07,0xa6,0xd8,0xd0,0xd3,0xb1,0x2e,0xab,0xb4,0x04,
0xe8,0x0c,0x86,0x08,0xe7,0x14,0xac,0x43,0xd6,0xe9,0x62,0xc9,0x7e,0xf2,0x2e,0x88,
0xc9,0xac,0x66,0x2d,0x48,0x4d,0x24,0xe2,0xa8,0xcf,0x03,0x98,0xe0,0x91,0x85,0xe3,
0x36,0x80,0xb1,0x25,0x95,0xe9,0x77,0x62,0x37,0x14,0xb3,0xf7,0x09,0xb6,0xfd,0xdc,
0x18,0x61,0xc5,0x3c,0x4a,0xfd,0xb3,0xfd,0xed,0xf7,0xdc,0x4d,0xcf,0x5b,0xb0,0xe2,
},
},
{
.no_reseed = false,
.pr = true,
.entropy_len = 256 / 8,
.nonce_len = 128 / 8,
.pers_len = 256 / 8,
.add_len = 0,
.prnd_len = 2048 / 8,

.inst.entropy = (unsigned char []){
0x62,0xba,0x7f,0x80,0xb7,0x78,0xdf,0x98,0x6a,0x17,0x68,0x7f,0xc2,0xd2,0x41,0xc1,
0x7d,0x8a,0xe0,0x9d,0x26,0xef,0x1a,0xfe,0x06,0x95,0x7b,0xfc,0x1e,0x94,0x14,0xcd,
},
.inst.nonce = (unsigned char []){
0x1a,0xcd,0x0e,0xd5,0xaa,0xa6,0xb9,0x6a,0xb1,0x7e,0x23,0x20,0x87,0xb7,0x1d,0xe6,
},
.inst.pers = (unsigned char []){
0x5e,0x44,0x97,0x60,0x37,0xe2,0x92,0x13,0x53,0x14,0x1e,0x0e,0xcf,0x0c,0x56,0x20,
0x6b,0xd8,0x5c,0x05,0x19,0x43,0xe7,0xb8,0x10,0xbd,0x0e,0xf8,0xe5,0xd3,0xa0,0x2a,
},
.inst.v = (unsigned char []){
0x93,0x2b,0xba,0x3b,0x03,0xb0,0x11,0xad,0xef,0x52,0x87,0xc1,0x49,0xc5,0x42,0x57,
0x03,0x15,0x9f,0x64,0x1a,0xad,0x52,0xf9,0xd9,0x44,0xf4,0x25,0xf5,0x4c,0x49,0x72,
0xd8,0x7e,0x01,0xe8,0xf2,0x66,0xcc,0x61,0xdc,0xaa,0x0d,0x9d,0x4e,0xc9,0xc6,0xb9,
0x2b,0x7e,0x6a,0x29,0x63,0xf4,0x53,0x78,0xbf,0x63,0x19,0x27,0x63,0xca,0xcf,0x58,
0x01,0xc3,0x7d,0x47,0x6e,0xf3,0x9a,0x4b,0x7a,0x2f,0x44,0x13,0x44,0xbe,0x96,0xeb,
0x72,0x04,0xb3,0xd7,0xd4,0x20,0x2f,0xbf,0x67,0xae,0x8a,0x57,0x1a,0x66,0x61,0xe2,
0x86,0x34,0x8c,0xaa,0xc9,0x13,0x8b,0x89,0x68,0x7d,0x9f,0x8e,0x94,0x02,0xb6,
},
.inst.c = (unsigned char []){
0x43,0x45,0xf7,0x0e,0x8e,0x44,0x16,0xf0,0x92,0xdd,0xe1,0x9b,0x10,0x74,0x12,0x33,
0xec,0x6b,0xe1,0x57,0x54,0x47,0xb9,0xb6,0xff,0x3e,0x06,0xc9,0x4c,0x61,0x0d,0xfc,
0xf7,0xd8,0xc3,0xc8,0x25,0x23,0x38,0x89,0x0f,0x5d,0xf7,0x05,0x4d,0x2e,0x3e,0xc2,
0xa2,0xf0,0x7f,0x6a,0x01,0x30,0x20,0x6b,0x8b,0x87,0x51,0x7e,0x8f,0x0b,0xec,0xe7,
0x5c,0x31,0x34,0x78,0x9f,0xea,0x34,0x88,0x04,0xa7,0x27,0x27,0xe9,0xe4,0xe1,0x13,
0x9e,0xe5,0xf6,0x0e,0x90,0x3c,0x15,0xbc,0x0f,0xed,0xf9,0x09,0xb3,0x0e,0x08,0x8d,
0xeb,0xa7,0x2f,0x77,0x57,0x21,0xa8,0x15,0x1c,0x30,0x1f,0xda,0xd7,0x01,0x71,
},
.inst.reseed_ctr = 1,

.gen1.entropy = (unsigned char []){
0x7b,0xd0,0xf0,0x37,0x1b,0x81,0x79,0xbc,0x40,0x48,0x10,0xdc,0xcf,0xa9,0xc5,0x53,
0x2f,0xa7,0xae,0xca,0x7b,0x41,0x5d,0xb7,0xee,0x31,0x7d,0x5a,0x0c,0xf6,0x3c,0xb9,
},
.gen1.v = (unsigned char []){
0x95,0x17,0xc5,0xa2,0x6a,0x1d,0x73,0xf3,0x37,0xbe,0x44,0xc7,0x65,0x63,0x49,0x6f,
0x9a,0xf2,0xb6,0x75,0x46,0xf4,0x83,0xb1,0x77,0x67,0x99,0xe4,0xc0,0x08,0xe2,0xcf,
0xfd,0xef,0x37,0x59,0xa8,0x08,0x69,0x76,0xce,0x78,0x13,0x15,0x3c,0x37,0xd3,0x21,
0x8b,0x4c,0xba,0xe2,0xb6,0x90,0x19,0xc9,0x93,0xa7,0x5d,0x52,0x6e,0xff,0xe0,0x5e,
0x0c,0x0d,0x65,0x19,0xea,0x42,0x60,0x48,0xbb,0x39,0x79,0x44,0x64,0x59,0x69,0x5c,
0x53,0x11,0x6b,0x42,0x84,0xa6,0xab,0x41,0xff,0xea,0x18,0xe9,0x7d,0x0d,0xe8,0x7a,
0xee,0xf1,0x08,0xa0,0xe9,0xd8,0x48,0x93,0x85,0x02,0x98,0xd0,0xa8,0xad,0x6b,
},
.gen1.c = (unsigned char []){
0x76,0x3d,0x01,0xa4,0xef,0xc5,0x04,0xf8,0x40,0x8d,0xc6,0x12,0xbc,0x33,0xc2,0x61,
0x40,0x24,0x79,0x6e,0x2e,0x56,0x26,0x11,0x3b,0xae,0x91,0xb2,0xc3,0x76,0x6e,0x64,
0x8a,0x6a,0xac,0x83,0x98,0x89,0xc4,0x8a,0x0c,0xa5,0x1d,0xa8,0xec,0xa3,0x39,0xbb,
0xee,0x80,0xd0,0xc1,0x62,0x78,0xbd,0x85,0x88,0x10,0x74,0x7b,0x3b,0x9d,0xdb,0x2b,
0xe0,0x7a,0x5d,0x1e,0xbe,0xb1,0xf4,0x35,0x20,0x45,0x3b,0xb4,0x00,0x87,0x29,0xf1,
0x69,0x33,0x46,0xd4,0x27,0x27,0x0c,0x37,0x7e,0x09,0x0a,0xb2,0x3f,0x12,0x0a,0xf6,
0x66,0x50,0x65,0x79,0xa8,0x5b,0x6c,0xd1,0x75,0xea,0x7a,0xfa,0x6e,0x1d,0x74,
},
.gen1.reseed_ctr = 2,

.gen2.entropy = (unsigned char []){
0xe1,0x0b,0xed,0x57,0x82,0xd4,0xe9,0xd1,0x51,0x86,0x67,0x14,0xde,0xe9,0x97,0x97,
0xb6,0x40,0x9f,0xec,0x81,0xf3,0x14,0xfa,0x22,0xbb,0xe6,0x46,0xcd,0xb9,0x25,0xa9,
},
.gen2.v = (unsigned char []){
0xdc,0x49,0x58,0x39,0x3b,0x4b,0x3e,0xb0,0xc2,0xdc,0x24,0x87,0x55,0xf0,0x20,0xa0,
0xd7,0xa8,0x9f,0x44,0xd9,0x4e,0xc5,0x3a,0x89,0x1b,0xd4,0x0f,0x2c,0x23,0x22,0xc2,
0xc1,0x7f,0x5c,0x9b,0x34,0x5d,0xd0,0x27,0x21,0xf7,0xe5,0x06,0x09,0xd7,0x27,0x58,
0x8a,0x6b,0x4a,0x0b,0x6b,0x98,0xe5,0x87,0xd6,0x96,0x7c,0xa7,0x5b,0xc3,0x84,0x9a,
0x6b,0xc6,0x3d,0x75,0xe6,0xb5,0x38,0x6c,0x86,0x57,0xf4,0x12,0xc4,0xd1,0x4b,0x9f,
0x00,0x95,0xa6,0x03,0x86,0x89,0x27,0xa0,0xd2,0x07,0x79,0xdb,0xa9,0x1c,0x20,0xf0,
0xb7,0xdb,0xc1,0x11,0x6e,0x5a,0x73,0x03,0x3c,0xae,0xbf,0x6c,0xfd,0xae,0xb8,
},
.gen2.c = (unsigned char []){
0xef,0x08,0x9b,0x11,0xdb,0x37,0xf5,0xa7,0xb9,0x6b,0xb4,0x9f,0x7a,0xff,0x41,0x56,
0x41,0x93,0x33,0x36,0xa9,0x85,0xd4,0x7c,0x3e,0xe6,0x06,0x02,0xbd,0x9f,0xa7,0x0d,
0x6a,0x46,0x05,0x55,0x05,0xcc,0x31,0x6f,0x0f,0x88,0x59,0x12,0x7c,0x81,0x6c,0x77,
0x47,0x23,0x16,0x58,0x5d,0x14,0x5d,0xf3,0xc4,0x6f,0xa8,0x14,0x96,0x56,0x06,0x16,
0x62,0x93,0xd0,0x7f,0x2d,0x1f,0x90,0x63,0x6f,0x68,0xd3,0xe7,0xaa,0x41,0x04,0x7b,
0x61,0x01,0x28,0x74,0x70,0xce,0x14,0xbc,0x15,0xb7,0x22,0xe4,0xcd,0xd0,0xf5,0x37,
0xf0,0x76,0xfd,0x02,0x47,0x6d,0x7e,0x9c,0x70,0x4b,0xf1,0xb7,0x31,0xdc,0xb0,
},
.gen2.reseed_ctr = 2,

.prnd = (unsigned char []){
0xde,0x3d,0xdd,0xcb,0xfc,0x99,0xc2,0x25,0x8a,0x98,0xbd,0x04,0x2e,0x57,0x15,0x99,
0xa3,0x11,0xd8,0x6b,0x5f,0x23,0xfe,0x50,0xe8,0x9a,0x42,0x58,0x8d,0x51,0x60,0x6f,
0x51,0xb3,0x5e,0xc7,0xd4,0xe7,0x5a,0x49,0x26,0x16,0x35,0x6a,0x63,0x5e,0x7b,0x2d,
0x3f,0x1e,0xea,0xa5,0xdb,0x19,0xb8,0xfc,0xb7,0xa9,0x4e,0x27,0x4f,0x36,0x9e,0x73,
0xa6,0x7b,0xc2,0x8a,0x8b,0xf5,0xd4,0x95,0x59,0xfa,0xdc,0xda,0xfd,0x1f,0xaf,0x30,
0xca,0x52,0x41,0x10,0x12,0xa5,0xe3,0xff,0xb6,0x68,0x65,0xf4,0x1c,0x0c,0x98,0x80,
0x50,0x61,0x44,0x6e,0x62,0xfe,0x69,0x79,0xbc,0x5d,0x09,0x5f,0x1d,0x87,0x2b,0x96,
0x37,0x5e,0x33,0x5c,0xde,0x53,0x80,0x31,0x98,0x8f,0xc4,0xec,0x32,0xab,0xe4,0xda,
0xb5,0xff,0x18,0xf4,0x5f,0xb5,0x12,0x7a,0x14,0x07,0x9f,0x0f,0x0a,0x82,0x51,0x93,
0x20,0x65,0xb1,0x6a,0xf2,0x8d,0xd4,0xb0,0x62,0x07,0x3d,0xcb,0x0b,0xa1,0xb2,0x04,
0x95,0x97,0xdf,0x45,0xc1,0x7c,0x0b,0xc0,0x6a,0x23,0xd2,0x4b,0x8f,0xdd,0x66,0x97,
0x8a,0x24,0x39,0x8b,0x5d,0x42,0x63,0x7c,0xbb,0x53,0xd1,0x98,0xa5,0x6b,0x61,0x93,
0xb3,0x57,0x52,0x1d,0xb7,0x11,0x7b,0x7b,0x44,0x2e,0x75,0xbf,0x16,0x9e,0x42,0xbb,
0x07,0x98,0xfa,0x59,0xfa,0xdc,0x19,0xf5,0xd9,0xe1,0x05,0x0a,0x43,0x45,0x35,0x8b,
0xd7,0xe4,0x0f,0x90,0x96,0x45,0xd8,0x05,0x20,0xa0,0x93,0xb2,0x78,0x90,0xa3,0xd1,
0x6e,0xd2,0x27,0x79,0x90,0xc3,0xa4,0xb3,0x6a,0x3a,0x0c,0x5f,0x37,0x45,0xec,0x59,
},
},
{
.no_reseed = false,
.pr = true,
.entropy_len = 256 / 8,
.nonce_len = 128 / 8,
.pers_len = 256 / 8,
.add_len = 256 / 8,
.prnd_len = 2048 / 8,

.inst.entropy = (unsigned char []){
0x19,0xff,0x49,0x7e,0x90,0x6d,0x08,0x11,0xa0,0xb7,0xf7,0x75,0x86,0x18,0xaf,0xab,
0x44,0xfc,0xe8,0xed,0x35,0x38,0xab,0xfd,0x10,0x1c,0x58,0x6a,0x9c,0xea,0xf6,0x2d,
},
.inst.nonce = (unsigned char []){
0x49,0x80,0xcb,0x99,0x83,0x87,0x0a,0x35,0xd8,0x00,0xff,0xf9,0x32,0x78,0x40,0xc5,
},
.inst.pers = (unsigned char []){
0x2e,0x34,0xbe,0xa4,0xc1,0x28,0x43,0x4a,0xa2,0x85,0x9c,0xc3,0x39,0x13,0x71,0x6b,
0xc8,0x05,0xa1,0x83,0xbb,0x17,0x62,0xfc,0x3a,0xb5,0x44,0x89,0xd9,0xbd,0x1c,0x56,
},
.inst.v = (unsigned char []){
0x6c,0xf5,0x7e,0xdf,0x5f,0xec,0x61,0x34,0x5e,0xf8,0xcc,0x5e,0x39,0x1d,0xcc,0x3f,
0x88,0x0a,0xd5,0x51,0xaa,0xfb,0x16,0x37,0x57,0x4e,0x92,0xc1,0xfa,0x8f,0x3e,0xd8,
0x15,0x19,0xa0,0xf1,0xe9,0x32,0xa1,0xf8,0xb2,0xa5,0x53,0xb0,0xfa,0xb9,0xf9,0x9e,
0x99,0x51,0xee,0x49,0x9c,0x59,0xfd,0x23,0xf3,0x1a,0x1b,0x73,0x8f,0x3a,0x54,0x7d,
0xca,0xb6,0x36,0x8e,0x80,0x2c,0x57,0x89,0x6a,0x4f,0xcf,0xeb,0x3e,0xd0,0xe6,0x26,
0x02,0x97,0xd6,0x16,0x5f,0xb2,0xe4,0x10,0x88,0x6f,0xe2,0xaf,0x76,0xe5,0x0e,0x20,
0x23,0xc2,0xf2,0xcd,0xb3,0xf7,0xa8,0x1f,0x48,0x9a,0x76,0xfd,0x46,0x57,0x0e,
},
.inst.c = (unsigned char []){
0x70,0xc9,0xfc,0xe3,0xd5,0x7e,0x39,0x4b,0x6a,0xc5,0xc0,0x04,0x8b,0xfd,0xae,0x4e,
0xb5,0xa3,0xed,0x6a,0xa2,0xb2,0xfa,0xf5,0xe6,0xc5,0xb7,0xc2,0x9e,0xc6,0x54,0x3c,
0x47,0xf5,0x0e,0x7b,0xe8,0x49,0x66,0xb5,0xa6,0x73,0x9a,0xa3,0x19,0x1c,0xa3,0xb2,
0x1b,0xee,0xb4,0x16,0xaa,0x8f,0x7a,0x10,0xd0,0xee,0x15,0x46,0xf5,0x51,0x20,0x3d,
0xd3,0xca,0xce,0x74,0xb6,0xaf,0x8d,0xce,0xb4,0x82,0xe7,0x53,0x50,0x0f,0xf4,0x17,
0x65,0xdd,0xde,0xee,0x3a,0x72,0x5d,0x83,0xbf,0x5b,0xc0,0xda,0xff,0x35,0x86,0xc3,
0xf5,0x75,0x0f,0x76,0xde,0x15,0x3c,0x24,0xda,0x52,0xfe,0xfe,0x50,0xb4,0x24,
},
.inst.reseed_ctr = 1,

.gen1.add = (unsigned char []){
0x47,0xdd,0x5c,0x72,0xea,0x44,0xdb,0xf1,0xd2,0x35,0x04,0x0b,0x18,0xa8,0x21,0xc2,
0x0f,0xb4,0x0b,0x88,0xc0,0x39,0x1a,0x5c,0x16,0x3f,0x9c,0x90,0x2f,0x6c,0x6e,0xb7,
},
.gen1.entropy = (unsigned char []){
0xbe,0x45,0xca,0x98,0x78,0x24,0x7b,0x00,0x8e,0x5d,0xe8,0xc5,0x88,0xb6,0xcb,0x84,
0x04,0x8c,0x0d,0x90,0x35,0x7d,0xbd,0xfc,0x98,0xe9,0xe2,0x87,0x91,0x2f,0xdb,0x33,
},
.gen1.v = (unsigned char []){
0x2b,0xec,0x9f,0xc2,0x78,0x1d,0xb4,0xcb,0xfa,0x98,0x50,0x18,0x15,0x5b,0xdf,0x51,
0x02,0xe3,0x1b,0x0f,0x34,0x0c,0x32,0x9a,0xa0,0xa5,0xca,0x1a,0x70,0xb6,0xc3,0x90,
0x6a,0x6a,0x8e,0x03,0xba,0x62,0xfd,0x2f,0xd8,0x1d,0xc9,0xc2,0xc0,0xdc,0x86,0xea,
0x8f,0x8d,0xea,0x1f,0xb5,0x38,0xa0,0xdb,0xf5,0x69,0x8e,0x27,0xcc,0x2d,0x83,0xf8,
0x0a,0xfb,0x4a,0xfe,0xfc,0x46,0x8c,0xda,0xf6,0xeb,0xce,0x06,0xee,0x35,0xcd,0xd0,
0x59,0x77,0xb2,0x32,0x53,0x40,0xd1,0x91,0xff,0x03,0xb2,0xc4,0x33,0x90,0x8c,0x31,
0xf2,0xbc,0x28,0xd6,0x23,0x72,0xaa,0xd9,0x67,0x1d,0xe0,0x59,0x8a,0x35,0x09,
},
.gen1.c = (unsigned char []){
0xcf,0xf7,0xfc,0x70,0x9b,0x81,0x74,0x33,0x71,0x24,0x54,0xd6,0xbc,0xd5,0x74,0xde,
0xae,0x3f,0x2b,0xc2,0xc2,0x00,0xf0,0x88,0x38,0x8b,0x07,0x37,0xfa,0x0d,0xfd,0x8f,
0x7c,0x31,0x1b,0x8b,0x67,0xd3,0x7e,0x0a,0x7f,0x2a,0x55,0x1c,0x69,0x86,0x33,0xcd,
0x4e,0x01,0x56,0x4d,0x36,0x72,0xae,0x9c,0x01,0xdb,0x90,0x6d,0xff,0xc2,0xa4,0xc3,
0xf0,0x45,0x56,0x38,0xda,0x06,0x60,0x51,0x80,0x00,0x4b,0xb7,0xae,0x1d,0x01,0x86,
0x19,0xda,0x3d,0xdf,0xab,0xef,0xf7,0x37,0x11,0xe2,0x5c,0xff,0xf0,0x65,0x77,0x2a,
0xcf,0xa0,0xd6,0x79,0x54,0xbe,0xe4,0x02,0xc0,0x41,0xf6,0xc7,0x82,0x60,0xf3,
},
.gen1.reseed_ctr = 2,

.gen2.add = (unsigned char []){
0xcf,0x20,0x40,0xe9,0x04,0x6a,0x69,0xdd,0x96,0x38,0xde,0x94,0x1f,0x00,0x90,0xb7,
0x53,0x5c,0x51,0xcf,0xa9,0xf1,0xc7,0xbb,0x2a,0x56,0xa3,0x32,0x32,0x69,0x43,0xd3,
},
.gen2.entropy = (unsigned char []){
0xb8,0x71,0x61,0x1f,0x8f,0xcb,0x8c,0x86,0x0a,0x72,0xc4,0xfd,0x40,0x6d,0x49,0x39,
0x33,0x5a,0x03,0x1e,0x0d,0xe9,0xf2,0xd4,0x36,0xd4,0x73,0x6b,0x6b,0x06,0x0c,0x2d,
},
.gen2.v = (unsigned char []){
0x59,0x44,0xf8,0xec,0xc9,0xa4,0x4c,0x0f,0x05,0x3f,0xfe,0x7e,0xa3,0x64,0x27,0xfc,
0xa5,0x6f,0x26,0xa3,0xd0,0x96,0x68,0xff,0x08,0x52,0xde,0x82,0x2c,0xa1,0x1a,0x88,
0x74,0x68,0xd6,0x36,0x25,0xf1,0xe8,0x6f,0xa8,0xaa,0x2a,0x12,0xc9,0x45,0x59,0xc5,
0xa4,0x01,0xe1,0x9f,0x8c,0xd5,0x8a,0xd9,0x53,0x60,0x12,0x8b,0x85,0x2f,0xef,0xbe,
0xbe,0x6f,0x8b,0x9e,0x96,0x7f,0x80,0xd1,0x84,0x9d,0x8b,0xb7,0x9c,0x0e,0x34,0x9b,
0x74,0x27,0xdf,0x85,0xa5,0x52,0xd5,0x48,0x7f,0xeb,0xd5,0x72,0xc1,0x17,0x8e,0x23,
0x2b,0x2c,0xd6,0x1b,0x2b,0xf9,0x67,0x45,0xba,0xe5,0xd4,0x60,0x64,0x0b,0x56,
},
.gen2.c = (unsigned char []){
0x34,0x0e,0xb7,0x02,0xa2,0x0b,0x75,0xed,0x0a,0x68,0x57,0x39,0x4d,0xcb,0x10,0xfc,
0xc2,0xad,0xd4,0xfa,0xf4,0x98,0x94,0xbe,0x67,0xd6,0xf2,0x03,0xf7,0x7b,0x71,0xd0,
0x35,0xe7,0xa7,0x37,0xee,0x24,0xdc,0x3b,0x6e,0x22,0xb5,0x95,0xf4,0xc0,0x86,0xb9,
0x2d,0x57,0xab,0x21,0x18,0xb9,0xa5,0x0d,0xad,0x6d,0x1f,0xf4,0xb4,0xfd,0x7b,0x3b,
0x51,0x09,0x88,0xed,0x49,0x93,0x9e,0xec,0x82,0x2b,0xf9,0x14,0x93,0x61,0x2b,0xe6,
0x54,0x66,0x5b,0x27,0x8e,0xab,0x9e,0x6d,0x8b,0xeb,0x25,0xf3,0xe2,0x08,0xd8,0xc3,
0x68,0x60,0x5a,0x39,0x9a,0x94,0x7e,0x13,0xa0,0x02,0xe3,0x67,0xed,0x50,0x3c,
},
.gen2.reseed_ctr = 2,

.prnd = (unsigned char []){
0x2d,0x99,0x0f,0x0d,0xe4,0x3d,0x3a,0x4b,0x29,0x30,0x54,0x2c,0x27,0xad,0x27,0x45,
0x8e,0x88,0x65,0xca,0x6b,0x8f,0x27,0xfd,0x7a,0x96,0x9c,0xf4,0xe2,0xa0,0x32,0x3e,
0x38,0xfe,0x6f,0x50,0x5a,0x2d,0xba,0x48,0x8e,0xa6,0xb0,0x43,0x65,0x20,0x9c,0x6d,
0xb7,0x86,0xcb,0xbf,0x0a,0x7c,0x73,0xb4,0xfd,0x56,0xd2,0x49,0x87,0x71,0x9d,0xb0,
0xfd,0xba,0x1a,0x3f,0x07,0x14,0x95,0x21,0xdc,0xf5,0xb7,0x75,0x9c,0x61,0x0d,0xa2,
0x2d,0x15,0x10,0x57,0xac,0xef,0xe7,0x0d,0xf1,0xcc,0xae,0xb6,0x7a,0x97,0x51,0x59,
0xb8,0x99,0x6a,0xca,0x93,0xd7,0xa4,0x80,0x96,0x92,0x6d,0xb4,0x38,0x1b,0xbc,0xe4,
0x81,0x27,0x7d,0x7a,0xb2,0x7c,0xbc,0x03,0x88,0xf0,0xb7,0xce,0xdb,0xbf,0xb8,0x42,
0x1c,0xb1,0xdc,0x5f,0x2a,0x9c,0x67,0x7f,0x62,0xac,0xf9,0x6a,0xb2,0x5e,0x7e,0x40,
0x6c,0xe8,0x2f,0x5b,0x96,0xbc,0xb4,0x71,0xaf,0xbd,0xf4,0xb3,0xf5,0xa6,0xfb,0xcb,
0x8d,0xa4,0x5d,0x22,0x58,0xe3,0x50,0xe7,0x7d,0x46,0x33,0xb0,0xc1,0xda,0x69,0x16,
0x62,0xdd,0x86,0x99,0x09,0xdc,0xfd,0x7c,0x8e,0xd0,0xf5,0x4b,0xa7,0xaf,0x0f,0x9c,
0x03,0x8e,0xb3,0x2d,0x32,0xb7,0x05,0xe5,0x1b,0x35,0xbb,0x3c,0x2e,0xef,0xf0,0x10,
0xbb,0x47,0xee,0x32,0x6c,0x23,0x18,0xb5,0xbc,0xda,0x96,0x3c,0x2d,0xad,0x41,0x9c,
0x59,0x23,0xe3,0x68,0xd9,0xb2,0x8f,0x25,0xb0,0x48,0xa8,0x7b,0xdb,0xa0,0xa9,0x0d,
0x98,0xc2,0x4c,0x81,0xb6,0xdb,0xde,0x0f,0x58,0x05,0x4a,0x41,0xa8,0x29,0x3a,0x65,
},
},
{
.no_reseed = false,
.pr = false,
.entropy_len = 256 / 8,
.nonce_len = 128 / 8,
.pers_len = 0,
.add_len = 0,
.prnd_len = 2048 / 8,

.inst.entropy = (unsigned char []){
0x31,0x44,0xe1,0x7a,0x10,0xc8,0x56,0x12,0x97,0x64,0xf5,0x8f,0xd8,0xe4,0x23,0x10,
0x20,0x54,0x69,0x96,0xc0,0xbf,0x6c,0xff,0x8e,0x91,0xc2,0x4e,0xe0,0x9b,0xe3,0x33,
},
.inst.nonce = (unsigned char []){
0xb1,0x6f,0xcb,0x1c,0xf0,0xc0,0x10,0xf3,0x1f,0xea,0xb7,0x33,0x58,0x8b,0x8e,0x04,
},
.inst.v = (unsigned char []){
0x3a,0x85,0xca,0x10,0xea,0xc6,0x83,0xd6,0xa9,0x27,0x05,0x94,0xd1,0x7f,0x33,0xa2,
0x1d,0xad,0x7b,0x9b,0x25,0x9c,0x2a,0x17,0x44,0x62,0xa5,0xe0,0xc9,0x09,0xa1,0x33,
0xdb,0x84,0xb4,0xee,0x2b,0xdb,0x0f,0x72,0xcd,0xce,0xf7,0xd6,0x28,0x54,0xe5,0x35,
0x46,0x84,0x52,0x28,0x5d,0xbe,0x8e,0x46,0xbe,0xd3,0x96,0x5d,0xc9,0xc6,0x69,0x52,
0xde,0xfa,0x48,0x87,0x94,0x93,0xed,0xc0,0x1b,0xc0,0x7e,0xd4,0x97,0x3c,0x11,0x5c,
0xfd,0xd9,0x94,0x7a,0x70,0x84,0x65,0x35,0x1b,0x78,0xb8,0x04,0x65,0x2e,0xc7,0xcb,
0xe7,0xf6,0xe2,0xa0,0x91,0x93,0xfa,0x35,0x2f,0xf9,0x91,0xd3,0x8c,0x94,0xac,
},
.inst.c = (unsigned char []){
0x74,0xea,0x43,0x7c,0x49,0x12,0x6f,0xf3,0x61,0xfe,0xab,0x56,0x39,0xa8,0xad,0x31,
0x8d,0x45,0x5c,0x94,0xb3,0xf9,0x99,0xff,0x16,0x06,0xf5,0x92,0xc2,0x7f,0x8b,0xf0,
0xbe,0x56,0x2c,0x7b,0xff,0xa2,0x97,0xde,0x85,0x12,0xef,0x44,0xb0,0xdf,0xc8,0xdb,
0x5c,0xb1,0x7c,0x96,0x92,0xac,0x0d,0x80,0xf0,0x66,0x96,0x1e,0x64,0x26,0x08,0x41,
0x08,0x08,0x9e,0xee,0x4a,0x75,0x9d,0x53,0x09,0xec,0x86,0x16,0x68,0xdd,0xeb,0x1c,
0x31,0xce,0xef,0x26,0xed,0xad,0x67,0x8b,0x6f,0x36,0xc3,0xeb,0xcb,0x9c,0x93,0x6c,
0xaf,0xce,0xe3,0xd9,0xa9,0x6a,0xe6,0x55,0x4e,0x22,0xd4,0x28,0x88,0xab,0x07,
},
.inst.reseed_ctr = 1,

.res.entropy = (unsigned char []){
0xa0,0xb3,0x58,0x4c,0x2c,0x84,0x12,0xf6,0x18,0x40,0x68,0x34,0x40,0x4d,0x1e,0xb0,
0xce,0x99,0x9b,0xa2,0x89,0x66,0x05,0x4d,0x7e,0x49,0x7e,0x0d,0xb6,0x08,0xb9,0x67,
},
.res.v = (unsigned char []){
0xb3,0x7f,0x9a,0xa3,0x9c,0x5a,0x80,0xdf,0x56,0xc0,0x40,0x40,0x24,0x07,0x96,0x0e,
0xf6,0xf8,0x89,0x2d,0x1a,0x68,0x8f,0xfc,0x93,0xba,0xd6,0xeb,0xe6,0xaf,0x44,0xd5,
0x5c,0xcd,0x66,0xc1,0xf4,0x4e,0xb5,0x31,0xe9,0xda,0xc1,0xc9,0x44,0x76,0x81,0xd7,
0xb2,0x7b,0x2b,0x70,0x3b,0x49,0x00,0x32,0x69,0x6b,0x32,0x33,0x0b,0x5e,0xdd,0x12,
0x3e,0x5e,0xce,0x7c,0x40,0xef,0xe7,0x0a,0x29,0x82,0x2e,0xa8,0xe4,0xe4,0x54,0xbb,
0x72,0x08,0x5c,0x6b,0x03,0x7a,0x86,0x52,0xec,0x22,0x7f,0x89,0x9d,0xd0,0x14,0x55,
0xdb,0x8e,0xe7,0xb6,0xb2,0xe9,0x21,0x14,0xf6,0xf9,0xfb,0x67,0x8e,0x63,0x32,
},
.res.c = (unsigned char []){
0x90,0x8a,0xd8,0x58,0xdb,0x2c,0x5d,0x21,0xfa,0x1c,0xd8,0x60,0x21,0x7b,0xd7,0x5a,
0xd0,0xba,0x1d,0xf2,0xfd,0x24,0xe3,0x03,0x96,0x4c,0x01,0x11,0x3a,0x0b,0x02,0x4a,
0x1e,0x53,0x64,0x0d,0x5a,0xe3,0x39,0x04,0x0b,0x43,0x57,0xc1,0xf3,0xc0,0xbe,0x2f,
0x14,0x60,0x7b,0x13,0x85,0xe9,0x68,0x18,0x3c,0x53,0xec,0xd9,0xa3,0x3d,0xdb,0x04,
0xb3,0xac,0x36,0xdf,0xc1,0x35,0x3d,0x85,0x71,0x15,0x9a,0x0b,0x31,0xb8,0x1b,0x5d,
0x3d,0xe2,0x4b,0x8a,0xe6,0x53,0x0c,0x83,0x8f,0xa8,0x71,0x2e,0xa5,0xd4,0xd5,0x87,
0x63,0xf2,0xbe,0x0a,0xb1,0x98,0x99,0x87,0xc5,0x6b,0xfd,0x31,0x5d,0xf5,0x21,
},
.res.reseed_ctr = 1,

.gen1.v = (unsigned char []){
0x44,0x0a,0x72,0xfc,0x77,0x86,0xde,0x01,0x50,0xdd,0x18,0xa0,0x45,0x83,0x6d,0x69,
0xc7,0xb2,0xa7,0x20,0x17,0x8d,0x73,0x00,0x2a,0x06,0xd7,0xfd,0x20,0xba,0x47,0x1f,
0x7b,0x20,0xca,0xcf,0x4f,0x31,0xee,0x35,0xf5,0x1e,0x19,0x8b,0x38,0x37,0x40,0xfb,
0x34,0x72,0x4a,0x07,0x47,0xe2,0x61,0xc8,0x00,0xfa,0x0f,0x74,0x4b,0xdc,0x84,0x2d,
0x37,0x19,0x9f,0x6a,0xcf,0x5f,0x4a,0xf0,0x41,0xa6,0x60,0x08,0x78,0xcf,0x72,0xa7,
0xce,0xaa,0x75,0x0f,0xa1,0xc2,0x35,0x46,0xf9,0x62,0xaf,0xe9,0x7c,0x05,0x56,0x83,
0xea,0xf5,0x13,0x1d,0x9f,0x9c,0x88,0x2e,0xdb,0x93,0xc5,0x0a,0xdb,0xa9,0x63,
},
.gen1.c = (unsigned char []){
0x90,0x8a,0xd8,0x58,0xdb,0x2c,0x5d,0x21,0xfa,0x1c,0xd8,0x60,0x21,0x7b,0xd7,0x5a,
0xd0,0xba,0x1d,0xf2,0xfd,0x24,0xe3,0x03,0x96,0x4c,0x01,0x11,0x3a,0x0b,0x02,0x4a,
0x1e,0x53,0x64,0x0d,0x5a,0xe3,0x39,0x04,0x0b,0x43,0x57,0xc1,0xf3,0xc0,0xbe,0x2f,
0x14,0x60,0x7b,0x13,0x85,0xe9,0x68,0x18,0x3c,0x53,0xec,0xd9,0xa3,0x3d,0xdb,0x04,
0xb3,0xac,0x36,0xdf,0xc1,0x35,0x3d,0x85,0x71,0x15,0x9a,0x0b,0x31,0xb8,0x1b,0x5d,
0x3d,0xe2,0x4b,0x8a,0xe6,0x53,0x0c,0x83,0x8f,0xa8,0x71,0x2e,0xa5,0xd4,0xd5,0x87,
0x63,0xf2,0xbe,0x0a,0xb1,0x98,0x99,0x87,0xc5,0x6b,0xfd,0x31,0x5d,0xf5,0x21,
},
.gen1.reseed_ctr = 2,

.gen2.v = (unsigned char []){
0xd4,0x95,0x4b,0x55,0x52,0xb3,0x3b,0x23,0x4a,0xf9,0xf1,0x00,0x66,0xff,0x44,0xc4,
0x98,0x6c,0xc5,0x13,0x14,0xb2,0x56,0x03,0xc0,0x52,0xd9,0x0e,0x5a,0xc5,0x49,0x69,
0x99,0x74,0x2e,0xdc,0xaa,0x15,0x27,0x3a,0x00,0x61,0x71,0x4d,0x2b,0xf7,0xff,0xb3,
0x2b,0x70,0x00,0xbf,0xde,0xb1,0x06,0x05,0xf3,0x61,0x74,0xeb,0x33,0xa4,0x8a,0x4c,
0xc0,0x07,0xc2,0x3b,0xb0,0x35,0x97,0xb4,0xd8,0xa6,0x37,0x3c,0xa7,0x03,0x7e,0x8a,
0x8f,0xf0,0x8f,0x63,0x77,0x9d,0xa9,0xe6,0x18,0x78,0xb1,0x88,0x6c,0xb0,0x84,0xba,
0x68,0xce,0xef,0x8a,0xd4,0xe5,0xba,0x77,0x20,0xac,0xbd,0x3b,0x26,0x28,0x22,
},
.gen2.c = (unsigned char []){
0x90,0x8a,0xd8,0x58,0xdb,0x2c,0x5d,0x21,0xfa,0x1c,0xd8,0x60,0x21,0x7b,0xd7,0x5a,
0xd0,0xba,0x1d,0xf2,0xfd,0x24,0xe3,0x03,0x96,0x4c,0x01,0x11,0x3a,0x0b,0x02,0x4a,
0x1e,0x53,0x64,0x0d,0x5a,0xe3,0x39,0x04,0x0b,0x43,0x57,0xc1,0xf3,0xc0,0xbe,0x2f,
0x14,0x60,0x7b,0x13,0x85,0xe9,0x68,0x18,0x3c,0x53,0xec,0xd9,0xa3,0x3d,0xdb,0x04,
0xb3,0xac,0x36,0xdf,0xc1,0x35,0x3d,0x85,0x71,0x15,0x9a,0x0b,0x31,0xb8,0x1b,0x5d,
0x3d,0xe2,0x4b,0x8a,0xe6,0x53,0x0c,0x83,0x8f,0xa8,0x71,0x2e,0xa5,0xd4,0xd5,0x87,
0x63,0xf2,0xbe,0x0a,0xb1,0x98,0x99,0x87,0xc5,0x6b,0xfd,0x31,0x5d,0xf5,0x21,
},
.gen2.reseed_ctr = 3,

.prnd = (unsigned char []){
0xef,0xa3,0x5d,0xd0,0x36,0x2a,0xdb,0x76,0x26,0x45,0x6b,0x36,0xfa,0xc7,0x4d,0x3c,
0x28,0xd0,0x1d,0x92,0x64,0x20,0x27,0x5a,0x28,0xbe,0xa9,0xc9,0xdd,0x75,0x47,0xc1,
0x5e,0x79,0x31,0x85,0x2a,0xc1,0x27,0x70,0x76,0x56,0x75,0x35,0x23,0x9c,0x1f,0x42,
0x9c,0x7f,0x75,0xcf,0x74,0xc2,0x26,0x7d,0xeb,0x6a,0x3e,0x59,0x6c,0xf3,0x26,0x15,
0x6c,0x79,0x69,0x41,0x28,0x3b,0x8d,0x58,0x3f,0x17,0x1c,0x2f,0x6e,0x33,0x23,0xf7,
0x55,0x5e,0x1b,0x18,0x1f,0xfd,0xa3,0x05,0x07,0x21,0x0c,0xb1,0xf5,0x89,0xb2,0x3c,
0xd7,0x18,0x80,0xfd,0x44,0x37,0x0c,0xac,0xf4,0x33,0x75,0xb0,0xdb,0x7e,0x33,0x6f,
0x12,0xb3,0x09,0xbf,0xd4,0xf6,0x10,0xbb,0x8f,0x20,0xe1,0xa1,0x5e,0x25,0x3a,0x4f,
0xe5,0x11,0xa0,0x27,0x96,0x8d,0xf0,0xb1,0x05,0xa1,0xd7,0x3a,0xff,0x7c,0x7a,0x82,
0x6d,0x39,0xf6,0x40,0xdf,0xb8,0xf5,0x22,0x25,0x9e,0xd4,0x02,0x28,0x2e,0x2c,0x2e,
0x9d,0x3a,0x49,0x8f,0x51,0x72,0x5f,0xe4,0x14,0x1b,0x06,0xda,0x55,0x98,0xa4,0x2a,
0xc1,0xe0,0x49,0x4e,0x99,0x7d,0x56,0x6a,0x1a,0x39,0xb6,0x76,0xb9,0x6a,0x60,0x03,
0xa4,0xc5,0xdb,0x84,0xf2,0x46,0x58,0x4e,0xe6,0x5a,0xf7,0x0f,0xf2,0x16,0x02,0x78,
0x16,0x6d,0xa1,0x6d,0x91,0xc9,0xb8,0xf2,0xde,0xb0,0x27,0x51,0xa1,0x08,0x8a,0xd6,
0xbe,0x4e,0x80,0xef,0x96,0x6e,0xb7,0x3e,0x66,0xbc,0x87,0xca,0xd8,0x7c,0x77,0xc0,
0xb3,0x4a,0x21,0xba,0x1d,0xa0,0xba,0x6d,0x16,0xca,0x50,0x46,0xdc,0x4a,0xbd,0xa0,
},
},
{
.no_reseed = false,
.pr = false,
.entropy_len = 256 / 8,
.nonce_len = 128 / 8,
.pers_len = 0,
.add_len = 256 / 8,
.prnd_len = 2048 / 8,

.inst.entropy = (unsigned char []){
0xc7,0x3a,0x78,0x20,0xf0,0xf5,0x3e,0x8b,0xbf,0xc3,0xb7,0xb7,0x1d,0x99,0x41,0x43,
0xcf,0x6e,0x98,0x64,0x2e,0x9e,0xa6,0xd8,0xdf,0x5d,0xcc,0xbc,0x43,0xdb,0x87,0x20,
},
.inst.nonce = (unsigned char []){
0x20,0xcc,0x98,0x34,0xb5,0x88,0xad,0xcb,0x1b,0xbd,0xe6,0x4f,0x0d,0x2a,0x34,0xcb,
},
.inst.v =(unsigned char []){
0x85,0x2d,0x1b,0x4f,0xdd,0x41,0xcd,0x7e,0x0f,0x59,0x7c,0x45,0xc8,0xe4,0xb4,0x01,
0xa5,0xfe,0xcf,0x92,0x29,0xb6,0x07,0x24,0x51,0xca,0x65,0xb5,0x28,0x98,0x82,0xc6,
0x86,0xe7,0x91,0x99,0x22,0xce,0x82,0xde,0x2f,0xaa,0xc8,0x3c,0xd4,0xc4,0xed,0xdf,
0xa2,0xcd,0xcf,0x62,0x44,0xa4,0xd2,0xac,0xdd,0x34,0xc0,0x23,0x21,0x36,0x40,0x9b,
0xb5,0x0e,0xa2,0x4d,0x0c,0x33,0xfc,0xfd,0x1a,0xaf,0x1c,0xc1,0x10,0xb5,0x35,0x3d,
0x32,0xe4,0xe6,0xdf,0x59,0xae,0x25,0xec,0x12,0x40,0x00,0xde,0x62,0xfc,0xfa,0x8b,
0xb4,0xcb,0x3f,0x3b,0x72,0xe2,0xda,0x20,0x66,0xef,0x00,0xcd,0x66,0xd9,0xe9,
},
.inst.c = (unsigned char []){
0xf7,0xb0,0xc9,0xcf,0x2c,0xcf,0x58,0xfd,0x8c,0x8b,0x69,0xda,0xa4,0xcf,0x24,0xa8,
0x74,0xc9,0x5b,0x57,0xa9,0xf5,0xbe,0x16,0xaa,0xa7,0x1e,0xc3,0x00,0x70,0xac,0x8f,
0x22,0x2f,0xe2,0x17,0x88,0xfe,0xc1,0x4b,0x8a,0x9a,0xd7,0xad,0x20,0x91,0x2c,0x05,
0xa6,0xf9,0x45,0x48,0x64,0x67,0x79,0xa1,0x6c,0x78,0x7b,0x13,0x5c,0xe8,0xd0,0x8c,
0x49,0xf7,0xe2,0x34,0xcb,0xd2,0xc7,0x73,0x35,0x71,0xf5,0xad,0x64,0x79,0xb5,0xfc,
0x50,0x40,0x34,0x96,0x58,0x1b,0x48,0x61,0xef,0x8e,0xc8,0x48,0xaf,0xfb,0xd2,0x07,
0x7a,0xb1,0x64,0xfc,0x6b,0xb2,0xdd,0x7b,0x00,0x8a,0x65,0x05,0x04,0xbf,0xd8,
},
.inst.reseed_ctr = 1,

.res.entropy = (unsigned char []){
0x12,0xdd,0x2a,0xca,0x88,0x79,0x04,0x6d,0x23,0x16,0x5c,0x60,0xf8,0xae,0xdc,0x20,
0x41,0x57,0x83,0xe1,0x56,0xd4,0x2a,0x94,0x34,0x68,0x26,0xaa,0xeb,0x02,0xea,0xcf,
},
.res.add = (unsigned char []){
0x9b,0x59,0xff,0x78,0xa3,0x4e,0xab,0xe0,0x06,0x0c,0x27,0x92,0xca,0x9b,0x49,0xe9,
0x78,0x1e,0x6b,0x80,0x2b,0xad,0xf7,0xdb,0xde,0x27,0xca,0xae,0xd3,0x34,0x37,0x06,
},
.res.v = (unsigned char []){
0x18,0x1a,0x30,0x23,0x52,0xd9,0xeb,0xf0,0xb6,0x69,0x73,0x0b,0x24,0x41,0xa9,0xf4,
0xc1,0x6a,0x4b,0x9d,0x25,0xeb,0xc8,0x4e,0xd0,0x1c,0x46,0x0d,0x29,0x3c,0xd3,0xe8,
0xb7,0xbf,0xf1,0xac,0xa3,0x2b,0x0e,0xa8,0xd2,0x81,0xdf,0x0e,0xf8,0xd1,0xae,0x09,
0xd4,0xcf,0x97,0x69,0x0c,0x94,0x4f,0x47,0x13,0xad,0xb9,0xed,0xe9,0x07,0x63,0xf3,
0xed,0x77,0x08,0x1c,0x37,0xc0,0xfc,0x60,0xf8,0xb6,0x0b,0x51,0x08,0xcf,0x62,0x76,
0xc8,0x0d,0xb1,0x4a,0x82,0xaa,0xef,0x1b,0xf8,0xda,0x03,0x78,0x14,0x45,0xcf,0xcc,
0x7c,0xdc,0x02,0xb1,0xc7,0xa2,0x74,0x08,0x74,0xdd,0x94,0x81,0x18,0xf7,0xef,
},
.res.c = (unsigned char []){
0x28,0xb6,0x38,0xd6,0x31,0xf0,0x54,0xeb,0xa5,0x62,0x32,0x0e,0x9d,0x15,0x1f,0x90,
0x58,0x63,0xdd,0x6c,0x04,0xd8,0xba,0x41,0x16,0x7b,0xcf,0x3b,0x02,0x36,0xd4,0xe5,
0xdd,0xe1,0xdc,0x7b,0xf6,0x90,0xe6,0x1b,0x4a,0x65,0x99,0x7b,0xd9,0xc6,0x7f,0xf9,
0x08,0xfe,0x7e,0x24,0x43,0xd0,0x1c,0x8e,0xac,0x15,0xb2,0xea,0x5c,0x80,0xba,0x89,
0xf0,0x9a,0xa9,0xb8,0xa8,0x1d,0x56,0x12,0x4b,0xb7,0x15,0x86,0x81,0x28,0x27,0xf4,
0x63,0xde,0x90,0x31,0x87,0x27,0x10,0x2d,0xbd,0x5e,0x59,0xca,0x5f,0x1a,0xf7,0x8a,
0xb7,0x38,0x44,0x69,0x5e,0xee,0x09,0x77,0xb7,0x54,0x85,0x4e,0x52,0x50,0x97,
},
.res.reseed_ctr = 1,

.gen1.add = (unsigned char []){
0xdc,0x74,0xa9,0xe4,0x80,0xa6,0xff,0x6f,0x6b,0xce,0x53,0xab,0x9c,0x7b,0xdd,0xe4,
0xb1,0x3d,0x70,0xfb,0x51,0x96,0xcd,0xd5,0xe3,0xa0,0x55,0x5c,0xcf,0x06,0xfe,0x91,
},
.gen1.v = (unsigned char []){
0x40,0xd0,0x68,0xf9,0x84,0xca,0x40,0xdc,0x5b,0xcb,0xa5,0x19,0xc1,0x56,0xc9,0x85,
0x19,0xce,0x29,0x09,0x2a,0xc4,0x82,0x8f,0xe6,0x98,0x15,0x48,0x2b,0x73,0xa8,0xce,
0x95,0xa1,0xce,0x28,0x99,0xbb,0xf4,0xc4,0x1c,0xe7,0x78,0x8a,0xd2,0x98,0x2e,0x3c,
0xea,0x32,0x66,0xf4,0xca,0xdc,0x50,0xae,0x52,0x8d,0xc6,0x1a,0xa7,0xc5,0x21,0x48,
0x98,0x69,0xe3,0xef,0xc6,0xc8,0x2c,0xce,0xfb,0xba,0xb4,0x56,0x73,0xe0,0xf5,0x9d,
0x56,0x54,0xcf,0x91,0x0f,0xa1,0x46,0xd9,0x84,0xa4,0x2c,0x5f,0x17,0xfb,0x60,0x34,
0x0c,0x86,0xd0,0xd0,0x7c,0x7e,0x2f,0x2e,0x6d,0xf3,0xcf,0xfd,0x72,0x2a,0x0e,
},
.gen1.c = (unsigned char []){
0x28,0xb6,0x38,0xd6,0x31,0xf0,0x54,0xeb,0xa5,0x62,0x32,0x0e,0x9d,0x15,0x1f,0x90,
0x58,0x63,0xdd,0x6c,0x04,0xd8,0xba,0x41,0x16,0x7b,0xcf,0x3b,0x02,0x36,0xd4,0xe5,
0xdd,0xe1,0xdc,0x7b,0xf6,0x90,0xe6,0x1b,0x4a,0x65,0x99,0x7b,0xd9,0xc6,0x7f,0xf9,
0x08,0xfe,0x7e,0x24,0x43,0xd0,0x1c,0x8e,0xac,0x15,0xb2,0xea,0x5c,0x80,0xba,0x89,
0xf0,0x9a,0xa9,0xb8,0xa8,0x1d,0x56,0x12,0x4b,0xb7,0x15,0x86,0x81,0x28,0x27,0xf4,
0x63,0xde,0x90,0x31,0x87,0x27,0x10,0x2d,0xbd,0x5e,0x59,0xca,0x5f,0x1a,0xf7,0x8a,
0xb7,0x38,0x44,0x69,0x5e,0xee,0x09,0x77,0xb7,0x54,0x85,0x4e,0x52,0x50,0x97,
},
.gen1.reseed_ctr = 2,

.gen2.add = (unsigned char []){
0x8f,0x3f,0x22,0x90,0x11,0x20,0x9b,0x2f,0x39,0x90,0x96,0xaf,0xb0,0x54,0xbc,0xcc,
0xa6,0xbc,0x46,0xaa,0xee,0x98,0x84,0x58,0x38,0xfb,0x1f,0xb7,0x8b,0x66,0xf3,0xbd,
},
.gen2.v = (unsigned char []){
0x69,0x86,0xa1,0xcf,0xb6,0xba,0x95,0xc8,0x01,0x2d,0xd7,0x28,0x5e,0x6b,0xe9,0x15,
0x72,0x32,0x06,0x75,0x2f,0x9d,0x3c,0xd0,0xfd,0x13,0xe4,0x83,0x2d,0xaa,0x7d,0xb4,
0x73,0x83,0xaa,0xa4,0x90,0x4c,0xda,0xdf,0x67,0x4d,0x12,0x06,0xac,0x5e,0xaf,0xa9,
0x9d,0xe1,0x30,0x4f,0xc0,0xb6,0xa1,0xb5,0xe3,0x2e,0x34,0xa7,0xf4,0x14,0x1e,0x89,
0x35,0x38,0x78,0xc0,0xd3,0xf6,0xa0,0xba,0x5b,0x9e,0xd4,0x52,0xd6,0x12,0x60,0xde,
0x9e,0x5a,0xcb,0xf8,0x13,0x44,0x85,0xb3,0xb9,0xe9,0x90,0xf5,0x9f,0x34,0xd4,0xd4,
0x33,0x07,0xe4,0x0a,0xd0,0xd0,0xa5,0x05,0xef,0xdb,0x24,0xb7,0x2f,0x80,0x7b,
},
.gen2.c = (unsigned char []){
0x28,0xb6,0x38,0xd6,0x31,0xf0,0x54,0xeb,0xa5,0x62,0x32,0x0e,0x9d,0x15,0x1f,0x90,
0x58,0x63,0xdd,0x6c,0x04,0xd8,0xba,0x41,0x16,0x7b,0xcf,0x3b,0x02,0x36,0xd4,0xe5,
0xdd,0xe1,0xdc,0x7b,0xf6,0x90,0xe6,0x1b,0x4a,0x65,0x99,0x7b,0xd9,0xc6,0x7f,0xf9,
0x08,0xfe,0x7e,0x24,0x43,0xd0,0x1c,0x8e,0xac,0x15,0xb2,0xea,0x5c,0x80,0xba,0x89,
0xf0,0x9a,0xa9,0xb8,0xa8,0x1d,0x56,0x12,0x4b,0xb7,0x15,0x86,0x81,0x28,0x27,0xf4,
0x63,0xde,0x90,0x31,0x87,0x27,0x10,0x2d,0xbd,0x5e,0x59,0xca,0x5f,0x1a,0xf7,0x8a,
0xb7,0x38,0x44,0x69,0x5e,0xee,0x09,0x77,0xb7,0x54,0x85,0x4e,0x52,0x50,0x97,
},
.gen2.reseed_ctr = 3,

.prnd = (unsigned char []){
0xe6,0xc9,0x64,0x42,0x58,0x28,0x11,0xec,0x90,0xe5,0x87,0x52,0x5f,0x36,0xc5,0x55,
0xe2,0xfd,0x63,0x61,0xa0,0xc5,0xb0,0x28,0x49,0x17,0xa4,0xfa,0x6f,0x6e,0x8a,0xce,
0x83,0xf1,0x1a,0x1f,0xb2,0x6c,0xea,0x66,0x92,0xb2,0x25,0xae,0x7c,0x5b,0xe2,0x86,
0xdd,0x27,0x47,0x1f,0x32,0x3d,0x7a,0x2e,0x44,0x31,0x72,0x2b,0xb3,0x37,0xb1,0xba,
0x0e,0x64,0x8e,0xa2,0xe9,0xf0,0x91,0x8b,0x50,0xe9,0x11,0x1f,0x23,0x77,0x63,0x6b,
0xa6,0x9b,0x0e,0x1c,0xb5,0x29,0x50,0x78,0xd7,0x6c,0x54,0x9c,0x86,0x56,0x94,0x0e,
0xb1,0x5c,0xa5,0xad,0xed,0x7a,0xdc,0x46,0xe6,0xfa,0x4b,0x86,0x94,0x8f,0x21,0x2f,
0xea,0x3f,0x3b,0xef,0xde,0xec,0xe8,0xb2,0x0e,0x42,0x0c,0xa8,0x4c,0x76,0x01,0x96,
0xdd,0xf0,0xb0,0x74,0xdf,0x0a,0x9f,0x09,0x7a,0x5d,0xb8,0xf6,0x12,0x58,0x00,0xf5,
0xfe,0x74,0x6a,0x62,0xdf,0x12,0x08,0x04,0x2f,0x12,0x55,0xb5,0x24,0x46,0x5a,0x17,
0xef,0xcf,0x6a,0x53,0x76,0x12,0x96,0x84,0x30,0xe2,0xad,0xcf,0xf3,0x0f,0x74,0x07,
0xa5,0x1e,0xd7,0x30,0x53,0x34,0x38,0x4e,0x51,0x2e,0x00,0x36,0x42,0xcc,0xa1,0x75,
0x63,0x68,0x19,0xf0,0x21,0xc7,0x6a,0x2f,0x44,0xe8,0x9e,0x6f,0xe3,0x9c,0xf1,0x64,
0x47,0x79,0x10,0x37,0x9c,0xd3,0x14,0xf7,0x35,0xc3,0x57,0xf9,0x37,0x9d,0xe2,0x24,
0x95,0x27,0x6b,0x40,0x1c,0x98,0xff,0xb0,0x9a,0x6d,0xc0,0x3e,0x48,0x4b,0x35,0x5a,
0x94,0x64,0x51,0x14,0x01,0xee,0xaa,0x05,0xb4,0x55,0x6e,0x73,0xb5,0x52,0x27,0xf8,
},
},
{
.no_reseed = false,
.pr = false,
.entropy_len = 256 / 8,
.nonce_len = 128 / 8,
.pers_len = 256 / 8,
.add_len = 0,
.prnd_len = 2048 / 8,

.inst.entropy = (unsigned char []){
0x12,0xc2,0x20,0x31,0xd0,0x3a,0x85,0x0f,0x76,0x96,0xc9,0x27,0xd4,0xdc,0x8d,0xad,
0x4c,0x3d,0xd7,0x17,0xe6,0x0e,0xe5,0x68,0x1b,0x05,0xdb,0x64,0x26,0xa0,0xe1,0x67,
},
.inst.nonce = (unsigned char []){
0xcd,0x32,0xe3,0xeb,0x3f,0x83,0x34,0xd9,0x69,0x8d,0xec,0x62,0x7b,0x2e,0xaa,0xbe,
},
.inst.pers = (unsigned char []){
0x75,0x63,0x63,0xf6,0x81,0x78,0xda,0xc0,0x9a,0x5c,0x8d,0x64,0xef,0xfb,0xab,0x23,
0x87,0x3a,0x3b,0xc2,0xdf,0xba,0xc3,0x9b,0x6d,0x47,0xeb,0xf9,0x29,0xad,0x98,0x54,
},
.inst.v =(unsigned char []){
0x29,0x5e,0x0b,0x18,0x92,0x1d,0x33,0xaa,0x64,0xe3,0xde,0xec,0x80,0x04,0x4b,0x41,
0xac,0x4e,0x49,0x90,0x04,0xc7,0x7b,0x7d,0x1a,0xc0,0x21,0xdd,0x1c,0x4e,0x2e,0x6a,
0x85,0x2d,0x6a,0xd6,0x27,0x6c,0xe2,0xa9,0x01,0x5f,0xa3,0x1e,0x22,0x34,0x96,0x76,
0x14,0x3d,0xa2,0x02,0xb7,0x7c,0xc0,0x08,0x26,0xea,0x17,0xec,0x67,0xc0,0xdb,0x12,
0x75,0xa2,0xd0,0x7c,0x0e,0xae,0x8a,0x87,0xdf,0xbe,0x0f,0xd5,0x34,0x7a,0x2b,0x38,
0x36,0x40,0x29,0xf4,0x38,0x8e,0x6e,0x32,0xe8,0x53,0xad,0xec,0x3a,0xe4,0x79,0xc4,
0x07,0x75,0xf7,0xab,0x17,0x08,0x09,0x90,0x57,0x2b,0xac,0x29,0x47,0x7a,0x31,
},
.inst.c = (unsigned char []){
0x24,0x70,0x96,0x11,0x1b,0xa9,0x0f,0x4f,0x56,0x11,0xe2,0x65,0x84,0xa9,0xf1,0x83,
0xc9,0x60,0xda,0xe8,0x5c,0x57,0x1a,0xa5,0xae,0xfc,0x13,0xe3,0xdd,0x04,0x96,0x8c,
0x53,0xa3,0xd0,0xb9,0xdd,0x69,0x9a,0xa6,0xf8,0x33,0x0c,0x5d,0xa4,0x44,0x7e,0x30,
0x5f,0xf8,0xa3,0xdc,0x21,0xbe,0xf0,0x27,0x07,0xc0,0x64,0xa8,0xeb,0xf4,0xa7,0x34,
0x0c,0xab,0x11,0x33,0x35,0x58,0x97,0xd1,0xea,0xc2,0x0f,0xf7,0xb8,0x10,0x7f,0xce,
0x28,0xf8,0xef,0x64,0xbf,0x6b,0x39,0x0d,0x58,0x9d,0xec,0x19,0xe7,0x47,0x65,0x03,
0x73,0x44,0x88,0x72,0x39,0xf3,0x0f,0x4b,0x2b,0x04,0xc8,0x11,0xe6,0x00,0x69,
},
.inst.reseed_ctr = 1,

.res.entropy = (unsigned char []){
0x17,0xed,0x31,0xbd,0xa6,0x4b,0x08,0xec,0xe5,0x0e,0xdf,0x5b,0x91,0xf6,0xa0,0x86,
0x2e,0x56,0x90,0x18,0x17,0x34,0xa0,0x13,0x4e,0x05,0xe3,0x66,0x64,0x0b,0x7e,0x85,
},
.res.v = (unsigned char []){
0xba,0xa7,0xae,0xbc,0x24,0x31,0x32,0x82,0x49,0x0e,0x40,0xd8,0x90,0x3d,0x2b,0x4c,
0x23,0xf3,0x0a,0xfc,0x59,0x8d,0x36,0x7f,0xa8,0xb5,0x70,0xc3,0xf9,0xa7,0x92,0x39,
0x9c,0xeb,0xff,0x96,0xb3,0xf4,0x58,0xd3,0xa8,0xd2,0xd8,0xb9,0xb1,0x40,0xdb,0x40,
0xd9,0x8d,0xcf,0x5a,0x44,0x8e,0xce,0xb7,0x6d,0x25,0x19,0xd2,0xbf,0xd5,0xef,0x0d,
0x1c,0x80,0x5f,0x50,0xbc,0xde,0xdb,0x38,0x24,0x8d,0x2e,0xd7,0xcd,0x66,0xab,0x07,
0xf9,0x7e,0x13,0xd1,0xd2,0x69,0x10,0xb8,0x5f,0x7e,0xec,0x8f,0x1b,0x26,0xdc,0x1f,
0x3a,0x23,0x02,0x91,0xa7,0x32,0x25,0xe3,0xf2,0xcf,0xb7,0xf4,0xde,0x76,0x24,
},
.res.c = (unsigned char []){
0xd2,0x96,0xd0,0x95,0x7e,0x86,0x06,0xb5,0xe2,0x7b,0x74,0xac,0x72,0x98,0x08,0x60,
0x4c,0x48,0x35,0x03,0x52,0x53,0x4c,0x3d,0x8b,0x1a,0xc0,0x55,0x1c,0x9e,0xfc,0x80,
0x8b,0x55,0x7d,0x8f,0x5c,0x7a,0x1b,0x62,0x7b,0x86,0x85,0xb5,0x87,0xa5,0x12,0x1e,
0x73,0xd5,0xbe,0x13,0x78,0xe8,0xea,0x03,0x5f,0xab,0x8a,0xab,0x96,0x2a,0xf8,0x6d,
0xc4,0x1f,0xda,0x58,0x8d,0xcc,0xde,0xfc,0xfa,0xaa,0x78,0xde,0x18,0xfc,0x7a,0x50,
0x63,0xaf,0x35,0xfd,0xb2,0x4e,0xf6,0x04,0xda,0x15,0x3e,0x64,0x87,0x8a,0xaa,0xf6,
0x37,0x9a,0x95,0xe7,0x63,0x57,0xe6,0xd5,0xfd,0xd4,0x7e,0x07,0x3a,0x0f,0xfe,
},
.res.reseed_ctr = 1,

.gen1.v = (unsigned char []){
0x8d,0x3e,0x7f,0x51,0xa2,0xb7,0x39,0x38,0x2b,0x89,0xb5,0x85,0x02,0xd5,0x33,0xac,
0x70,0x3b,0x3f,0xff,0xab,0xe0,0x82,0xbd,0x33,0xd0,0x31,0x19,0x16,0x46,0x8e,0xba,
0x28,0x41,0x7d,0x26,0x10,0x6e,0x74,0x36,0x24,0x59,0x5e,0x6f,0x38,0xe5,0xed,0x9a,
0xac,0x0e,0x71,0xaa,0xb9,0x38,0xce,0xf1,0x40,0x29,0x43,0xa4,0x47,0x38,0xdd,0xf5,
0x7b,0x90,0x74,0x21,0x41,0xda,0x08,0x8a,0x32,0x22,0x5c,0x24,0x96,0x82,0x4f,0x7b,
0xca,0x1d,0xf8,0x3b,0xaa,0x8d,0x04,0xfc,0x7e,0xa7,0x63,0x55,0x22,0x27,0xed,0xec,
0xd9,0x4c,0x71,0x3d,0x64,0x92,0x6b,0x13,0x7b,0x1c,0xa2,0x01,0x21,0x2f,0x2b,
},
.gen1.c = (unsigned char []){
0xd2,0x96,0xd0,0x95,0x7e,0x86,0x06,0xb5,0xe2,0x7b,0x74,0xac,0x72,0x98,0x08,0x60,
0x4c,0x48,0x35,0x03,0x52,0x53,0x4c,0x3d,0x8b,0x1a,0xc0,0x55,0x1c,0x9e,0xfc,0x80,
0x8b,0x55,0x7d,0x8f,0x5c,0x7a,0x1b,0x62,0x7b,0x86,0x85,0xb5,0x87,0xa5,0x12,0x1e,
0x73,0xd5,0xbe,0x13,0x78,0xe8,0xea,0x03,0x5f,0xab,0x8a,0xab,0x96,0x2a,0xf8,0x6d,
0xc4,0x1f,0xda,0x58,0x8d,0xcc,0xde,0xfc,0xfa,0xaa,0x78,0xde,0x18,0xfc,0x7a,0x50,
0x63,0xaf,0x35,0xfd,0xb2,0x4e,0xf6,0x04,0xda,0x15,0x3e,0x64,0x87,0x8a,0xaa,0xf6,
0x37,0x9a,0x95,0xe7,0x63,0x57,0xe6,0xd5,0xfd,0xd4,0x7e,0x07,0x3a,0x0f,0xfe,
},
.gen1.reseed_ctr = 2,

.gen2.v = (unsigned char []){
0x5f,0xd5,0x4f,0xe7,0x21,0x3d,0x3f,0xee,0x0e,0x05,0x2a,0x31,0x75,0x6d,0x3c,0x0c,
0xbc,0x83,0x75,0x02,0xfe,0x33,0xce,0xfa,0xbe,0xea,0xf1,0x6e,0x32,0xe5,0x8b,0x3a,
0xb3,0x96,0xfa,0xb5,0x6c,0xe8,0x8f,0x98,0x9f,0xdf,0xe4,0x24,0xc0,0x8b,0x00,0xb6,
0x46,0x6e,0x50,0xc6,0x6f,0x3f,0xa1,0x36,0x6f,0x7e,0xa2,0x0d,0x6d,0x4b,0xec,0xf1,
0x67,0x77,0xff,0x54,0xd6,0xea,0xc0,0xcb,0x8b,0x86,0xb0,0x2f,0x2b,0x88,0x5c,0xc7,
0x80,0x75,0x0a,0x8c,0xef,0x52,0x5f,0x0f,0xe1,0x25,0x8d,0xf1,0xb4,0xc2,0x53,0x45,
0x29,0xe6,0x8b,0xe9,0x3e,0xbb,0x30,0xb1,0x61,0x4d,0xb7,0x97,0x80,0xd9,0x9e,
},
.gen2.c = (unsigned char []){
0xd2,0x96,0xd0,0x95,0x7e,0x86,0x06,0xb5,0xe2,0x7b,0x74,0xac,0x72,0x98,0x08,0x60,
0x4c,0x48,0x35,0x03,0x52,0x53,0x4c,0x3d,0x8b,0x1a,0xc0,0x55,0x1c,0x9e,0xfc,0x80,
0x8b,0x55,0x7d,0x8f,0x5c,0x7a,0x1b,0x62,0x7b,0x86,0x85,0xb5,0x87,0xa5,0x12,0x1e,
0x73,0xd5,0xbe,0x13,0x78,0xe8,0xea,0x03,0x5f,0xab,0x8a,0xab,0x96,0x2a,0xf8,0x6d,
0xc4,0x1f,0xda,0x58,0x8d,0xcc,0xde,0xfc,0xfa,0xaa,0x78,0xde,0x18,0xfc,0x7a,0x50,
0x63,0xaf,0x35,0xfd,0xb2,0x4e,0xf6,0x04,0xda,0x15,0x3e,0x64,0x87,0x8a,0xaa,0xf6,
0x37,0x9a,0x95,0xe7,0x63,0x57,0xe6,0xd5,0xfd,0xd4,0x7e,0x07,0x3a,0x0f,0xfe,
},
.gen2.reseed_ctr = 3,

.prnd = (unsigned char []){
0xbc,0x5d,0x8d,0x11,0xbe,0x22,0x67,0x9e,0x33,0xf4,0x03,0x74,0x17,0x57,0x16,0xe6,
0x7c,0x8b,0x5f,0xe6,0x81,0x9a,0xc5,0x3a,0x9b,0x20,0x8d,0xc0,0x58,0x17,0x04,0x31,
0xff,0xc2,0x9e,0xbd,0x1a,0x81,0x51,0xca,0xa3,0xe9,0x15,0x6d,0x4c,0x7e,0x89,0xe3,
0x9c,0x12,0x4f,0x71,0x94,0x09,0x51,0x02,0xe8,0x69,0x31,0x06,0x74,0x47,0x1a,0x7f,
0x1d,0xad,0x4e,0x58,0xf4,0x78,0x6e,0x96,0xb7,0xaa,0x1a,0xd7,0xa5,0x11,0x59,0x23,
0xba,0x01,0xd4,0xe7,0xa6,0x0a,0x8f,0x11,0xee,0x9c,0x47,0x26,0x6c,0x0f,0x1a,0xe4,
0x34,0x16,0x8b,0x7b,0x1f,0xb6,0x1e,0xc0,0xfc,0x29,0x2c,0x6c,0x2d,0x3a,0x87,0x78,
0xdc,0x7b,0x88,0x16,0x42,0xfb,0x8a,0x3e,0x9f,0xa5,0xff,0x37,0x20,0xf7,0x00,0xff,
0x89,0x00,0x1d,0x21,0xe9,0x7c,0x61,0xc2,0x46,0xdd,0x4f,0x87,0xbd,0x8a,0x64,0xfb,
0xcb,0x92,0x01,0x4d,0x52,0xf6,0xe6,0x41,0x83,0xbd,0xca,0x84,0xec,0x25,0xed,0x35,
0x24,0xb9,0xab,0xc8,0x6d,0xf2,0xbc,0x4d,0xff,0x2b,0x76,0x29,0x98,0x55,0xde,0x61,
0xb7,0xda,0x7e,0xde,0xc0,0x27,0x89,0x3a,0xb4,0xed,0xd0,0xd6,0xea,0x63,0x48,0xe7,
0x61,0x0e,0x6d,0x94,0x0a,0xf4,0x22,0x54,0x63,0x88,0x68,0x59,0xea,0x4f,0x5c,0x53,
0xfe,0xa2,0xc3,0x98,0xac,0x2f,0xe7,0x4a,0x9b,0x31,0x8b,0x11,0x5d,0xd4,0x6b,0xbe,
0xc6,0x88,0x4f,0x07,0x78,0x35,0xaa,0x95,0x15,0x0e,0xf0,0xb3,0xed,0x34,0xd6,0xd5,
0xb1,0x44,0xff,0x1c,0x1e,0x23,0x88,0x48,0x3d,0x9b,0x5f,0xca,0x8a,0x2c,0x5b,0xf3,
},
},
{
.no_reseed = false,
.pr = false,
.entropy_len = 256 / 8,
.nonce_len = 128 / 8,
.pers_len = 256 / 8,
.add_len = 256 / 8,
.prnd_len = 2048 / 8,

.inst.entropy = (unsigned char []){
0x2c,0x48,0x2c,0xd3,0x92,0xd9,0xb8,0x67,0x7f,0xf3,0x19,0xa7,0xad,0x53,0x9c,0xbd,
0x8a,0x24,0xeb,0xdc,0x2b,0x8d,0xbc,0xeb,0xbc,0x1d,0x1d,0x0d,0x0b,0xcb,0x69,0x8a,
},
.inst.nonce = (unsigned char []){
0x27,0xcb,0x9f,0xff,0xcc,0xdd,0x7a,0x3b,0x28,0x56,0xfe,0xdd,0x68,0x17,0xf1,0xaa,
},
.inst.pers = (unsigned char []){
0xbe,0x95,0x71,0x17,0x54,0xbe,0xb1,0xb2,0x5a,0x46,0xd5,0x04,0xd1,0x9b,0xbf,0x44,
0x23,0x2c,0x39,0x53,0x32,0x33,0xf6,0xea,0x84,0x14,0x0c,0xa1,0xd4,0x70,0xfb,0x10,
},
.inst.v = (unsigned char []){
0x36,0xf3,0xe7,0x49,0x11,0x77,0xc5,0xb9,0x1e,0xf4,0xbd,0x81,0x38,0x1b,0xec,0xc9,
0xfe,0x82,0x64,0x72,0x17,0x75,0x3b,0xf1,0xdf,0xc5,0x2d,0x3d,0x8c,0x56,0x38,0x30,
0xf3,0xa5,0x98,0xd7,0xe3,0x0d,0x6d,0xd8,0xf6,0x3e,0x37,0x07,0xb2,0x59,0xa9,0x8d,
0xb5,0x87,0x76,0x79,0x4d,0x32,0xaf,0x6a,0x55,0x5b,0x57,0x72,0x11,0x4d,0xa6,0x22,
0xf3,0xd2,0xa1,0x9d,0xf2,0x49,0xfc,0x88,0xbf,0x8f,0xaf,0x67,0x95,0xed,0x72,0xc5,
0x20,0x26,0xfe,0x22,0x84,0xd6,0x0d,0x91,0x69,0x43,0x12,0xd9,0x89,0xf3,0x16,0xd9,
0xc6,0xe8,0x1b,0xf1,0x91,0x6a,0x00,0x8d,0xcf,0x5f,0x6b,0xe4,0x9b,0x87,0xe3,
},
.inst.c = (unsigned char []){
0xb4,0x92,0x33,0x33,0x00,0x5a,0xb7,0x14,0x0e,0xc1,0x08,0xc1,0x20,0xa9,0x77,0x17,
0x47,0x23,0x57,0xc2,0x56,0x08,0x85,0x33,0xae,0xed,0x55,0x84,0x41,0x17,0xda,0x6c,
0x28,0xd1,0x11,0xd7,0x05,0x1c,0x5f,0x6b,0xda,0xee,0xe0,0x73,0x10,0x15,0x10,0xdb,
0x95,0x14,0xbb,0xb8,0xb0,0xb4,0x36,0x23,0xc2,0xb3,0x7f,0x2f,0x1d,0x3f,0x20,0xa9,
0xc9,0x2e,0xdb,0x2e,0x7b,0xc1,0x86,0x9e,0x43,0xc9,0x14,0xc1,0x0c,0x32,0xc1,0xbf,
0x81,0xbd,0xff,0x58,0x4c,0x9b,0x5c,0x47,0x80,0xfb,0xba,0xfa,0xa0,0xca,0x01,0x0a,
0xfb,0x7e,0x8b,0x05,0x2b,0xe8,0x38,0xf5,0x8c,0x04,0x3e,0x0c,0x2b,0x6a,0xd2,
},
.inst.reseed_ctr = 1,

.res.entropy = (unsigned char []){
0x4f,0x39,0xa2,0xb6,0x30,0x9a,0xf3,0x2a,0x1c,0x72,0x1b,0xd2,0xa3,0x1a,0x97,0x81,
0x48,0x07,0x64,0x43,0x03,0xb1,0xd1,0x32,0x23,0x93,0x64,0xa3,0x6d,0x26,0x6f,0x10,
},
.res.add = (unsigned char []){
0x94,0x7d,0x7b,0x37,0x2c,0x90,0x9e,0xae,0x57,0xba,0xe6,0x4d,0x7a,0x9b,0xbf,0x75,
0x23,0x4f,0x5c,0xb0,0x5c,0x09,0x39,0x27,0x03,0x7f,0x63,0x2d,0xd8,0x56,0x76,0x93,
},
.res.v = (unsigned char []){
0xe8,0x80,0xaa,0xd5,0x78,0x88,0xbc,0x81,0xbe,0xb6,0xe1,0x6b,0xdc,0xe7,0x5f,0x6d,
0xbd,0xd0,0xad,0xab,0x7e,0xb0,0x4a,0xeb,0x62,0xc4,0xd6,0xa4,0x27,0x4c,0x8e,0xaa,
0xcb,0x9c,0x3a,0x84,0x22,0xa4,0xdf,0x14,0xe6,0x57,0x0a,0x42,0x7b,0x01,0x35,0x61,
0xfd,0xd6,0x04,0x60,0x29,0x84,0xef,0xa4,0xe3,0xc0,0x17,0xb7,0xa5,0x6e,0x13,0x31,
0xe8,0xca,0xe2,0x54,0xe4,0x69,0xd2,0x54,0x38,0x70,0x99,0x94,0xbb,0x8b,0x37,0x07,
0xa5,0x2f,0xed,0xa8,0x6f,0x1b,0x0c,0xbf,0x3b,0x3c,0xa3,0x4a,0x97,0x35,0x7c,0x10,
0x98,0x6d,0x96,0x71,0x4d,0x2e,0xa2,0x66,0x0b,0xc6,0x71,0x30,0xeb,0xf8,0xe6,
},
.res.c = (unsigned char []){
0xec,0xed,0x12,0x31,0x3f,0x5f,0x03,0xdd,0xd2,0xdc,0x63,0xf0,0x99,0x8e,0xf7,0x23,
0x28,0x39,0xfe,0xd2,0xdd,0x04,0xd4,0x0f,0x43,0xf5,0x1c,0xb1,0x89,0xc5,0x4a,0xa4,
0xa0,0x15,0x24,0x62,0xbb,0x45,0x5e,0x89,0x7f,0xe8,0x15,0x0f,0x32,0x31,0x7d,0x6e,
0xba,0x74,0x84,0xd3,0xb2,0x07,0x89,0xf5,0xd7,0xa7,0x70,0x96,0x80,0xa6,0x90,0xa4,
0x31,0xdf,0x7d,0xe9,0x18,0xc0,0xa5,0x0e,0x8f,0x3f,0x3f,0x1a,0x4c,0xe0,0xdf,0xfa,
0x1e,0x74,0xba,0x70,0x85,0x69,0xde,0x35,0xa0,0xcd,0x52,0xed,0x31,0xd2,0x94,0x29,
0x87,0xd8,0xef,0x52,0xce,0x26,0xa8,0xb2,0x9c,0x0b,0x94,0xa6,0x5a,0xb0,0x65,
},
.res.reseed_ctr = 1,

.gen1.add = (unsigned char []){
0x65,0x1f,0x03,0xa8,0x02,0x34,0x34,0xb2,0xaf,0x28,0xa1,0xfa,0xd9,0x4b,0x31,0x91,
0x01,0x52,0xd4,0x9a,0x1b,0xbc,0xea,0xe3,0x2b,0xcc,0xd7,0x4d,0xde,0x60,0xd9,0x2f,
},
.gen1.v = (unsigned char []){
0xd5,0x6d,0xbd,0x06,0xb7,0xe7,0xc0,0x5f,0x91,0x93,0x45,0x5c,0x76,0x76,0x56,0x90,
0xe6,0x0a,0xac,0x7e,0x5b,0xb5,0x1e,0xfa,0xa6,0xb9,0xf3,0x55,0xb1,0x11,0xd9,0x4f,
0x6b,0xb1,0x5e,0xe6,0xdd,0xea,0x3d,0x9e,0x66,0x3f,0x1f,0x51,0xad,0x32,0xb3,0xb1,
0x2f,0x75,0xf5,0x94,0xe1,0x94,0xa0,0xeb,0xb8,0xc6,0xbd,0xb9,0x9a,0x42,0x88,0x31,
0x91,0x95,0xdb,0x4c,0xa3,0xe7,0x50,0x5f,0xa3,0x3a,0xb4,0x5f,0xa1,0x17,0x8f,0x51,
0xb9,0x66,0x84,0x0b,0x40,0x4e,0xfe,0x35,0xe0,0xbb,0xb3,0x87,0xe4,0x00,0xe8,0xd7,
0x5b,0x43,0x09,0xef,0x6c,0x3d,0xe2,0x8f,0x80,0x48,0x1e,0x10,0x1c,0xbf,0x3f,
},
.gen1.c = (unsigned char []){
0xec,0xed,0x12,0x31,0x3f,0x5f,0x03,0xdd,0xd2,0xdc,0x63,0xf0,0x99,0x8e,0xf7,0x23,
0x28,0x39,0xfe,0xd2,0xdd,0x04,0xd4,0x0f,0x43,0xf5,0x1c,0xb1,0x89,0xc5,0x4a,0xa4,
0xa0,0x15,0x24,0x62,0xbb,0x45,0x5e,0x89,0x7f,0xe8,0x15,0x0f,0x32,0x31,0x7d,0x6e,
0xba,0x74,0x84,0xd3,0xb2,0x07,0x89,0xf5,0xd7,0xa7,0x70,0x96,0x80,0xa6,0x90,0xa4,
0x31,0xdf,0x7d,0xe9,0x18,0xc0,0xa5,0x0e,0x8f,0x3f,0x3f,0x1a,0x4c,0xe0,0xdf,0xfa,
0x1e,0x74,0xba,0x70,0x85,0x69,0xde,0x35,0xa0,0xcd,0x52,0xed,0x31,0xd2,0x94,0x29,
0x87,0xd8,0xef,0x52,0xce,0x26,0xa8,0xb2,0x9c,0x0b,0x94,0xa6,0x5a,0xb0,0x65,
},
.gen1.reseed_ctr = 2,

.gen2.add = (unsigned char []){
0x93,0x4a,0x97,0xd6,0xee,0x46,0x85,0x25,0x0a,0xc5,0xe5,0x11,0x59,0xe5,0xbc,0xbc,
0x48,0xa6,0xf9,0x2c,0x7c,0xe7,0xbf,0xd1,0xc6,0x1d,0x0e,0x7b,0x0a,0x0e,0x7f,0x61,
},
.gen2.v = (unsigned char []){
0xc2,0x5a,0xcf,0x37,0xf7,0x46,0xc4,0x3d,0x64,0x6f,0xa9,0x4d,0x10,0x05,0x4d,0xb4,
0x0e,0x44,0xab,0x51,0x38,0xb9,0xf3,0x09,0xea,0xaf,0x10,0x07,0x3a,0xd7,0x23,0xf4,
0x0b,0xc6,0x83,0x49,0x99,0x2f,0x9c,0x27,0xe6,0x27,0x34,0x60,0xdf,0x64,0x31,0xa9,
0xff,0xd8,0x0c,0x15,0xed,0x67,0x40,0xd8,0x72,0xc4,0x64,0x68,0x24,0xee,0x9e,0x36,
0x2d,0x7a,0xce,0x7c,0xf6,0x5d,0x37,0x31,0x8c,0x2d,0x48,0x30,0x19,0x62,0x78,0x04,
0x4a,0xd9,0x8b,0x6e,0x0d,0x87,0x99,0xa0,0x55,0x37,0x4d,0x71,0x19,0x34,0xe5,0xbd,
0xa4,0x51,0xc0,0x82,0x4d,0x51,0xf3,0x83,0x8f,0xf1,0xf7,0x99,0x0e,0xef,0x1a,
},
.gen2.c = (unsigned char []){
0xec,0xed,0x12,0x31,0x3f,0x5f,0x03,0xdd,0xd2,0xdc,0x63,0xf0,0x99,0x8e,0xf7,0x23,
0x28,0x39,0xfe,0xd2,0xdd,0x04,0xd4,0x0f,0x43,0xf5,0x1c,0xb1,0x89,0xc5,0x4a,0xa4,
0xa0,0x15,0x24,0x62,0xbb,0x45,0x5e,0x89,0x7f,0xe8,0x15,0x0f,0x32,0x31,0x7d,0x6e,
0xba,0x74,0x84,0xd3,0xb2,0x07,0x89,0xf5,0xd7,0xa7,0x70,0x96,0x80,0xa6,0x90,0xa4,
0x31,0xdf,0x7d,0xe9,0x18,0xc0,0xa5,0x0e,0x8f,0x3f,0x3f,0x1a,0x4c,0xe0,0xdf,0xfa,
0x1e,0x74,0xba,0x70,0x85,0x69,0xde,0x35,0xa0,0xcd,0x52,0xed,0x31,0xd2,0x94,0x29,
0x87,0xd8,0xef,0x52,0xce,0x26,0xa8,0xb2,0x9c,0x0b,0x94,0xa6,0x5a,0xb0,0x65,
},
.gen2.reseed_ctr = 3,

.prnd = (unsigned char []){
0x77,0xce,0x58,0xaf,0x82,0x2a,0x11,0xf1,0xd6,0xdf,0xb2,0xbe,0x24,0x6a,0xa6,0x54,
0x9e,0x4c,0x08,0x9e,0x15,0x32,0xae,0xba,0x3e,0x6f,0x13,0xd1,0x96,0x35,0x93,0x0f,
0x70,0x24,0xf9,0x43,0x99,0xba,0x6c,0x84,0xdf,0xa3,0xcd,0x9c,0x78,0x06,0xd7,0xb7,
0x69,0xc0,0x96,0xd3,0x34,0x15,0xa5,0x6b,0xe1,0x09,0x66,0x18,0xd5,0xbd,0x80,0xd7,
0xd6,0x49,0x95,0x0e,0xbc,0xfe,0x74,0x18,0xe9,0xf2,0x33,0xac,0xdb,0xb7,0xc2,0xed,
0x36,0x81,0xf5,0xdc,0xd0,0x7e,0x74,0x1e,0xec,0xd2,0xa4,0x67,0x5a,0xc8,0xdb,0x7b,
0xec,0xd2,0xeb,0x77,0xf1,0x64,0xcf,0x5c,0x76,0x72,0xa6,0xe9,0xe6,0x19,0x25,0x4c,
0x5f,0x23,0x1c,0x85,0x34,0x4c,0x34,0x61,0xfa,0xdb,0x5f,0x19,0x17,0x06,0xda,0xe9,
0xfc,0x2c,0xb5,0xd6,0x1c,0x8b,0x36,0xb3,0xe1,0xaf,0x3e,0x82,0xce,0xf8,0xe2,0xa9,
0xca,0x76,0xfa,0xfb,0xaf,0x4d,0x4b,0xf9,0xe0,0xf9,0xb0,0xcd,0x59,0x07,0xd4,0x06,
0x9e,0x0e,0xf4,0xe1,0xf1,0x1a,0x25,0x99,0x63,0xc9,0xd8,0x28,0x2f,0x36,0xb2,0x7b,
0x75,0x68,0x39,0x58,0xa1,0x65,0x07,0x3e,0x69,0x2d,0xd9,0x82,0x08,0xd6,0xdc,0x17,
0xa7,0x91,0xfd,0x9c,0x3d,0xd5,0x45,0x54,0xca,0x6b,0x65,0x67,0x50,0xb7,0x7b,0x19,
0x61,0xc9,0x17,0xf4,0x86,0xba,0x3d,0xcf,0xfe,0x78,0xde,0x2b,0x34,0x6c,0xd3,0x8b,
0x4d,0xe2,0x34,0xaf,0x1a,0x35,0x66,0xcc,0xb8,0x6b,0x76,0x14,0xe9,0x9d,0x96,0x62,
0x4a,0xb3,0x91,0x95,0x23,0xdf,0xe9,0x38,0x1d,0xd5,0x7d,0x4d,0x00,0xf0,0x0b,0x6d,
},
},
};

#ifdef ICA_FIPS
const size_t AES_ECB_TV_LEN = sizeof(AES_ECB_TV) / sizeof(AES_ECB_TV[0]);
const size_t AES_CBC_TV_LEN = sizeof(AES_CBC_TV) / sizeof(AES_CBC_TV[0]);
const size_t AES_CBC_CS_TV_LEN = sizeof(AES_CBC_CS_TV)
    / sizeof(AES_CBC_CS_TV[0]);
const size_t AES_CFB_TV_LEN = sizeof(AES_CFB_TV) / sizeof(AES_CFB_TV[0]);
const size_t AES_OFB_TV_LEN = sizeof(AES_OFB_TV) / sizeof(AES_OFB_TV[0]);
const size_t AES_CTR_TV_LEN = sizeof(AES_CTR_TV) / sizeof(AES_CTR_TV[0]);
const size_t AES_CCM_TV_LEN = sizeof(AES_CCM_TV) / sizeof(AES_CCM_TV[0]);
const size_t AES_GCM_TV_LEN = sizeof(AES_GCM_TV) / sizeof(AES_GCM_TV[0]);
const size_t AES_XTS_TV_LEN = sizeof(AES_XTS_TV) / sizeof(AES_XTS_TV[0]);
const size_t AES_CMAC_TV_LEN = sizeof(AES_CMAC_TV) / sizeof(AES_CMAC_TV[0]);
const size_t DES3_ECB_TV_LEN = sizeof(DES3_ECB_TV) / sizeof(DES3_ECB_TV[0]);
const size_t DES3_CBC_TV_LEN = sizeof(DES3_CBC_TV) / sizeof(DES3_CBC_TV[0]);
const size_t DES3_CBC_CS_TV_LEN = sizeof(DES3_CBC_CS_TV)
    / sizeof(DES3_CBC_CS_TV[0]);
const size_t DES3_CFB_TV_LEN = sizeof(DES3_CFB_TV) / sizeof(DES3_CFB_TV[0]);
const size_t DES3_OFB_TV_LEN = sizeof(DES3_OFB_TV) / sizeof(DES3_OFB_TV[0]);
const size_t DES3_CTR_TV_LEN = sizeof(DES3_CTR_TV) / sizeof(DES3_CTR_TV[0]);
const size_t DES3_CMAC_TV_LEN = sizeof(DES3_CMAC_TV) / sizeof(DES3_CMAC_TV[0]);
const size_t RSA_TV_LEN = sizeof(RSA_TV) / sizeof(RSA_TV[0]);
const size_t SHA1_TV_LEN = sizeof(SHA1_TV) / sizeof(SHA1_TV[0]);
const size_t SHA224_TV_LEN = sizeof(SHA224_TV) / sizeof(SHA224_TV[0]);
const size_t SHA256_TV_LEN = sizeof(SHA256_TV) / sizeof(SHA256_TV[0]);
const size_t SHA384_TV_LEN = sizeof(SHA384_TV) / sizeof(SHA384_TV[0]);
const size_t SHA512_TV_LEN = sizeof(SHA512_TV) / sizeof(SHA512_TV[0]);
#endif /* ICA_FIPS */
const size_t DRBG_SHA512_TV_LEN = sizeof(DRBG_SHA512_TV)
    / sizeof(DRBG_SHA512_TV[0]);
